package ru.yandex.qe.dispenser.ws.bot

import io.swagger.annotations.Api
import io.swagger.annotations.ApiOperation
import io.swagger.annotations.ApiParam
import io.swagger.annotations.ApiResponse
import io.swagger.annotations.ApiResponses
import io.swagger.annotations.Authorization
import org.springframework.stereotype.Service
import ru.yandex.qe.dispenser.domain.hierarchy.Session
import ru.yandex.qe.dispenser.swagger.DispenserSecurityDefinition
import ru.yandex.qe.dispenser.swagger.SwaggerTags
import ru.yandex.qe.dispenser.ws.Access
import ru.yandex.qe.dispenser.ws.ServiceBase
import ru.yandex.qe.dispenser.ws.common.impl.ResultToResponse
import ru.yandex.qe.dispenser.ws.common.model.response.ErrorResponse
import javax.ws.rs.DELETE
import javax.ws.rs.GET
import javax.ws.rs.POST
import javax.ws.rs.PUT
import javax.ws.rs.Path
import javax.ws.rs.PathParam
import javax.ws.rs.Produces
import javax.ws.rs.QueryParam
import javax.ws.rs.core.Response

@Path("/v1/big-orders")
@Produces(ServiceBase.APPLICATION_JSON_UTF_8)
@Service("big-orders")
@Api(tags = [SwaggerTags.DISPENSER_API], authorizations = [Authorization(value = DispenserSecurityDefinition.AUTHORIZATION_SCHEME_NAME)])
open class BigOrderService(
    private val bigOrderManager: BigOrderManager
) {

    @GET
    @Access
    @Path("/{id}")
    @ApiOperation(value = "Get big order by id.")
    @ApiResponses(
        ApiResponse(code = 200, message = "Big order.", response = BigOrderDto::class),
        ApiResponse(code = 404, message = "'Not found' response.", response = ErrorResponse::class)
    )
    open fun getById(
        @ApiParam("Big order id.", required = true)
        @PathParam("id")
        id: Long
    ): Response {
        val result = bigOrderManager.getById(id, Session.USER_LOCALE.get())
        return ResultToResponse.toResponse(result)
    }

    @GET
    @Access
    @ApiOperation(value = "Get big orders page.")
    @ApiResponses(
        ApiResponse(code = 200, message = "Big orders page.", response = BigOrdersPageDto::class),
        ApiResponse(code = 400, message = "'Illegal argument' response.", response = ErrorResponse::class)
    )
    open fun getPage(
        @ApiParam("Continuation token.", required = false)
        @QueryParam("from")
        from: Long?,
        @ApiParam("Result limit.", required = false)
        @QueryParam("limit")
        limit: Long?
    ): Response {
        val result = bigOrderManager.getPage(from, limit, Session.USER_LOCALE.get())
        return ResultToResponse.toResponse(result)
    }

    @POST
    @Access(dispenserAdmin = true)
    @ApiOperation(value = "Create big order.")
    @ApiResponses(
        ApiResponse(code = 200, message = "Big order.", response = BigOrderDto::class),
        ApiResponse(code = 422, message = "'Illegal argument' response.", response = ErrorResponse::class)
    )
    open fun create(body: CreateBigOrderDto): Response {
        val result = bigOrderManager.create(body, Session.USER_LOCALE.get())
        return ResultToResponse.toResponse(result)
    }

    @PUT
    @Access(dispenserAdmin = true)
    @Path("/{id}")
    @ApiOperation(value = "Update big order.")
    @ApiResponses(
        ApiResponse(code = 200, message = "Big order.", response = BigOrderDto::class),
        ApiResponse(code = 404, message = "'Not found' response.", response = ErrorResponse::class),
        ApiResponse(code = 422, message = "'Illegal argument' response.", response = ErrorResponse::class)
    )
    open fun update(
        @ApiParam("Big order id.", required = true)
        @PathParam("id")
        id: Long,
        body: UpdateBigOrderDto
    ): Response {
        val result = bigOrderManager.update(id, body, Session.USER_LOCALE.get())
        return ResultToResponse.toResponse(result)
    }

    @DELETE
    @Access(dispenserAdmin = true)
    @Path("/{id}")
    @ApiOperation(value = "Delete big order.")
    @ApiResponses(
        ApiResponse(code = 204, message = "Successfully deleted.", response = BigOrderDto::class),
        ApiResponse(code = 404, message = "'Not found' response.", response = ErrorResponse::class),
        ApiResponse(code = 422, message = "'Illegal argument' response.", response = ErrorResponse::class)
    )
    open fun delete(
        @ApiParam("Big order id.", required = true)
        @PathParam("id")
        id: Long
    ): Response {
        val result = bigOrderManager.delete(id, Session.USER_LOCALE.get())
        return ResultToResponse.toResponse(result, { Response.noContent().build() }, true)
    }

}
