package ru.yandex.qe.dispenser.ws.bot;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.PATCH;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;

import com.google.common.collect.Sets;
import io.swagger.annotations.Api;
import io.swagger.annotations.Authorization;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.stereotype.Service;
import org.springframework.web.bind.annotation.RequestBody;
import ru.yandex.qe.dispenser.api.v1.DiBotCampaignGroup;
import ru.yandex.qe.dispenser.api.v1.response.DiListResponse;
import ru.yandex.qe.dispenser.domain.BotCampaignGroup;
import ru.yandex.qe.dispenser.domain.BotCampaignGroupUpdate;
import ru.yandex.qe.dispenser.domain.Campaign;
import ru.yandex.qe.dispenser.domain.CampaignForBot;
import ru.yandex.qe.dispenser.domain.bot.BigOrder;
import ru.yandex.qe.dispenser.domain.dao.bot.preorder.MappedPreOrderDao;
import ru.yandex.qe.dispenser.domain.dao.bot.settings.BotCampaignGroupDao;
import ru.yandex.qe.dispenser.domain.dao.campaign.CampaignDao;
import ru.yandex.qe.dispenser.domain.util.CollectionUtils;
import ru.yandex.qe.dispenser.swagger.DispenserSecurityDefinition;
import ru.yandex.qe.dispenser.swagger.SwaggerTags;
import ru.yandex.qe.dispenser.ws.Access;
import ru.yandex.qe.dispenser.ws.BigOrderUtils;
import ru.yandex.qe.dispenser.ws.ServiceBase;
import ru.yandex.qe.dispenser.ws.reqbody.BotCampaignGroupBodyCreate;
import ru.yandex.qe.dispenser.ws.reqbody.BotCampaignGroupBodyUpdate;

@ParametersAreNonnullByDefault
@Controller
@Path("/v1/bot/campaign-groups")
@Produces(ServiceBase.APPLICATION_JSON_UTF_8)
@Service("botCampaignGroupsService")
@Api(tags = {SwaggerTags.DISPENSER_API}, authorizations = {@Authorization(value = DispenserSecurityDefinition.AUTHORIZATION_SCHEME_NAME)})
public class BotCampaignGroupService {

    public static final String CAMPAIGN_GROUP_ID = "campaign_group_id";

    @Autowired
    private BotCampaignGroupDao botCampaignGroupDao;

    @Autowired
    private BigOrderManager bigOrderManager;

    @Autowired
    private CampaignDao campaignDao;

    @Autowired
    private MappedPreOrderDao mappedPreOrderDao;

    @NotNull
    @GET
    @Path("/{" + CAMPAIGN_GROUP_ID + "}")
    @Access(dispenserAdmin = true)
    public DiBotCampaignGroup getById(@NotNull @PathParam(CAMPAIGN_GROUP_ID) final Long id) {
        return botCampaignGroupDao.read(id).toView();
    }

    @NotNull
    @GET
    @Access(dispenserAdmin = true)
    public DiListResponse<DiBotCampaignGroup> getAll() {
        return new DiListResponse<>(botCampaignGroupDao.getAll().stream()
                .map(BotCampaignGroup::toView)
                .collect(Collectors.toList())
        );
    }

    @NotNull
    @PATCH
    @Access(dispenserAdmin =  true)
    @Path("/{" + CAMPAIGN_GROUP_ID + "}")
    public DiBotCampaignGroup update(@NotNull @PathParam(CAMPAIGN_GROUP_ID) final Long id,
                                     @NotNull @RequestBody final BotCampaignGroupBodyUpdate update) {
        final BotCampaignGroup botCampaignGroup = botCampaignGroupDao.readForUpdate(id);
        final Set<Long> currentCampaignIds = CollectionUtils.ids(botCampaignGroup.getCampaigns());
        final Set<Long> campaignIds = update.getCampaigns() != null ? update.getCampaigns() : currentCampaignIds;
        final Map<Long, CampaignForBot> campaignsById = campaignDao.readForBot(Sets.union(campaignIds, currentCampaignIds));
        checkCampaigns(campaignsById, campaignIds, currentCampaignIds);

        final Set<Long> orders = update.getBigOrders() != null ? update.getBigOrders() : CollectionUtils.ids(botCampaignGroup.getBigOrders());
        final Map<Long, BigOrder> orderById = BigOrderUtils.validateBigOrders(bigOrderManager, orders);

        final List<CampaignForBot> loadedNewCampaigns = campaignIds.stream()
                .map(campaignsById::get)
                .collect(Collectors.toList());
        checkCampaignsUnexpectedBigOrders(loadedNewCampaigns, orderById.keySet());

        final BotCampaignGroupUpdate groupUpdate = update.toEntity();

        checkAttachedPreOrders(id, groupUpdate);
        return botCampaignGroupDao.partialUpdate(botCampaignGroup, groupUpdate).toView();
    }

    @NotNull
    @POST
    @Access(dispenserAdmin = true)
    public DiBotCampaignGroup create(@NotNull @RequestBody final BotCampaignGroupBodyCreate body) {
        final Map<Long, BigOrder> validatedBigOrders = BigOrderUtils.validateBigOrders(bigOrderManager, body.getBigOrders());
        final Map<Long, CampaignForBot> campaignsById = campaignDao.readForBot(body.getCampaigns() != null ? body.getCampaigns() : Collections.emptySet());
        final BotCampaignGroup botCampaignGroup = body.toEntity(validatedBigOrders, campaignsById);
        return botCampaignGroupDao.create(botCampaignGroup).toView();
    }

    @NotNull
    @DELETE
    @Path("/{" + CAMPAIGN_GROUP_ID + "}")
    @Access(dispenserAdmin = true)
    public DiBotCampaignGroup delete(@NotNull @PathParam(CAMPAIGN_GROUP_ID) final Long id) {
        final BotCampaignGroup campaignGroup = botCampaignGroupDao.readForUpdate(id);
        if (!campaignGroup.getActiveCampaigns().isEmpty()) {
            throw new IllegalArgumentException("Campaign group cannot be removed with active campaign");
        }
        if (mappedPreOrderDao.hasPreOrdersInCampaignGroup(id)) {
            throw new IllegalArgumentException("Campaign group removal is forbidden when group already has pre-orders");
        }
        botCampaignGroupDao.delete(campaignGroup);
        return campaignGroup.toView();
    }

    public static void checkCampaignsUnexpectedBigOrders(final Collection<CampaignForBot> campaigns, final Set<Long> bigOrders) {
        final Set<Long> campaignBigOrders = campaigns.stream()
                .flatMap(c -> c.getBigOrders().stream())
                .map(BigOrder::getId)
                .collect(Collectors.toSet());
        final Set<Long> unexpected = Sets.difference(campaignBigOrders, bigOrders);

        if (!unexpected.isEmpty()) {
            throw new IllegalArgumentException(String.format("Orders %s not present in available orders %s", unexpected, bigOrders));
        }
    }

    public static void checkCampaigns(final Map<Long, CampaignForBot> campaignsById, final Set<Long> newCampaignIds, final Set<Long> oldCampaigns) {
        final Set<Long> missingCampaignIds = Sets.difference(newCampaignIds, campaignsById.keySet());
        if (!missingCampaignIds.isEmpty()) {
            throw new IllegalArgumentException(String.format("Campaigns with ids %s not found", missingCampaignIds));
        }
        final Sets.SetView<Long> detachedCampaignIds = Sets.difference(oldCampaigns, newCampaignIds);
        for (Long detachedCampaignId : detachedCampaignIds) {
            final CampaignForBot campaign = campaignsById.get(detachedCampaignId);
            if (campaign.getStatus() == Campaign.Status.ACTIVE) {
                throw new IllegalArgumentException(String.format("Cannot detach active campaign %s from group", detachedCampaignId));
            }
        }

        final Sets.SetView<Long> attachedCampaignIds = Sets.difference(newCampaignIds, oldCampaigns);
        for (Long attachedCampaignId : attachedCampaignIds) {
            final CampaignForBot campaign = campaignsById.get(attachedCampaignId);
            if (campaign.getStatus() == Campaign.Status.ACTIVE) {
                throw new IllegalArgumentException(String.format("Cannot attach active campaign %s to group", attachedCampaignId));
            }
        }
    }

    private void checkAttachedPreOrders(final long campaignGroupId, final BotCampaignGroupUpdate update) {
        if (update.getBigOrders() != null) {
            if (mappedPreOrderDao.hasPreOrdersInCampaignGroupForOrdersOtherThan(campaignGroupId, update.getBigOrders())) {
                throw new IllegalArgumentException("Big order unbinding is forbidden when campaign group already has pre-orders for them");
            }
        }
    }

}
