package ru.yandex.qe.dispenser.ws.bot;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.inject.Inject;
import javax.ws.rs.BeanParam;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.UriInfo;

import io.swagger.annotations.Api;
import io.swagger.annotations.Authorization;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.dao.DuplicateKeyException;
import org.springframework.dao.IncorrectResultSizeDataAccessException;
import ru.yandex.qe.dispenser.api.v1.DiBotConfigurationReserve;
import ru.yandex.qe.dispenser.api.v1.response.DiListPageResponse;
import ru.yandex.qe.dispenser.domain.Campaign;
import ru.yandex.qe.dispenser.domain.Person;
import ru.yandex.qe.dispenser.domain.Segment;
import ru.yandex.qe.dispenser.domain.Segmentation;
import ru.yandex.qe.dispenser.domain.dao.bot.service.reserve.configurations.BotConfigurationReserve;
import ru.yandex.qe.dispenser.domain.dao.bot.service.reserve.configurations.BotConfigurationReserveDao;
import ru.yandex.qe.dispenser.domain.dao.campaign.CampaignDao;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;
import ru.yandex.qe.dispenser.domain.hierarchy.HierarchySupplier;
import ru.yandex.qe.dispenser.domain.hierarchy.Session;
import ru.yandex.qe.dispenser.swagger.DispenserSecurityDefinition;
import ru.yandex.qe.dispenser.swagger.SwaggerTags;
import ru.yandex.qe.dispenser.ws.Access;
import ru.yandex.qe.dispenser.ws.ServiceBase;
import ru.yandex.qe.dispenser.ws.param.BotServiceReserveFilterParam;
import ru.yandex.qe.dispenser.ws.param.PaginationParam;
import ru.yandex.qe.dispenser.ws.reqbody.BotConfigurationReserveBody;
import ru.yandex.qe.dispenser.ws.reqbody.BotConfigurationReserveCreate;

import static ru.yandex.qe.dispenser.ws.bot.BotServiceReserveService.RESERVE_ID;
import static ru.yandex.qe.dispenser.ws.bot.BotServiceReserveService.checkThatUserCanEditReserves;
import static ru.yandex.qe.dispenser.ws.param.PageParam.LIMIT_PARAM;
import static ru.yandex.qe.dispenser.ws.param.PageParam.PAGE_PARAM;
import static ru.yandex.qe.dispenser.ws.param.PageParam.PAGE_SIZE_PARAM;

@ParametersAreNonnullByDefault
@org.springframework.stereotype.Service("botConfigurationReserveService")
@Path("/v1/bot/reserves/configurations")
@Produces(ServiceBase.APPLICATION_JSON_UTF_8)
@Api(tags = {SwaggerTags.DISPENSER_API}, authorizations = {@Authorization(value = DispenserSecurityDefinition.AUTHORIZATION_SCHEME_NAME)})
@Deprecated
public class BotConfigurationReserveService extends ServiceBase {
    private static final Logger LOG = LoggerFactory.getLogger(BotConfigurationReserveService.class);

    private final BotConfigurationReserveDao botConfigurationReserveDao;
    private final CampaignDao campaignDao;
    private final HierarchySupplier hierarchySupplier;
    private final String locationSegmentationKey;

    @Inject
    public BotConfigurationReserveService(final BotConfigurationReserveDao botConfigurationReserveDao,
                                          final CampaignDao campaignDao,
                                          final HierarchySupplier hierarchySupplier,
                                          @Value("${dispenser.location.segmentation.key}") final String locationSegmentationKey) {
        this.botConfigurationReserveDao = botConfigurationReserveDao;
        this.campaignDao = campaignDao;
        this.hierarchySupplier = hierarchySupplier;
        this.locationSegmentationKey = locationSegmentationKey;
    }

    @GET
    @NotNull
    @Access
    @Deprecated
    public DiListPageResponse<DiBotConfigurationReserve> getReserves(@BeanParam final BotServiceReserveFilterParam filterParam,
                                                                     @QueryParam(PAGE_PARAM) final Long pageNumberParam,
                                                                     @QueryParam(PAGE_SIZE_PARAM) final Long pageSizeParam,
                                                                     @QueryParam(LIMIT_PARAM) final Long limitParam,
                                                                     @Context final UriInfo uriInfo) {
        final PaginationParam paginationParam = PaginationParam.from(pageNumberParam, pageSizeParam, limitParam, uriInfo);
        final Hierarchy hierarchy = hierarchySupplier.get();
        return collectToPage(paginationParam,
                filterParam.getFilteredReserveStream(botConfigurationReserveDao, campaignDao),
                reserve -> reserve.toView(hierarchy));
    }

    @GET
    @NotNull
    @Path("/{" + RESERVE_ID + "}")
    @Access
    @Deprecated
    public DiBotConfigurationReserve getReserve(@NotNull @PathParam(RESERVE_ID) final Long id) {
        return getBotConfigurationReserve(id).toView(hierarchySupplier.get());
    }

    @POST
    @NotNull
    @Access
    @Deprecated
    public DiBotConfigurationReserve createReserve(@NotNull final BotConfigurationReserveCreate body) {
        final Campaign campaign = campaignDao.readOptional(body.getCampaignId())
                .orElseThrow(() -> new IllegalArgumentException("Reserve can not be created without campaign"));

        final boolean noBigOrders = campaign.getBigOrders().stream().noneMatch(bo -> bo.getBigOrderId().equals(body.getBigOrderId()));
        if (noBigOrders) {
            throw new IllegalArgumentException("Big order '" + body.getBigOrderId() + "' is not in the active campaign");
        }

        if (body.getStorageId() != null && (body.getStorageQuantity() == null || body.getStorageQuantity() < 0)) {
            throw new IllegalArgumentException("Reserve can not be created with 'storageId' and empty or negative 'storageQuantity'");
        }

        final Hierarchy hierarchy = hierarchySupplier.get();

        final BotConfigurationReserve reserve = body.toBuilder(hierarchy)
                .campaignId(campaign.getId())
                .locationSegmentId(getLocationSegmentId(body.getLocationSegmentKey()))
                .build();

        final Person performer = Session.WHOAMI.get();
        checkThatUserCanEditReserves(performer, reserve);
        LOG.debug("User {} is trying to create reserve {}", performer, reserve);

        final BotConfigurationReserve createdReserve;
        try {
            createdReserve = botConfigurationReserveDao.create(reserve);
        } catch (DuplicateKeyException e) {
            throw new IllegalArgumentException("Reserve with same key = " + reserve.getKey() + " already exists", e);
        }
        return createdReserve.toView(hierarchy);
    }

    @PUT
    @NotNull
    @Path("/{" + RESERVE_ID + "}")
    @Access
    @Deprecated
    public DiBotConfigurationReserve updateReserve(@PathParam(RESERVE_ID) @NotNull final Long id,
                                                   @NotNull final BotConfigurationReserveBody body) {
        final BotConfigurationReserve old = getBotConfigurationReserve(id);
        final Person performer = Session.WHOAMI.get();
        checkThatUserCanEditReserves(performer, old);
        if (body.isEmpty()) {
            throw new IllegalArgumentException("Empty update!");
        }
        if (body.getBigOrderId() != null) {
            final Campaign campaign = campaignDao.readWithDeleted(old.getKey().getCampaignId());
            if (campaign.getBigOrders().stream().noneMatch(o -> o.getBigOrderId().equals(body.getBigOrderId()))) {
                throw new IllegalArgumentException("Big order '" + body.getBigOrderId() + "' is not in the campaign");
            }
        }
        final Hierarchy hierarchy = hierarchySupplier.get();

        final BotConfigurationReserve.Builder bodyBuilder = body.applyToBuilder(old.copyBuilder(), hierarchy);
        if (body.getLocationSegmentKey() != null) {
            bodyBuilder.locationSegmentId(getLocationSegmentId(body.getLocationSegmentKey()));
        }
        final BotConfigurationReserve update = bodyBuilder.build();

        try {
            botConfigurationReserveDao.update(update);
        } catch (DuplicateKeyException e) {
            throw new IllegalArgumentException("Reserve with same key = " + update.getKey() + " already exists", e);
        }

        return update.toView(hierarchy);
    }

    @DELETE
    @NotNull
    @Path("/{" + RESERVE_ID + "}")
    @Access
    @Deprecated
    public DiBotConfigurationReserve deleteReserve(@NotNull @PathParam(RESERVE_ID) final Long id) {
        final BotConfigurationReserve reserveToDelete = getBotConfigurationReserve(id);
        final Person performer = Session.WHOAMI.get();
        checkThatUserCanEditReserves(performer, reserveToDelete);
        botConfigurationReserveDao.delete(reserveToDelete);
        return reserveToDelete.toView(hierarchySupplier.get());
    }

    private BotConfigurationReserve getBotConfigurationReserve(final @NotNull Long id) {
        try {
            return botConfigurationReserveDao.read(id);
        } catch (IncorrectResultSizeDataAccessException e) {
            throw new IllegalArgumentException("No reserve with id = " + id);
        }
    }

    private long getLocationSegmentId(final String locationSegmentKey) {
        final Segment segment = hierarchySupplier.get().getSegmentReader().read(locationSegmentKey);
        final Segmentation segmentation = segment.getSegmentation();
        final String segmentKey = segmentation.getKey().getPublicKey();
        if (!segmentKey.equals(locationSegmentationKey)) {
            throw new IllegalArgumentException("Segment from invalid segmentation '" + segmentKey + "', must be location");
        }
        return segment.getId();
    }
}
