package ru.yandex.qe.dispenser.ws.bot;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonProperty;

public class BotPreOrderErrorsCollection {

    private final Set<String> createErrors;
    private final Map<String, Set<Long>> updateErrors;
    private final Map<String, Set<Long>> deleteErrors;
    private final Set<String> errors;
    private long errorsCount;

    @JsonCreator
    public BotPreOrderErrorsCollection(
            @JsonProperty("createErrors") final Set<String> createErrors,
            @JsonProperty("updateErrors") final Map<String, Set<Long>> updateErrors,
            @JsonProperty("deleteErrors") final Map<String, Set<Long>> deleteErrors,
            @JsonProperty("errors") final Set<String> errors,
            @JsonProperty("errorsCount") final long errorsCount) {
        this.createErrors = createErrors;
        this.updateErrors = updateErrors;
        this.deleteErrors = deleteErrors;
        this.errors = errors;
    }

    public BotPreOrderErrorsCollection() {
        this.createErrors = new HashSet<>();
        this.updateErrors = new HashMap<>();
        this.deleteErrors = new HashMap<>();
        this.errors = new HashSet<>();
        this.errorsCount = 0L;
    }

    public void addCreationError(final Exception error) {
        addCreationError(error.toString());
    }

    public void addCreationError(final String error) {
        createErrors.add(error);
        errorsCount++;
    }

    public void addUpdateError(final long preOrderId, final Exception error) {
        addUpdateError(preOrderId, error.toString());
    }

    public void addUpdateError(final long preOrderId, final String error) {
        updateErrors.computeIfAbsent(error, k -> new HashSet<>()).add(preOrderId);
        errorsCount++;
    }

    public void addDeleteError(final long preOrderId, final Exception error) {
        addDeleteError(preOrderId, error.toString());
    }

    public void addDeleteError(final long preOrderId, final String error) {
        deleteErrors.computeIfAbsent(error, k -> new HashSet<>()).add(preOrderId);
        errorsCount++;
    }

    public void addError(final String error) {
        errors.add(error);
        errorsCount++;
    }

    public void addError(final Throwable error) {
        addError(error.toString());
    }

    public Set<String> getCreateErrors() {
        return createErrors;
    }

    public Map<String, Set<Long>> getUpdateErrors() {
        return updateErrors;
    }

    public Map<String, Set<Long>> getDeleteErrors() {
        return deleteErrors;
    }

    public Set<String> getErrors() {
        return errors;
    }

    public long getErrorsCount() {
        return errorsCount;
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        final BotPreOrderErrorsCollection that = (BotPreOrderErrorsCollection) o;
        return errorsCount == that.errorsCount &&
                Objects.equals(createErrors, that.createErrors) &&
                Objects.equals(updateErrors, that.updateErrors) &&
                Objects.equals(deleteErrors, that.deleteErrors) &&
                Objects.equals(errors, that.errors);
    }

    @Override
    public int hashCode() {
        return Objects.hash(createErrors, updateErrors, deleteErrors, errors, errorsCount);
    }

    @Override
    public String toString() {
        return "BotPreOrderErrorsCollection{" +
                "createErrors=" + createErrors +
                ", updateErrors=" + updateErrors +
                ", deleteErrors=" + deleteErrors +
                ", errors=" + errors +
                ", errorsCount=" + errorsCount +
                '}';
    }

    @JsonIgnore
    public boolean isEmpty() {
        return errorsCount == 0;
    }

}
