package ru.yandex.qe.dispenser.ws.bot;

import java.util.Optional;

import org.jetbrains.annotations.Nullable;

import ru.yandex.qe.dispenser.domain.Resource;
import ru.yandex.qe.dispenser.domain.bot.ConfigurationComponent;
import ru.yandex.qe.dispenser.ws.quota.request.workflow.ResourceWorkflow;
import ru.yandex.qe.dispenser.domain.util.PropertyUtils;

public enum BotResourceType {
    CPU {
        @Override
        public boolean is(final ConfigurationComponent component) {
            return hasTypeCode(component, "cpu");
        }

        @Override
        public long getAmountPerComponent(final ConfigurationComponent component) {
            return getAttributeWithName(component, "threads quantity (per 1cpu)")
                    .map(ConfigurationComponent.Value::getIntegerValue)
                    .map(value -> value * 1_000L)
                    .orElse(0L);
        }
    },
    GPU {
        @Override
        public boolean is(final ConfigurationComponent component) {
            return hasTypeCode(component, "gpu");
        }

        @Override
        public long getAmountPerComponent(final ConfigurationComponent component) {
            return 1_000;
        }
    },
    RAM {
        @Override
        public boolean is(final ConfigurationComponent component) {
            return hasTypeCode(component, "ram_bulk");
        }

        @Override
        public long getAmountPerComponent(final ConfigurationComponent component) {
            return getSizeInBytes(component, "Total size (GB)", RAM_BYTES_IN_GB);
        }
    },
    SSD {
        @Override
        public boolean is(final ConfigurationComponent component) {
            return hasTypeCode(component, "drive") &&
                    getAttributeWithName(component, "Type")
                            .map(value -> "SSD".equals(value.getStringValue()))
                            .orElse(false);
        }

        @Override
        public long getAmountPerComponent(final ConfigurationComponent component) {
            return getSizeInBytes(component);
        }
    },
    HDD {
        @Override
        public boolean is(final ConfigurationComponent component) {
            return hasTypeCode(component, "drive") &&
                    getAttributeWithName(component, "Type")
                            .map(value -> "HDD".equals(value.getStringValue()))
                            .orElse(false);
        }

        @Override
        public long getAmountPerComponent(final ConfigurationComponent component) {
            return getSizeInBytes(component);
        }
    },
    NVME {
        @Override
        public boolean is(final ConfigurationComponent component) {
            return hasTypeCode(component, "drive") &&
                    getAttributeWithName(component, "Type")
                            .map(value -> "NVMe".equals(value.getStringValue()))
                            .orElse(false);
        }

        @Override
        public long getAmountPerComponent(final ConfigurationComponent component) {
            return getSizeInBytes(component);
        }
    };

    public static final long BYTES_IN_GIGABYTE = 1_000_000_000L;
    public static final long BYTES_IN_TERABYTE = 1_000_000_000_000L;
    public static final long RAM_BYTES_IN_GB = 1024L * 1024 * 1024;

    protected static boolean hasTypeCode(final ConfigurationComponent component, final String typeCode) {
        return component.getComponent().getTypeCode().equals(typeCode);
    }

    protected static Optional<ConfigurationComponent.Value> getAttributeWithName(final ConfigurationComponent component,
                                                                                 final String name) {
        return component.getComponent().getValues().stream()
                .filter(value -> value.getAttribute().getName().equals(name))
                .findFirst();
    }

    protected static Long getTotalSizeInBytes(final ConfigurationComponent component) {
        return getSizeInBytes(component, "Total size", BYTES_IN_GIGABYTE);
    }

    protected static Long getSizeInBytes(final ConfigurationComponent component) {
        return getSizeInBytes(component, "Size (GB)", BYTES_IN_GIGABYTE);
    }

    protected static Long getSizeInBytes(final ConfigurationComponent component, final String attributeName, final Long bytesCount) {
        return getAttributeWithName(component, attributeName)
                .map(ConfigurationComponent.Value::getIntegerValue)
                .map(value -> value * bytesCount)
                .orElse(0L);
    }

    @Nullable
    public static BotResourceType valueOf(final Resource resource) {
        final String publicKey = resource.getPublicKey();

        final String mappedType = PropertyUtils.readString(ResourceWorkflow.PROPERTY_RESOURCE_PREORDER_ENTITY_KEY,
                "mapping_" + resource.getService().getKey() + "_" + resource.getPublicKey());

        if (mappedType != null) {
            return fromStringType(mappedType);
        }

        return fromStringType(publicKey);
    }

    @Nullable
    private static BotResourceType fromStringType(final String type) {
        switch (type) {
            case "cpu":
                return CPU;
            case "gpu":
                return GPU;
            case "hdd":
                return HDD;
            case "ssd":
                return SSD;
            case "ram":
                return RAM;
            case "nvme":
                return NVME;
            default:
                return null;
        }
    }

    public abstract boolean is(final ConfigurationComponent component);

    public abstract long getAmountPerComponent(final ConfigurationComponent component);

    public final long getAmount(final ConfigurationComponent component) {
        return getAmountPerComponent(component) * component.getQuantity();
    }
}
