package ru.yandex.qe.dispenser.ws.bot;

import java.util.Arrays;
import java.util.Collections;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collector;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.SetMultimap;
import org.jetbrains.annotations.Nullable;

import ru.yandex.qe.dispenser.domain.BotCampaignGroup;
import ru.yandex.qe.dispenser.domain.Service;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;
import ru.yandex.qe.dispenser.domain.util.MoreFunctions;


@ParametersAreNonnullByDefault
public enum Provider {
    YT("yt"),
    MDS("mds"),
    MDS_NEW("mds_new"),
    S3("s3"),
    AVATARS("avatars"),
    YP("yp", YT),
    GEN_CFG("gencfg", YT),
    SAAS("saas", YT),
    NIRVANA("nirvana", YT, MDS),
    QLOUD("qloud", YT),
    LOGFELLER("logfeller", YT),
    LOGBROKER_YT_PROXY("logbroker_yt_proxy", YT),

    SOLOMON("solomon"),
    MDB("dbaas", MDS),

    RTMR("rtmr"),
    RTMR_PROCESSING("rtmr_processing", RTMR),
    RTMR_MIRROR("rtmr_mirror", RTMR),

    YDB("ydb"),
    LOGBROKER("logbroker"),
    SANDBOX("sandbox", MDS),
    SQS("sqs"),
    DISTBUILD("distbuild"),

    STRM("strm"),
    ;

    private static final Map<String, Provider> PROVIDERS_BY_SERVICE_KEY = Arrays.stream(values())
            .collect(Collectors.toMap(Provider::getServiceKey, Function.identity()));

    @Nonnull
    public static Optional<Provider> fromService(final Service service) {
        return Optional.ofNullable(PROVIDERS_BY_SERVICE_KEY.get(service.getKey()));
    }

    @Nonnull
    public static Optional<Provider> fromKey(final String serviceKey) {
        return Optional.ofNullable(PROVIDERS_BY_SERVICE_KEY.get(serviceKey));
    }

    private final String serviceKey;
    private final Set<Provider> baseProviders;

    Provider(final String serviceKey, final Provider... baseProvider) {
        this.serviceKey = serviceKey;
        this.baseProviders = Set.of(baseProvider);
    }

    @Nonnull
    public String getServiceKey() {
        return serviceKey;
    }

    public Set<Provider> getBaseProviders(BotCampaignGroup campaignGroup) {
        // TODO Awful shortcut
        Set<Provider> adjustedBaseProviders = adjustedBaseProviders(this, campaignGroup);
        return adjustedBaseProviders.isEmpty() ? Collections.singleton(this) : adjustedBaseProviders;
    }

    public boolean is(final Service service) {
        return serviceKey.equals(service.getKey());
    }

    @Nullable
    public Service getService() {
        return Hierarchy.get().getServiceReader().readOrNull(serviceKey);
    }

    @Nonnull
    public Set<Provider> getMetaProviders(BotCampaignGroup campaignGroup) {
        // TODO Awful shortcut
        SetMultimap<Provider, Provider> metaProviderByBaseProvider = Arrays.stream(values())
                .collect(Collector.of(HashMultimap::create, (m, p) -> {
                    if (p.isMetaProvider(campaignGroup)) {
                        for (Provider baseProvider : adjustedBaseProviders(p, campaignGroup)) {
                            m.put(baseProvider, p);
                        }
                    }
                }, MoreFunctions.leftProjection()));
        return metaProviderByBaseProvider.get(this);
    }

    public boolean isMetaProvider(BotCampaignGroup campaignGroup) {
        // TODO Awful shortcut
        Set<Provider> adjustedBaseProviders = adjustedBaseProviders(this, campaignGroup);
        return !adjustedBaseProviders.isEmpty();
    }

    private Set<Provider> adjustedBaseProviders(Provider provider, BotCampaignGroup campaignGroup) {
        // TODO Awful shortcut
        if (campaignGroup.ifYpInYt()) {
            return provider.baseProviders;
        } else if (provider.equals(YP)) {
            return Collections.emptySet();
        } else if (provider.equals(QLOUD)) {
            return Collections.singleton(YP);
        } else if (provider.equals(GEN_CFG)) {
            return Collections.singleton(YP);
        } else if (provider.equals(SAAS)) {
            return Collections.singleton(YP);
        } else {
            return provider.baseProviders;
        }
    }

}
