package ru.yandex.qe.dispenser.ws.common.domain.result;

import java.util.Objects;
import java.util.function.Consumer;
import java.util.function.Function;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public interface Result<T, E> {

    @NotNull
    static <I, F> Result<I, F> success(final I result) {
        return new Result<I, F>() {
            @Override
            public <R> R match(@NotNull final Cases<I, F, R> cases) {
                return cases.success(result);
            }

            @Override
            public <R> R match(@NotNull final Function<I, R> onSuccess, @NotNull final Function<F, R> onFailure) {
                return onSuccess.apply(result);
            }

            @NotNull
            @Override
            public <O> Result<O, F> apply(@NotNull final Function<I, O> map) {
                return Result.success(map.apply(result));
            }

            @NotNull
            @Override
            public <O> Result<O, F> andThen(@NotNull final Function<I, Result<O, F>> map) {
                return map.apply(result);
            }

            @NotNull
            @SuppressWarnings("unchecked")
            @Override
            public <U> Result<I, U> mapError(@NotNull final Function<F, U> map) {
                return (Result<I, U>) this;
            }

            @Override
            public void doOnSuccess(@NotNull final Consumer<I> consumer) {
                consumer.accept(result);
            }

            @Override
            public void doOnFailure(@NotNull final Consumer<F> consumer) {
            }

            @Override
            public boolean isSuccess() {
                return true;
            }

            @Override
            public boolean isError() {
                return false;
            }

            @Override
            public int hashCode() {
                return Objects.hash(result);
            }

            @SuppressWarnings("unchecked")
            @Override
            public boolean equals(final Object obj) {
                if (this == obj) {
                    return true;
                }
                if (!(obj instanceof Result)) {
                    return false;
                }
                final Result<Object, Object> result = (Result<Object, Object>) obj;
                return result.match(new Cases<Object, Object, Boolean>() {
                    @Override
                    public Boolean success(@NotNull final Object otherResult) {
                        return Objects.equals(result, otherResult);
                    }

                    @Override
                    public Boolean failure(@NotNull final Object error) {
                        return false;
                    }
                });
            }

            @Override
            public String toString() {
                return "Result::success{"
                        + "result=" + result
                        + '}';
            }
        };
    }

    @NotNull
    static <I, F> Result<I, F> failure(final F error) {
        Objects.requireNonNull(error, "Error is required.");
        return new Result<I, F>() {
            @Override
            public <R> R match(@NotNull final Cases<I, F, R> cases) {
                return cases.failure(error);
            }

            @Override
            public <R> R match(@NotNull Function<I, R> onSuccess, @NotNull Function<F, R> onFailure) {
                return onFailure.apply(error);
            }

            @NotNull
            @SuppressWarnings("unchecked")
            @Override
            public <O> Result<O, F> apply(@NotNull final Function<I, O> map) {
                return (Result<O, F>) this;
            }

            @NotNull
            @SuppressWarnings("unchecked")
            @Override
            public <O> Result<O, F> andThen(@NotNull final Function<I, Result<O, F>> map) {
                return (Result<O, F>) this;
            }

            @NotNull
            @Override
            public <U> Result<I, U> mapError(@NotNull final Function<F, U> map) {
                return Result.failure(map.apply(error));
            }

            @Override
            public void doOnSuccess(@NotNull final Consumer<I> consumer) {
            }

            @Override
            public void doOnFailure(@NotNull final Consumer<F> consumer) {
                consumer.accept(error);
            }

            @Override
            public boolean isSuccess() {
                return false;
            }

            @Override
            public boolean isError() {
                return true;
            }

            @Override
            public int hashCode() {
                return Objects.hash(error);
            }

            @SuppressWarnings("unchecked")
            @Override
            public boolean equals(final Object obj) {
                if (this == obj) {
                    return true;
                }
                if (!(obj instanceof Result)) {
                    return false;
                }
                final Result<Object, Object> result = (Result<Object, Object>) obj;
                return result.match(new Cases<Object, Object, Boolean>() {
                    @Override
                    public Boolean success(@NotNull final Object result) {
                        return false;
                    }

                    @Override
                    public Boolean failure(@NotNull final Object otherError)
                    {
                        return Objects.equals(error, otherError);
                    }
                });
            }

            @Override
            public String toString() {
                return "Result::failure{"
                        + "error=" + error
                        + '}';
            }
        };
    }

    <R> R match(@NotNull final Cases<T, E, R> cases);

    <R> R match(@NotNull final Function<T, R> onSuccess, @NotNull final Function<E, R> onFailure);

    @NotNull
    <R> Result<R, E> apply(@NotNull final Function<T, R> map);

    @NotNull
    <R> Result<R, E> andThen(@NotNull final Function<T, Result<R, E>> map);

    @NotNull
    <U> Result<T, U> mapError(@NotNull final Function<E, U> map);

    void doOnSuccess(@NotNull final Consumer<T> consumer);

    void doOnFailure(@NotNull final Consumer<E> consumer);

    boolean isSuccess();

    boolean isError();

    interface Cases<I, E, O> {

        @Nullable
        O success(final I result);

        @Nullable
        O failure(@NotNull final E error);

    }
}
