package ru.yandex.qe.dispenser.ws.common.impl;

import java.util.Collections;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.ws.rs.core.Response;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import ru.yandex.qe.dispenser.ws.common.domain.errors.ErrorCollection;
import ru.yandex.qe.dispenser.ws.common.domain.errors.ErrorType;
import ru.yandex.qe.dispenser.ws.common.domain.errors.TypedError;
import ru.yandex.qe.dispenser.ws.common.domain.result.Result;
import ru.yandex.qe.dispenser.ws.common.model.response.ErrorResponse;
import ru.yandex.qe.dispenser.ws.param.DiExceptionMapper;

public final class ResultToResponse {

    private ResultToResponse() {
    }

    @NotNull
    public static <T> Response toResponse(@NotNull final Result<T, ErrorCollection<String, TypedError<String>>> result) {
        return toResponse(result, true);
    }

    @NotNull
    public static <T> Response toResponse(@NotNull final Result<T, ErrorCollection<String, TypedError<String>>> result,
                                          final boolean withErrorDetails) {
        return toResponse(result, r -> Response.ok(r).build(), e -> mapError(e, withErrorDetails));
    }

    @NotNull
    public static <T> Response toResponse(@NotNull final Result<T, ErrorCollection<String, TypedError<String>>> result,
                                          @NotNull final Function<T, Response> mapSuccess) {
        return toResponse(result, mapSuccess, true);
    }

    @NotNull
    public static <T> Response toResponse(@NotNull final Result<T, ErrorCollection<String, TypedError<String>>> result,
                                          @NotNull final Function<T, Response> mapSuccess, final boolean withErrorDetails) {
        return toResponse(result, mapSuccess, e -> mapError(e, withErrorDetails));
    }

    @NotNull
    private static Response mapError(@NotNull final ErrorCollection<String, TypedError<String>> error, final boolean withDetails) {
        final boolean hasForbidden = error.getErrors().stream().anyMatch(e -> e.getType() == ErrorType.FORBIDDEN)
                || error.getFieldErrors().values().stream().anyMatch(c -> c.stream()
                .anyMatch(e -> e.getType() == ErrorType.FORBIDDEN));
        if (hasForbidden) {
            final Set<String> errors = error.getErrors().stream().filter(e -> e.getType() == ErrorType.FORBIDDEN)
                    .map(TypedError::getError).collect(Collectors.toSet());
            final Map<String, Set<String>> fieldErrors = error.getFieldErrors().entrySet().stream()
                    .filter(e -> e.getValue().stream().anyMatch(v -> v.getType() == ErrorType.FORBIDDEN))
                    .collect(Collectors.toMap(Map.Entry::getKey, e -> e.getValue().stream()
                            .filter(v -> v.getType() == ErrorType.FORBIDDEN).map(TypedError::getError)
                            .collect(Collectors.toSet())));
            final ErrorResponse entity = new ErrorResponse(errors, fieldErrors, Collections.emptyMap());
            return Response.status(Response.Status.FORBIDDEN).entity(entity).build();
        }
        final Set<String> errors = error.getErrors().stream().map(TypedError::getError).collect(Collectors.toSet());
        final Map<String, Set<String>> fieldErrors = error.getFieldErrors().entrySet().stream()
                .collect(Collectors.toMap(Map.Entry::getKey, e -> e.getValue().stream().map(TypedError::getError)
                        .collect(Collectors.toSet())));
        final ErrorResponse entity = new ErrorResponse(errors, fieldErrors,
                withDetails ? error.getDetails() : Collections.emptyMap());
        final Response.StatusType statusType = getStatusType(error);
        return Response.status(statusType).entity(entity).build();
    }

    @NotNull
    private static Response.StatusType getStatusType(@NotNull final ErrorCollection<String, TypedError<String>> error) {
        final boolean hasNotFound = error.getErrors().stream().anyMatch(e -> e.getType() == ErrorType.NOT_FOUND)
                || error.getFieldErrors().values().stream().anyMatch(c -> c.stream()
                .anyMatch(e -> e.getType() == ErrorType.NOT_FOUND));
        if (hasNotFound) {
            return Response.Status.NOT_FOUND;
        }
        final boolean hasBadRequest = error.getErrors().stream().anyMatch(e -> e.getType() == ErrorType.BAD_REQUEST)
                || error.getFieldErrors().values().stream().anyMatch(c -> c.stream()
                .anyMatch(e -> e.getType() == ErrorType.BAD_REQUEST));
        if (hasBadRequest) {
            return Response.Status.BAD_REQUEST;
        }
        final boolean hasConflict = error.getErrors().stream().anyMatch(e -> e.getType() == ErrorType.CONFLICT)
                || error.getFieldErrors().values().stream().anyMatch(c -> c.stream()
                .anyMatch(e -> e.getType() == ErrorType.CONFLICT));
        if (hasConflict) {
            return Response.Status.CONFLICT;
        }
        return DiExceptionMapper.ExtraStatus.UNPROCESSABLE_ENTITY;
    }

    @NotNull
    private static <T> Response toResponse(@NotNull final Result<T, ErrorCollection<String, TypedError<String>>> result,
                                           @NotNull final Function<T, Response> mapSuccess,
                                           @NotNull final Function<ErrorCollection<String, TypedError<String>>, Response> mapError) {
        return result.match(new Result.Cases<T, ErrorCollection<String, TypedError<String>>, Response>() {
            @Nullable
            @Override
            public Response success(final T result) {
                return mapSuccess.apply(result);
            }
            @Nullable
            @Override
            public Response failure(@NotNull final ErrorCollection<String, TypedError<String>> error) {
                return mapError.apply(error);
            }
        });
    }

}
