package ru.yandex.qe.dispenser.ws.debug;

import java.io.Serializable;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.concurrent.TimeUnit;

import javax.naming.Referenceable;

import com.google.common.base.Stopwatch;
import com.mchange.v2.c3p0.AbstractComboPooledDataSource;
import com.mchange.v2.sql.filter.FilterConnection;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LoggingDataSource extends AbstractComboPooledDataSource implements Serializable, Referenceable {
    private static final Logger LOG = LoggerFactory.getLogger(LoggingDataSource.class);

    private long warningThreashold = 1000L;

    public LoggingDataSource() {
    }

    public LoggingDataSource(final boolean autoregister) {
        super(autoregister);
    }

    public LoggingDataSource(final String configName) {
        super(configName);
    }

    @Override
    public Connection getConnection() throws SQLException {
        //dont want to read spec for argument calculation order
        final Stopwatch stopwatch = Stopwatch.createStarted();
        return new LoggingConnectionWrapper(stopwatch, super.getConnection());
    }

    public void setWarningThreashold(final long warningThreashold) {
        this.warningThreashold = warningThreashold;
    }

    @Override
    public Connection getConnection(@NotNull final String user, @NotNull final String password) throws SQLException {
        final Stopwatch stopwatch = Stopwatch.createStarted();
        return new LoggingConnectionWrapper(stopwatch, super.getConnection(user, password));
    }

    private class LoggingConnectionWrapper extends FilterConnection {
        private final Stopwatch connectionStopwatch;

        private LoggingConnectionWrapper(final Stopwatch stopwatch, @NotNull final Connection conn) {
            super(conn);
            this.connectionStopwatch = Stopwatch.createStarted();
            long duration = stopwatch.elapsed(TimeUnit.MILLISECONDS);
            if (duration < warningThreashold) {
                LOG.debug("Connection acquired at {} ms.", duration);
            } else {
                LOG.warn("Connection acquired at {} ms.", duration);
            }
        }

        @Override
        public void close() throws SQLException {
            final long execTime = connectionStopwatch.elapsed(TimeUnit.MILLISECONDS);
            if (execTime < warningThreashold) {
                LOG.debug("Connection is closed ({} ms.)", execTime);
            } else {
                LOG.warn("Connection is closed ({} ms.)", execTime);
            }
            super.close();
        }
    }

}
