package ru.yandex.qe.dispenser.ws.hooks;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.collect.HashBasedTable;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Multimap;
import com.google.common.collect.Sets;
import com.google.common.collect.Table;
import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.qe.dispenser.api.v1.DiAmount;
import ru.yandex.qe.dispenser.api.v1.DiUnit;
import ru.yandex.qe.dispenser.domain.QuotaChangeRequest;
import ru.yandex.qe.dispenser.domain.Resource;
import ru.yandex.qe.dispenser.domain.Segment;
import ru.yandex.qe.dispenser.domain.util.MoreCollectors;
import ru.yandex.qe.dispenser.domain.util.PropertyUtils;
import ru.yandex.qe.dispenser.ws.quota.request.SetResourceAmountBody;

public final class ResourcePreorderFormUtils {
    private static final Logger LOG = LoggerFactory.getLogger(ResourcePreorderFormUtils.class);

    private static final String IGNORED_RESOURCES_PROPERTY = "ignored_resources";
    private static final String FORM_UTILS_ENTITY = "resource_preorder_form";

    private static final Table<String, String, String> RESOURCE_KEY_BY_SERVICE_AND_YP_KEY = HashBasedTable.create(3, 5);
    private static final Map<String, String> OLD_STYLE_RESOURCE_KEY_MAPPING = ImmutableMap.of("mem", "ram", "gpu_q", "gpu");

    static {
        RESOURCE_KEY_BY_SERVICE_AND_YP_KEY.put("YP", "cpu", "cpu_segmented");
        RESOURCE_KEY_BY_SERVICE_AND_YP_KEY.put("YP", "mem", "ram_segmented");
        RESOURCE_KEY_BY_SERVICE_AND_YP_KEY.put("YP", "hdd", "hdd_segmented");
        RESOURCE_KEY_BY_SERVICE_AND_YP_KEY.put("YP", "ssd", "ssd_segmented");
        RESOURCE_KEY_BY_SERVICE_AND_YP_KEY.put("YP", "gpu_q", "gpu_segmented");
        RESOURCE_KEY_BY_SERVICE_AND_YP_KEY.put("Qloud", "cpu", "cpu_segmented");
        RESOURCE_KEY_BY_SERVICE_AND_YP_KEY.put("Qloud", "mem", "ram_segmented");
        RESOURCE_KEY_BY_SERVICE_AND_YP_KEY.put("Qloud", "hdd", "hdd_segmented");
        RESOURCE_KEY_BY_SERVICE_AND_YP_KEY.put("RTC (GenCfg)", "mem", "ram");
        RESOURCE_KEY_BY_SERVICE_AND_YP_KEY.put("RTC (GenCfg)", "gpu_q", "gpu");
    }

    private ResourcePreorderFormUtils() {
    }

    static List<SetResourceAmountBody.ChangeBody> parseQuota(final ResourcePreorderFormBody body, final List<QuotaChangeRequest.Change> changes) {

        final boolean useNewResources = body.isNewResources();
        final List<QuotaBySegment> quotaBySegments = getQuotaBySegments(body.getService(), body.getQuota(), useNewResources);

        final List<SetResourceAmountBody.ChangeBody> result = new ArrayList<>();

        final Multimap<Pair<String, Set<String>>, QuotaChangeRequest.Change> changesByResourceKeyAndSegment = changes.stream()
                .collect(MoreCollectors.toLinkedMultimap(c -> Pair.of(c.getResource().getPublicKey(), c.getSegments().stream().map(Segment::getPublicKey).collect(Collectors.toSet())), Function.identity()));

        for (final QuotaBySegment quotaBySegment : quotaBySegments) {
            final Set<String> segments = new HashSet<>();
            segments.add(quotaBySegment.loc);
            if (useNewResources && quotaBySegment.segment != null) {
                segments.add(quotaBySegment.segment);
            }

            quotaBySegment.quotaByResource.forEach((res, amount) -> {
                final List<QuotaChangeRequest.Change> matchedChanges = new ArrayList<>(changesByResourceKeyAndSegment.get(Pair.of(res, segments)));
                matchedChanges.sort(Comparator.comparing(c1 -> c1.getKey().getBigOrder().getId()));

                if (matchedChanges.isEmpty()) {
                    throw new IllegalArgumentException("Выдаваемый ресурс (" + res + " , " + segments + ") не запрашивался");
                }

                QuotaChangeRequest.Change firstChage = matchedChanges.iterator().next();

                final Resource resource = firstChage.getResource();
                final DiUnit baseUnit = resource.getType().getBaseUnit();
                final long advanceAmount = baseUnit.convert(amount);

                final long unAllocated = matchedChanges.stream()
                        .mapToLong(c -> c.getAmount() - c.getAmountAllocated())
                        .sum();

                if (unAllocated < advanceAmount) {
                    final DiAmount.Humanized humanized = DiAmount.of(advanceAmount, baseUnit).humanize();
                    throw new IllegalArgumentException(String.format("Ресурс (%s, %s) невозможно выдать в заданном объеме (%s %s)", res, segments,
                            humanized.getStringValue(), humanized.getAbbreviation()));
                }

                long toAllocate = advanceAmount;
                for (final QuotaChangeRequest.Change change : matchedChanges) {
                    if (toAllocate == 0) {
                        break;
                    }
                    final long diff = Math.min(toAllocate, change.getAmount() - change.getAmountAllocated());
                    toAllocate -= diff;

                    final long amountAllocatedTotalBaseUnit = change.getAmountAllocated() + diff;
                    final DiAmount amountAllocatedTotal = DiAmount.of(amountAllocatedTotalBaseUnit, baseUnit);

                    result.add(new SetResourceAmountBody.ChangeBody(
                            change.getResource().getService().getKey(),
                            change.getKey().getBigOrder().getId(),
                            res, segments, null, amountAllocatedTotal));
                }
            });
        }

        LOG.debug("Parse result: {}", result);
        return result;
    }

    @NotNull
    static List<QuotaBySegment> getQuotaBySegments(final String service, final String quotas, final boolean newResources) {
        final List<QuotaBySegment> quotaBySegments = new ArrayList<>();
        for (final String quota : quotas.split("\n")) {
            if (quota.isEmpty()) {
                continue;
            }
            quotaBySegments.add(QuotaBySegment.valueOf(service, quota, newResources));
        }
        return quotaBySegments;
    }

    private static void throwParseError(final String message, final String quota) {
        throw new IllegalArgumentException("Ошибка разбора : " + message + ". Строка: " + quota);
    }

    static class QuotaBySegment {
        String loc;
        String segment;

        Map<String, DiAmount> quotaByResource = new HashMap<>(5);
        Set<String> ignoredResources;

        QuotaBySegment() {
            this.ignoredResources = PropertyUtils.readStringOptional(FORM_UTILS_ENTITY, IGNORED_RESOURCES_PROPERTY)
                    .<Set<String>>map(s -> Sets.newHashSet(s.split(",")))
                    .orElse(Collections.emptySet());
        }

        public static QuotaBySegment valueOf(final String service, final String quota, final boolean newResources) {
            final String[] splitByRes = quota.split("-");
            if (splitByRes.length < 2) {
                throwParseError("Ожидается хотя бы 2 части разделенные \"-\"", quota);
            }

            final QuotaBySegment result = new QuotaBySegment();

            try {
                for (final String res : splitByRes) {
                    final String[] split = res.split(":");
                    final String key = split[0];
                    if (split.length != 2) {
                        throwParseError("Ожидается значение после " + key + "\":\"", quota);
                    }
                    final String value = split[1];
                    final Map<String, String> mapping = newResources ? RESOURCE_KEY_BY_SERVICE_AND_YP_KEY.row(service) : OLD_STYLE_RESOURCE_KEY_MAPPING;
                    result.processResource(mapping, quota, key, value);
                }
            } catch (NumberFormatException e) {
                throwParseError(e.getMessage(), quota);
            }

            if (result.loc == null) {
                throwParseError("Ожидается поле loc в запросе", quota);
            }

            return result;
        }

        private void processResource(final Map<String, String> mapping, final String quota, final String key, final String value) {
            final String quotaKey = mapping.getOrDefault(key, key);
            switch (key) {
                case "loc":
                    loc = value;
                    break;
                case "seg":
                    segment = value;
                    break;
                case "cpu":
                    getDiAmountO(value, DiUnit.CORES, DiUnit.PERMILLE_CORES).ifPresent(x -> quotaByResource.put(quotaKey, x));
                    break;
                case "mem":
                    getDiAmountO(value, DiUnit.GIBIBYTE, DiUnit.BYTE).ifPresent(x -> quotaByResource.put(quotaKey, x));
                    break;
                case "hdd":
                    getDiAmountO(value, DiUnit.TEBIBYTE, DiUnit.BYTE).ifPresent(x -> quotaByResource.put(quotaKey, x));
                    break;
                case "ssd":
                    getDiAmountO(value, DiUnit.TEBIBYTE, DiUnit.BYTE).ifPresent(x -> quotaByResource.put(quotaKey, x));
                    break;
                case "gpu_q":
                    getDiAmountO(value, DiUnit.COUNT, DiUnit.COUNT).ifPresent(x -> quotaByResource.put(quotaKey, x));
                    break;
                case "io_ssd":
                case "io_hdd":
                    getDiAmountO(value, DiUnit.MIBPS, DiUnit.BINARY_BPS).ifPresent(x -> quotaByResource.put(quotaKey, x));
                    break;
                default:
                    if (ignoredResources.contains(key)) {
                        return;
                    }
                    throwParseError("Неизвестный ресурс " + key, quota);
            }
        }

        @NotNull
        private static Optional<DiAmount> getDiAmountO(final String value, final DiUnit sourceUnit, final DiUnit targetUnit) {
            final double sourceDouble = Double.parseDouble(value.replace(',', '.'));
            final double targetAmount = targetUnit.convert(sourceDouble, sourceUnit);
            final long target = Math.round(targetAmount);
            return target == 0 ? Optional.empty() : Optional.of(DiAmount.of(target, targetUnit));
        }
    }
}
