package ru.yandex.qe.dispenser.ws.intercept;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;

import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.ServletResponseWrapper;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.google.common.base.Stopwatch;
import org.apache.commons.lang3.StringUtils;
import org.eclipse.jetty.server.Response;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;

import ru.yandex.qe.dispenser.ws.sensors.SensorsHolder;

public final class AccessLogEntry {

    private static final Logger ACCESS_LOG = LoggerFactory.getLogger("ACCESS_LOG");

    private static final String UNDEFINED = "-";
    private static final String X_DI_SOURCE_IP = "x-di-source-ip";
    private static final String X_DI_REMOTE_IP = "x-di-remote-ip";
    private static final String X_DI_REQUEST_HOST = "x-di-request-host";
    private static final String X_DI_METHOD = "x-di-method";
    private static final String X_DI_PATH_QUERY = "x-di-path-query";
    private static final String X_DI_PROTOCOL = "x-di-protocol";
    private static final String X_DI_STATUS = "x-di-status";
    private static final String X_DI_RESPONSE_SIZE = "x-di-response-size";
    private static final String X_DI_RESPONSE_TIME = "x-di-response-time";
    private static final String X_DI_UID = "x-di-uid";
    private static final String X_DI_TVM_CLIENT_ID = "x-di-tvm-client-id";
    private static final String X_DI_OAUTH_CLIENT_ID = "x-di-oauth-client-id";
    private static final String X_DI_OAUTH_CLIENT_NAME = "x-di-oauth-client-name";
    private static final String X_DI_EXT_REQ_ID = "x-di-ext-req-id";
    private static final String X_DI_INT_REQ_ID = "x-di-int-req-id";
    private static final String X_DI_USER_AGENT = "x-di-user-agent";
    private static final String X_DI_REFERER = "x-di-referer";
    private static final String X_DI_ENDPOINT = "x-di-endpoint";
    private static final String X_DI_FORWARDED_FOR = "x-di-forwarded-for";

    private final String login;
    private final String uid;
    private final String tvmClientId;
    private final String clientId;
    private final String clientName;
    private final String requestId;
    private final String innerRequestId;
    private final String requestHost;
    private final String sourceIp;
    private final String remoteIp;
    private final String userAgent;
    private final String referer;
    private final String method;
    private final String pathQuery;
    private final String protocol;
    private final String status;
    private final String responseSize;
    private final String responseTime;
    private final String endpoint;
    private final String forwarders;
    private final String service;
    private final boolean knownEndpoint;
    private final Long responseSizeBytes;
    private final Long responseTimeMillis;
    private final Integer statusCode;

    private AccessLogEntry(final String login, final String uid, final String tvmClientId, final String clientId,
                           final String clientName, final String requestId, final String innerRequestId, final String requestHost,
                           final String sourceIp, final String remoteIp, final String userAgent, final String referer,
                           final String method, final String pathQuery, final String protocol, final String status,
                           final String responseSize, final String responseTime, final String endpoint, final String forwarders,
                           final String service, final boolean knownEndpoint, final Long responseSizeBytes, final Long responseTimeMillis,
                           final Integer statusCode) {
        this.login = login;
        this.uid = uid;
        this.tvmClientId = tvmClientId;
        this.clientId = clientId;
        this.clientName = clientName;
        this.requestId = requestId;
        this.innerRequestId = innerRequestId;
        this.requestHost = requestHost;
        this.sourceIp = sourceIp;
        this.remoteIp = remoteIp;
        this.userAgent = userAgent;
        this.referer = referer;
        this.method = method;
        this.pathQuery = pathQuery;
        this.protocol = protocol;
        this.status = status;
        this.responseSize = responseSize;
        this.responseTime = responseTime;
        this.endpoint = endpoint;
        this.forwarders = forwarders;
        this.service = service;
        this.knownEndpoint = knownEndpoint;
        this.responseSizeBytes = responseSizeBytes;
        this.responseTimeMillis = responseTimeMillis;
        this.statusCode = statusCode;
    }

    public static EntryBuilder builder(final ServletRequest servletRequest) {
        return new EntryBuilder(servletRequest);
    }

    public void log() {
        final List<String> added = new ArrayList<>();
        addToMdc(X_DI_SOURCE_IP, sourceIp, added);
        addToMdc(X_DI_REMOTE_IP, remoteIp, added);
        addToMdc(X_DI_REQUEST_HOST, requestHost, added);
        addToMdc(X_DI_METHOD, method, added);
        addToMdc(X_DI_PATH_QUERY, pathQuery, added);
        addToMdc(X_DI_PROTOCOL, protocol, added);
        addToMdc(X_DI_STATUS, status, added);
        addToMdc(X_DI_RESPONSE_SIZE, responseSize, added);
        addToMdc(X_DI_RESPONSE_TIME, responseTime, added);
        addToMdc(X_DI_UID, uid, added);
        addToMdc(X_DI_TVM_CLIENT_ID, tvmClientId, added);
        addToMdc(X_DI_OAUTH_CLIENT_ID, clientId, added);
        addToMdc(X_DI_OAUTH_CLIENT_NAME, clientName, added);
        addToMdc(X_DI_EXT_REQ_ID, requestId, added);
        addToMdc(X_DI_INT_REQ_ID, innerRequestId, added);
        addToMdc(X_DI_USER_AGENT, userAgent, added);
        addToMdc(X_DI_REFERER, referer, added);
        addToMdc(X_DI_ENDPOINT, endpoint, added);
        addToMdc(X_DI_FORWARDED_FOR, forwarders, added);
        ACCESS_LOG.info("{} {} {} {} {} {} {} {}", StringUtils.defaultString(sourceIp, UNDEFINED), StringUtils.defaultString(method, UNDEFINED),
                StringUtils.defaultString(pathQuery, UNDEFINED), StringUtils.defaultString(status, UNDEFINED),
                StringUtils.defaultString(responseSize, UNDEFINED), StringUtils.defaultString(responseTime, UNDEFINED),
                StringUtils.defaultString(uid, UNDEFINED), StringUtils.defaultString(tvmClientId, UNDEFINED));
        removeFromMdc(added);
    }

    public void updateSensors(final SensorsHolder sensorsHolder) {
        sensorsHolder.onRequest(endpoint, service, responseTimeMillis, responseSizeBytes, knownEndpoint, statusCode);
    }

    private void removeFromMdc(final List<String> keys) {
        for (final String key : keys) {
            MDC.remove(key);
        }
    }

    private void addToMdc(final String key, final String value, final List<String> added) {
        if (value != null) {
            MDC.put(key, value);
            added.add(key);
        }
    }

    public static final class EntryBuilder {

        private static final String USER_AGENT = "User-Agent";
        private static final String REQUEST_ID = "X-Request-ID";
        private static final String FORWARDED_FOR_Y = "X-Forwarded-For-Y";
        private static final String REFERER = "Referer";
        private static final String FORWARDED_FOR = "X-Forwarded-For";

        private final String requestId;
        private final String requestHost;
        private final String sourceIp;
        private final String remoteIp;
        private final String userAgent;
        private final String referer;
        private final String method;
        private final String pathQuery;
        private final String protocol;
        private final String forwarders;
        private String login;
        private String uid;
        private String tvmClientId;
        private String clientId;
        private String clientName;
        private String innerRequestId;
        private String status;
        private String responseSize;
        private String responseTime;
        private String endpoint;
        private String service;
        private boolean knownEndpoint;
        private Long responseSizeBytes;
        private Long responseTimeMillis;
        private Integer statusCode;

        private EntryBuilder(final ServletRequest servletRequest) {
            this.requestId = fromHttpServletRequest(servletRequest, r -> r.getHeader(REQUEST_ID));
            this.requestHost = servletRequest.getServerName();
            this.sourceIp = fromHttpServletRequest(servletRequest, this::getSourceIp);
            this.remoteIp = servletRequest.getRemoteAddr();
            this.userAgent = fromHttpServletRequest(servletRequest, r -> r.getHeader(USER_AGENT));
            this.referer = fromHttpServletRequest(servletRequest, r -> r.getHeader(REFERER));
            this.method = fromHttpServletRequest(servletRequest, HttpServletRequest::getMethod);
            this.pathQuery = fromHttpServletRequest(servletRequest, this::getPathQuery);
            this.protocol = servletRequest.getProtocol();
            this.forwarders = fromHttpServletRequest(servletRequest, r -> r.getHeader(FORWARDED_FOR));
        }

        public EntryBuilder setClientId(final String clientId) {
            this.clientId = clientId;
            return this;
        }

        public EntryBuilder setClientName(final String clientName) {
            this.clientName = clientName;
            return this;
        }

        public EntryBuilder addResult(final ServletResponse response, final ServletRequest request, final Stopwatch stopwatch) {
            this.statusCode = fromHttpServletResponse(response, HttpServletResponse::getStatus);
            this.status = String.valueOf(this.statusCode);
            this.responseSizeBytes = fromHttpServletResponse(response, this::getResponseSize);
            this.responseSize = String.valueOf(this.responseSizeBytes);
            final Object loginAttr = request.getAttribute(AccessLogFilter.LOGIN);
            final Object uidAttr = request.getAttribute(AccessLogFilter.UID);
            final Object innerRequestIdAttr = request.getAttribute(AccessLogFilter.INNER_REQ_ID);
            final Object endpointAttr = request.getAttribute(AccessLogFilter.ENDPOINT);
            final Object tvmClientIdAttr = request.getAttribute(AccessLogFilter.TVM_CLIENT_ID);
            final Object serviceAttr = request.getAttribute(AccessLogFilter.ENDPOINT_SERVICE);
            this.login = loginAttr instanceof String ? (String) loginAttr : null;
            this.uid = uidAttr instanceof String ? (String) uidAttr : null;
            this.innerRequestId = innerRequestIdAttr instanceof String ? (String) innerRequestIdAttr : null;
            this.endpoint = endpointAttr instanceof String ? (String) endpointAttr : null;
            this.responseTimeMillis = stopwatch.elapsed(TimeUnit.MILLISECONDS);
            this.responseTime = millisecondsToSecondsString(this.responseTimeMillis);
            this.tvmClientId = tvmClientIdAttr instanceof String ? (String) tvmClientIdAttr : null;
            this.service = serviceAttr instanceof String ? (String) serviceAttr : null;
            if (this.endpoint == null) {
                this.knownEndpoint = statusCode != HttpServletResponse.SC_NOT_FOUND;
                this.endpoint = fallbackEndpoint(request);
            } else {
                this.knownEndpoint = true;
            }

            // для запросов в swagger ставим статичный url, чтобы не плодить label'ы в solomon'е
            final String SWAGGER_ENDPOINT = "GET /api/api-docs";
            if (StringUtils.startsWith(endpoint, SWAGGER_ENDPOINT)) {
                this.endpoint = SWAGGER_ENDPOINT;
            }
            return this;
        }

        public AccessLogEntry build() {
            return new AccessLogEntry(login, uid, tvmClientId, clientId, clientName, requestId, innerRequestId, requestHost, sourceIp,
                    remoteIp, userAgent, referer, method, pathQuery, protocol, status, responseSize, responseTime, endpoint, forwarders,
                    service, knownEndpoint, responseSizeBytes, responseTimeMillis, statusCode);
        }

        private <T> T fromHttpServletRequest(final ServletRequest servletRequest, final Function<HttpServletRequest, T> f) {
            if (servletRequest instanceof HttpServletRequest) {
                return f.apply((HttpServletRequest) servletRequest);
            }
            return null;
        }

        private <T> T fromHttpServletResponse(final ServletResponse servletResponse, final Function<HttpServletResponse, T> f) {
            if (servletResponse instanceof HttpServletResponse) {
                return f.apply((HttpServletResponse) servletResponse);
            }
            return null;
        }

        private String getSourceIp(final HttpServletRequest request) {
            final String forwardedForY = request.getHeader(FORWARDED_FOR_Y);
            return forwardedForY != null ? forwardedForY : request.getRemoteAddr();
        }

        private String getPathQuery(final HttpServletRequest request) {
            final String requestUri = request.getRequestURI();
            final String queryString = request.getQueryString();
            if (requestUri == null && queryString == null) {
                return null;
            }
            if (queryString == null) {
                return requestUri;
            }
            if (requestUri == null) {
                return queryString;
            }
            return requestUri + "?" + queryString;
        }

        private Long getResponseSize(final HttpServletResponse response) {
            ServletResponse currentResponse = response;
            final Set<ServletResponse> seen = new HashSet<>();
            while (true) {
                if (seen.contains(currentResponse)) {
                    return null;
                }
                seen.add(currentResponse);
                if (currentResponse instanceof Response) {
                    return ((Response) currentResponse).getContentCount();
                }
                if (currentResponse instanceof ServletResponseWrapper) {
                    currentResponse = ((ServletResponseWrapper) currentResponse).getResponse();
                } else {
                    return null;
                }
            }
        }

        private String millisecondsToSecondsString(final long millis) {
            return millis / 1000 + "." + String.format("%03d", millis % 1000);
        }

        private String fallbackEndpoint(final ServletRequest request) {
            final String requestMethod = fromHttpServletRequest(request, HttpServletRequest::getMethod);
            final String requestUri = fromHttpServletRequest(request, HttpServletRequest::getRequestURI);
            if (requestMethod == null || requestUri == null) {
                return null;
            }
            return requestMethod + " " + requestUri;
        }

    }

}
