package ru.yandex.qe.dispenser.ws.intercept;

import java.io.IOException;
import java.util.HashSet;
import java.util.Set;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.jetbrains.annotations.NotNull;

/**
 * Created by Farrukh Karimov on 18.07.2019
 * Если запрос пришёл из 'доверенного' домена, то этот фильтр
 * добавляет нужные заголовки в ответе ( response ) - для осуществления кросс-доменных запросов
 * <p>
 * Доверенные домены - добавляются во время создания экземпляра этого класса.
 * Для тестирования, можно добавить пару своих ( локальных ) доменов, и
 * сделать из своих ( локальных ) доменов - любой запрос в Диспенсер
 */
public final class CorsAccessFilter implements Filter {
    private final Set<String> trustedDomainsNames;

    CorsAccessFilter() {
        trustedDomainsNames = new HashSet<>();

        trustedDomainsNames.add("yt.yandex-team.ru");
        trustedDomainsNames.add("http://yt.yandex-team.ru");
        trustedDomainsNames.add("https://yt.yandex-team.ru");

        trustedDomainsNames.add("beta.yt.yandex-team.ru");
        trustedDomainsNames.add("http://beta.yt.yandex-team.ru");
        trustedDomainsNames.add("https://beta.yt.yandex-team.ru");

        trustedDomainsNames.add("beta.yt.yandex.net");
        trustedDomainsNames.add("http://beta.yt.yandex.net");
        trustedDomainsNames.add("https://beta.yt.yandex.net");

        trustedDomainsNames.add("testing.yt.yandex-team.ru");
        trustedDomainsNames.add("http://testing.yt.yandex-team.ru");
        trustedDomainsNames.add("https://testing.yt.yandex-team.ru");

        trustedDomainsNames.add("yt.yandex.net");
        trustedDomainsNames.add("http://yt.yandex.net");
        trustedDomainsNames.add("https://yt.yandex.net");

        trustedDomainsNames.add("yp-quota-distributor.n.yandex-team.ru");
        trustedDomainsNames.add("http://yp-quota-distributor.n.yandex-team.ru");
        trustedDomainsNames.add("https://yp-quota-distributor.n.yandex-team.ru");

        // тут можно добавить пару локальных доменов для тестирования фильтра
    }

    @Override
    public void init(@NotNull final FilterConfig filterConfig) {
    }

    @Override
    public void doFilter(@NotNull final ServletRequest request,
                         @NotNull final ServletResponse response,
                         @NotNull final FilterChain chain) throws IOException, ServletException {
        final HttpServletRequest req = (HttpServletRequest) request;
        final HttpServletResponse resp = (HttpServletResponse) response;

        final String urlDomain = req.getHeader("origin");
        if (trustedDomainsNames.contains(urlDomain)) {
            addHeadersForCORS(req, resp, urlDomain);
        }

        chain.doFilter(request, response);
    }

    /**
     * Метод добавляет заголовки таким образом, чтобы кросс-доменный запрос для домена domain - проходили успешно
     *
     * @param request
     * @param response
     * @param domain
     */
    private void addHeadersForCORS(@NotNull final HttpServletRequest request, @NotNull final HttpServletResponse response, @NotNull final String domain) {
        response.addHeader("Access-Control-Allow-Origin", domain);
        response.addHeader("Access-Control-Allow-Methods", "DELETE,POST,GET,PUT,PATCH,OPTIONS,HEAD");
        response.addHeader("Access-Control-Allow-Credentials", "true");
        response.addHeader("Access-Control-Allow-Headers", request.getHeader("Access-Control-Request-Headers"));
    }

    @Override
    public void destroy() {
    }
}
