package ru.yandex.qe.dispenser.ws.intercept;

import java.io.IOException;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.annotation.WebFilter;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;

import org.springframework.web.context.WebApplicationContext;
import org.springframework.web.context.support.WebApplicationContextUtils;
import org.springframework.web.filter.DelegatingFilterProxy;

import ru.yandex.qe.spring.PropertyResolverBean;
import ru.yandex.qe.spring.RequestIdentity;

/**
 * @author lvovich
 */
@WebFilter(filterName = "springSecurity",
        urlPatterns = {"/*"},
        asyncSupported = true)
public class SecurityFilter implements Filter {

    public static final String YA_SESSION_ID_COOKIE = "session_id";
    public static final String YA_SSL_SESSION_ID_COOKIE = "sessionid2";
    public static final String X_FORWARDED_FOR_HEADER = "X-Forwarded-For";
    public static final String X_CSRF_TOKEN_HEADER ="X-CSRF-Token";
    public static final String AUTHORIZATION_HEADER = "Authorization";

    private Filter delegate = new NullFilter();

    @Override
    public void init(final FilterConfig filterConfig) throws ServletException {
        delegate = createDelegateFilter(filterConfig);
        delegate.init(filterConfig);
    }

    protected Filter createDelegateFilter(FilterConfig filterConfig) throws ServletException {
        WebApplicationContext wac =
                WebApplicationContextUtils.getWebApplicationContext(filterConfig.getServletContext());
        PropertyResolverBean propertyResolver = wac.getBean(PropertyResolverBean.class);
        String disableFlag = propertyResolver == null ? null : propertyResolver.getPropertyValue("security.disable");
        if ("true".equals(disableFlag)) {
            return delegate;
        }
        return new DelegatingFilterProxy("springSecurityFilterChain", wac);
    }

    @Override
    public void doFilter(final ServletRequest request, final ServletResponse response, final FilterChain chain) throws IOException, ServletException {
        RequestIdentity.reset();
        HttpServletRequest httpRequest = (HttpServletRequest) request;
        final RequestIdentity context = new RequestIdentity();
        copySessionCookieToContext(httpRequest, context);
        copyHeaderToContext(httpRequest, context, AUTHORIZATION_HEADER);
        copyHeaderToContext(httpRequest, context, X_FORWARDED_FOR_HEADER);
        copyHeaderToContext(httpRequest, context, X_CSRF_TOKEN_HEADER);
        context.setRemoteAddress(request.getRemoteAddr());
        context.enter();
        delegate.doFilter(request, response, chain);
    }

    private void copySessionCookieToContext(final HttpServletRequest httpRequest, final RequestIdentity context) {
        Cookie[] cookies = httpRequest.getCookies();
        if (cookies == null) {
            return;
        }
        for (final Cookie cookie : cookies) {
            if (cookie.getName().equalsIgnoreCase(YA_SESSION_ID_COOKIE)) {
                context.setSessionCookie(cookie);
            } else if (cookie.getName().equalsIgnoreCase(YA_SSL_SESSION_ID_COOKIE)) {
                context.setSslSessionCookie(cookie);
            }
        }
    }

    private void copyHeaderToContext(final HttpServletRequest httpRequest, final RequestIdentity context, final String headerName) {
        final String value = httpRequest.getHeader(headerName);
        context.setHeader(headerName, value);
    }

    @Override
    public void destroy() {
        delegate.destroy();
    }

    private static class NullFilter implements Filter {
        @Override
        public void init(final FilterConfig filterConfig) throws ServletException {
            // nothing to do
        }

        @Override
        public void doFilter(final ServletRequest request, final ServletResponse response, final FilterChain chain) throws IOException, ServletException {
            chain.doFilter(request, response);
        }

        @Override
        public void destroy() {
            // nothing to do
        }
    }
}
