package ru.yandex.qe.dispenser.ws.intercept;

import java.util.Arrays;
import java.util.Locale;

import org.apache.cxf.interceptor.Fault;
import org.apache.cxf.message.Message;
import org.apache.cxf.phase.Phase;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

import ru.yandex.qe.dispenser.api.DiRequest;
import ru.yandex.qe.dispenser.client.v1.impl.DiRequestImpl;
import ru.yandex.qe.dispenser.domain.hierarchy.HierarchySupplier;
import ru.yandex.qe.dispenser.domain.hierarchy.Session;
import ru.yandex.qe.dispenser.ws.Idempotent;

@Component
public final class SessionInitializer extends InterceptorBase<Message> {
    private static final String[] NO_CACHE_PATHS = {
            "/reinitialize"
    };

    private static final Logger LOG = LoggerFactory.getLogger(SessionInitializer.class);

    public static final Locale RU = new Locale("ru");

    @Autowired
    @Lazy
    private HierarchySupplier cachingHierarchy;

    @Value("${hierarchy.enabled}")
    private boolean hierarchyEnabled;

    public SessionInitializer() {
        super(Phase.PRE_UNMARSHAL);
    }

    @Override
    public void handleMessage(@NotNull final Message message) throws Fault {
        clear();
        init(message);
    }

    private void clear() {
        Session.HIERARCHY.remove();
        Session.ERROR.remove();
        Session.REQID.remove();
        Session.WHOAMI.remove();
        Session.USER_LOCALE.remove();
    }

    private void init(@NotNull final Message message) {
        final DiRequest<?> request = new DiRequestImpl(message);
        final String path = request.getPath();
        final boolean cache = this.hierarchyEnabled && Arrays.stream(NO_CACHE_PATHS).noneMatch(path::contains);
        if (cache) {
            LOG.debug("Hierarchy enabled for request");
            Session.HIERARCHY.set(cachingHierarchy.get());
        } else {
            LOG.debug("Hierarchy disabled for request");
        }

        final String reqId = request.getQueryParam(Idempotent.REQUEST_ID);
        if (reqId != null) {
            Session.REQID.set(reqId);
        }

        setLocalization(request);
    }

    private void setLocalization(final DiRequest<?> request) {
        final String langHeader = request.getFirstHeader("Accept-Language");
        if (langHeader == null) {
            return;
        }
        switch (langHeader) {
            case "ru" -> Session.USER_LOCALE.set(RU);
            case "en" -> Session.USER_LOCALE.set(Locale.ENGLISH);
            default -> Session.USER_LOCALE.set(Locale.ENGLISH);
        }
    }
}
