package ru.yandex.qe.dispenser.ws.param;

import java.io.IOException;
import java.io.InputStream;
import java.lang.annotation.Annotation;
import java.lang.reflect.Type;
import java.util.Collection;
import java.util.Collections;
import java.util.stream.Collectors;

import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.MultivaluedMap;
import javax.ws.rs.ext.MessageBodyReader;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonNode;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.qe.dispenser.api.util.SerializationUtils;
import ru.yandex.qe.dispenser.api.v1.request.DiServiceEntityReference;
import ru.yandex.qe.dispenser.domain.Entity;
import ru.yandex.qe.dispenser.domain.EntitySpec;
import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.Service;
import ru.yandex.qe.dispenser.domain.dao.entity.EntityDao;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;
import ru.yandex.qe.dispenser.ws.reqbody.EntityOwnershipBody;

public class EntityOwnershipBodyReader implements MessageBodyReader<EntityOwnershipBody> {

    @Autowired
    private EntityDao entityDao;

    @Override
    public boolean isReadable(final Class<?> type,
                              final Type genericType,
                              final Annotation[] annotations,
                              final MediaType mediaType) {
        return type.equals(EntityOwnershipBody.class);
    }

    @Override
    public EntityOwnershipBody readFrom(final Class<EntityOwnershipBody> type,
                                        final Type genericType,
                                        final Annotation[] annotations,
                                        final MediaType mediaType,
                                        final MultivaluedMap<String, String> httpHeaders,
                                        final InputStream entityStream) throws IOException, WebApplicationException {
        final JsonNode json = SerializationUtils.readValue(entityStream, JsonNode.class);

        final JsonNode entitiesNode = json.get("entities");
        final Collection<Entity.Key> entityKeys = entitiesNode == null ? Collections.emptySet() :
                SerializationUtils.convertValue(entitiesNode, new TypeReference<Collection<DiServiceEntityReference>>() {
                })
                        .stream()
                        .map(EntityOwnershipBodyReader::convertEntityReferenceToKey)
                        .collect(Collectors.toSet());
        final Collection<Entity> entities = entityDao.readPresent(entityKeys).values();

        final JsonNode projectsNode = json.get("projectKyes");
        final Collection<Project> projects = projectsNode == null ? Collections.emptySet() :
                SerializationUtils.toStringSet(projectsNode).stream()
                        .map(projectKey -> Hierarchy.get().getProjectReader().readExisting(projectKey))
                        .collect(Collectors.toSet());

        final JsonNode inLeafsNode = json.get("inLeafs");
        final boolean inLeafs = inLeafsNode != null && json.get("inLeafs").asBoolean();

        return new EntityOwnershipBody(entities, projects, inLeafs);
    }

    @NotNull
    private static Entity.Key convertEntityReferenceToKey(@NotNull final DiServiceEntityReference reference) {
        final Service service = Hierarchy.get().getServiceReader().read(reference.getServiceKey());
        final EntitySpec spec = Hierarchy.get().getEntitySpecReader().read(new EntitySpec.Key(reference.getSpecificationKey(), service));
        return new Entity.Key(reference.getKey(), spec);
    }
}
