package ru.yandex.qe.dispenser.ws.param;

import java.util.Optional;
import java.util.function.Supplier;
import java.util.stream.Stream;

import javax.ws.rs.QueryParam;

import com.google.common.base.Suppliers;

import ru.yandex.qe.dispenser.api.util.NumberUtils;

public class PageParam {
    public static final String PAGE_PARAM = "page";
    public static final String PAGE_SIZE_PARAM = "pageSize";
    public static final String LIMIT_PARAM = "limit";
    public static final long FIRST_PAGE_NUMBER = 1;

    private static final long DEFAULT_PAGE_SIZE = 50; // TODO
    private static final long MIN_PAGE_SIZE = 1; // TODO
    private static final long MAX_PAGE_SIZE = 1000; // TODO

    @QueryParam(PAGE_PARAM)
    private Long pageNumberParam;
    @QueryParam(PAGE_SIZE_PARAM)
    private Long pageSizeParam;
    @QueryParam(LIMIT_PARAM)
    private Long limitParam;

    public PageParam(final Long pageNumberParam, final Long pageSizeParam, final Long limitParam) {
        this.pageNumberParam = pageNumberParam;
        this.pageSizeParam = pageSizeParam;
        this.limitParam = limitParam;
    }

    private final Supplier<Long> pageNumberSupplier = Suppliers.memoize(this::computePageNumber)::get;
    private final Supplier<Long> pageSizeSupplier = Suppliers.memoize(this::computePageSize)::get;

    public long getPageNumber() {
        return pageNumberSupplier.get();
    }

    public long getPageSize() {
        return pageSizeSupplier.get();
    }

    private long computePageNumber() {
        final long number = Optional.ofNullable(pageNumberParam).orElse(FIRST_PAGE_NUMBER);
        return NumberUtils.requirePositive(number, "Page number");
    }

    private long computePageSize() {
        final long size = Stream.of(Optional.ofNullable(pageSizeParam), Optional.ofNullable(limitParam))
                .filter(Optional::isPresent)
                .map(Optional::get)
                .findFirst()
                .orElse(DEFAULT_PAGE_SIZE);
        return NumberUtils.requireInRange(size, MIN_PAGE_SIZE, MAX_PAGE_SIZE, "Page size");
    }
}
