package ru.yandex.qe.dispenser.ws.param;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.ws.rs.QueryParam;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.Iterables;
import com.google.common.collect.Multimap;
import org.jetbrains.annotations.NotNull;

import ru.yandex.qe.dispenser.api.v1.DiAmount;
import ru.yandex.qe.dispenser.api.v1.DiCheck;
import ru.yandex.qe.dispenser.api.v1.DiCheckCompareType;
import ru.yandex.qe.dispenser.api.v1.DiCheckValue;
import ru.yandex.qe.dispenser.api.v1.DiCheckValueType;
import ru.yandex.qe.dispenser.api.v1.DiUnit;
import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.Segment;
import ru.yandex.qe.dispenser.domain.dao.segment.SegmentUtils;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;
import ru.yandex.qe.dispenser.domain.util.MoreCollectors;

@ParametersAreNonnullByDefault
public class QuotaCheckParam {
    public static final String PROJECT_KEY = "projectKey";
    public static final String QUOTA_SPEC_KEY = "quotaSpecKey";
    public static final String VALUE_TYPE = "valueType";
    public static final String VALUE = "value";
    public static final String VALUE_UNIT = "valueUnit";
    public static final String COMPARE_TYPE = "compareType";
    public static final String SEGMENTS = "segments";

    @QueryParam(PROJECT_KEY)
    private Project project;

    @QueryParam(QUOTA_SPEC_KEY)
    private QuotaSpecFilterParam quotaSpecKey;

    @QueryParam(VALUE_TYPE)
    private DiCheckValueType valueType;

    @QueryParam(VALUE)
    private Double value;

    @Nullable
    @QueryParam(VALUE_UNIT)
    private DiUnit valueUnit;

    @QueryParam(COMPARE_TYPE)
    private DiCheckCompareType compareType;

    @QueryParam(SEGMENTS)
    private Set<String> segments;

    public QuotaCheckParam(final Project project, final QuotaSpecFilterParam quotaSpecKey, final DiCheckValueType valueType, final Double value,
                           @Nullable final DiUnit valueUnit, final DiCheckCompareType compareType, final Set<String> segments) {
        this.project = project;
        this.quotaSpecKey = quotaSpecKey;
        this.valueType = valueType;
        this.value = value;
        this.valueUnit = valueUnit;
        this.compareType = compareType;
        this.segments = segments;
    }

    public Value get() {
        return new Value(
                project,
                quotaSpecKey,
                valueType,
                value,
                valueUnit,
                compareType,
                segments
        );
    }

    private static <K, V> V first(final Multimap<K, V> multimap, final K key) {
        return Iterables.getFirst(multimap.get(key), null);
    }

    public static Value fromPath(final String path) {
        final String queryParams = path.substring(path.indexOf("?") + 1);
        final String[] split = queryParams.split("&");

        final Multimap<String, String> paramValues = Arrays.stream(split)
                .map(pair -> pair.split("="))
                .collect(MoreCollectors.toLinkedMultimap(pair -> pair[0], pair -> pair[1]));

        final DiCheckValueType valueType = DiCheckValueType.valueOf(first(paramValues, VALUE_TYPE));

        return new Value(
                Hierarchy.get().getProjectReader().readExisting(first(paramValues, PROJECT_KEY)),
                new QuotaSpecFilterParam(first(paramValues, QUOTA_SPEC_KEY)),
                valueType,
                Double.valueOf(first(paramValues, VALUE)),
                valueType == DiCheckValueType.ABSOLUTE ? DiUnit.valueOf(first(paramValues, VALUE_UNIT)) : null,
                DiCheckCompareType.valueOf(first(paramValues, COMPARE_TYPE)),
                new HashSet<>(paramValues.get(SEGMENTS))
        );
    }

    public static Value fromCheck(final Project project, final DiCheck.Body check) {

        final Object value = check.getAmount().getValue();

        final double resultValue;
        DiUnit valueUnit = null;
        if (value instanceof DiAmount) {
            resultValue = (double) ((DiAmount) value).getValue();
            valueUnit = ((DiAmount) value).getUnit();
        } else {
            resultValue = (double) value;
        }

        return new Value(
                project,
                new QuotaSpecFilterParam(check.getQuotaSpecKey()),
                check.getAmount().getType(),
                resultValue,
                valueUnit,
                check.getCompareType(),
                check.getSegments()
        );
    }

    @NotNull
    public static class Value {
        @NotNull
        private final Project project;

        @NotNull
        private final QuotaSpecFilterParam quotaSpecKey;

        @NotNull
        private final DiCheckValueType valueType;

        @NotNull
        private final Double value;

        @Nullable
        private final DiUnit valueUnit;

        @NotNull
        private final DiCheckCompareType compareType;

        @NotNull
        private final Set<Segment> segments;
        private final @NotNull Set<String> segmentKeys;

        public Value(final @NotNull Project project,
                     final @NotNull QuotaSpecFilterParam quotaSpecKey,
                     final @NotNull DiCheckValueType valueType, final @NotNull Double value,
                     @Nullable final DiUnit valueUnit,
                     final @NotNull DiCheckCompareType compareType,
                     final @NotNull Set<String> segments) {
            this.project = project;
            this.quotaSpecKey = quotaSpecKey;
            this.valueType = valueType;
            this.value = value;
            this.valueUnit = valueUnit;
            this.compareType = compareType;
            this.segments = SegmentUtils.getCompleteSegmentSet(quotaSpecKey.get().getResource(), segments);
            segmentKeys = segments;
        }


        @NotNull
        public Project getProject() {
            return project;
        }

        @NotNull
        public QuotaSpecFilterParam getQuotaSpecKey() {
            return quotaSpecKey;
        }

        @NotNull
        public Number getValue() {
            return value;
        }

        @Nullable
        public DiUnit getUnit() {
            return valueUnit;
        }

        @NotNull
        public DiCheckCompareType getCompareType() {
            return compareType;
        }

        @NotNull
        public DiCheckValue<?> getCheckAmount() {
            if (valueType == DiCheckValueType.ABSOLUTE) {
                return DiCheckValue.absolute(DiAmount.of(value.longValue(), valueUnit));
            }
            return DiCheckValue.percent(value);
        }

        @NotNull
        public Set<Segment> getSegments() {
            return segments;
        }

        public Set<String> getSegmentKeys() {
            return segmentKeys;
        }

        public String toQueryParams() {
            final HashMultimap<String, String> params = HashMultimap.create();
            params.put(PROJECT_KEY, project.getPublicKey());
            params.put(QUOTA_SPEC_KEY, quotaSpecKey.toString());
            params.put(VALUE_TYPE, valueType.name());
            params.put(VALUE, this.value.toString());
            params.put(COMPARE_TYPE, compareType.name());

            if (valueUnit != null) {
                params.put(VALUE_UNIT, valueUnit.name());
            }

            if (!segments.isEmpty()) {
                params.putAll(SEGMENTS, segmentKeys);
            }

            return params.entries()
                    .stream()
                    .map(entry -> entry.getKey() + "=" + entry.getValue())
                    .collect(Collectors.joining("&"));
        }

    }
}
