package ru.yandex.qe.dispenser.ws.param.batch;

import java.util.Collection;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import org.jetbrains.annotations.NotNull;

import ru.yandex.qe.dispenser.api.v1.DiPerformer;
import ru.yandex.qe.dispenser.api.v1.request.DiProcessingMode;
import ru.yandex.qe.dispenser.domain.Entity;
import ru.yandex.qe.dispenser.domain.Person;
import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.Service;
import ru.yandex.qe.dispenser.domain.distributed.Identifier;
import ru.yandex.qe.dispenser.domain.hierarchy.Role;
import ru.yandex.qe.dispenser.ws.aspect.AccessAspect;

final class Context {
    DiProcessingMode mode;
    Service service;
    Map<Entity.Key, Entity> entities;
    Map<String, Person> persons;
    Map<String, Project> projects;
    Map<Project.Key, Project> personalProjects;
    Identifier myIdentifier;

    @NotNull
    Set<Project> personalize(@NotNull final Collection<DiPerformer> performers) {
        return performers.stream()
                .map(p -> personalize(getProject(p), getPerson(p)))
                .collect(Collectors.toSet());
    }

    @NotNull
    private Project personalize(@NotNull final Project project, @NotNull final Person person) {
        return Project.withKey(Project.Key.of(project.getPublicKey(), person))
                .name(project.getName())
                .parent(project, true)
                .removed(project.isRemoved())
                .build();
    }

    void checkRealLeaf() {
        for (final Project p : projects.values()) {
            if (!p.isRealLeaf()) {
                throw new IllegalArgumentException("Can't acquire quota from not leaf project '" + p.getPublicKey() + "'!");
            }
        }
    }

    void checkMemberAccess(@NotNull final Collection<DiPerformer> performers) {
        performers.forEach(p -> AccessAspect.checkRole(getPerson(p), getProject(p), Role.MEMBER));
    }

    @NotNull
    Person getPerson(@NotNull final DiPerformer performer) {
        final String login = performer.getLogin();
        return Objects.requireNonNull(persons.get(login), "BUG! No person '" + login + "' in conext!");
    }

    @NotNull
    Project getProject(@NotNull final DiPerformer performer) {
        final String projectKey = performer.getProjectKey();
        return Objects.requireNonNull(projects.get(projectKey), "BUG! No project '" + projectKey + "' in context!");
    }

    @NotNull
    Project getPersonalProject(@NotNull final DiPerformer performer) {
        final Person person = getPerson(performer);
        final Project.Key key = Project.Key.of(performer.getProjectKey(), person);
        return Objects.requireNonNull(personalProjects.get(key), "BUG! No project '" + key + "' in context!");
    }
}
