package ru.yandex.qe.dispenser.ws.param.batch;

import java.io.IOException;
import java.io.InputStream;
import java.lang.annotation.Annotation;
import java.lang.reflect.Type;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.MultivaluedMap;
import javax.ws.rs.ext.MessageBodyReader;

import com.fasterxml.jackson.databind.JsonNode;
import com.google.common.collect.Sets;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.postgresql.util.PSQLException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.CannotAcquireLockException;
import org.springframework.dao.DataIntegrityViolationException;
import org.springframework.jdbc.UncategorizedSQLException;

import ru.yandex.qe.dispenser.api.util.SerializationUtils;
import ru.yandex.qe.dispenser.api.v1.DiPerformer;
import ru.yandex.qe.dispenser.api.v1.request.DiEntity;
import ru.yandex.qe.dispenser.api.v1.request.DiEntityReference;
import ru.yandex.qe.dispenser.api.v1.request.DiEntityUsage;
import ru.yandex.qe.dispenser.api.v1.request.DiOperation;
import ru.yandex.qe.dispenser.api.v1.request.DiProcessingMode;
import ru.yandex.qe.dispenser.domain.Entity;
import ru.yandex.qe.dispenser.domain.Person;
import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.dao.entity.EntityDao;
import ru.yandex.qe.dispenser.domain.distributed.Identifier;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;
import ru.yandex.qe.dispenser.domain.util.CollectionUtils;
import ru.yandex.qe.dispenser.domain.util.StreamUtils;
import ru.yandex.qe.dispenser.ws.common.domain.exceptions.ConflictException;
import ru.yandex.qe.dispenser.ws.common.domain.exceptions.TooManyRequestsException;
import ru.yandex.qe.dispenser.ws.intercept.TransactionWrapper;
import ru.yandex.qe.dispenser.ws.reqbody.QuotaChangeBody;

import static ru.yandex.qe.dispenser.api.v1.request.DiProcessingMode.valueOf;

public class QuotaChangeBodyReader implements MessageBodyReader<QuotaChangeBody> {
    private static final Logger LOG = LoggerFactory.getLogger(QuotaChangeBodyReader.class);

    @Autowired
    private EntityDao entityDao;

    @Autowired
    private Identifier myIdentifier;

    @Override
    public boolean isReadable(@NotNull final Class<?> type,
                              @NotNull final Type genericType,
                              @NotNull final Annotation[] annotations,
                              @NotNull final MediaType mediaType) {
        return type.equals(QuotaChangeBody.class);
    }

    @NotNull
    @Override
    public QuotaChangeBody readFrom(@NotNull final Class<QuotaChangeBody> type,
                                    @NotNull final Type genericType,
                                    @NotNull final Annotation[] annotations,
                                    @NotNull final MediaType mediaType,
                                    @NotNull final MultivaluedMap<String, String> httpHeaders,
                                    @NotNull final InputStream entityStream) throws IOException, WebApplicationException {
        final JsonNode json = SerializationUtils.readValue(entityStream, JsonNode.class);

        final Context ctx = new Context();
        ctx.mode = valueOf(json.get("mode").asText());
        ctx.service = Hierarchy.get().getServiceReader().read(json.get("serviceKey").asText());
        ctx.myIdentifier = myIdentifier;
        final Map<Integer, DiOperation<?>> id2operation = new HashMap<>();
        for (final JsonNode operationJson : json.get("operations")) {
            final int id = operationJson.get("id").asInt();
            final DiOperation<?> operation = SerializationUtils.convertValue(operationJson.get("operation"), DiOperation.class);
            id2operation.put(id, operation);
        }
        final Collection<DiOperation<?>> operations = id2operation.values();

        final List<DiPerformer> performers = operations.stream()
                .map(DiOperation::getPerformer)
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
        readRealProjects(ctx, performers);

        try {
            createEntities(ctx, operations);
        } catch (DataIntegrityViolationException e) {
            throw new ConflictException(e.getMessage(), e);
        } catch (CannotAcquireLockException e) {
            throw new TooManyRequestsException(e.getMessage(), e);
        } catch (UncategorizedSQLException e) {
            if ((e.getCause() instanceof PSQLException) &&
                    Objects.equals(((PSQLException) e.getCause()).getSQLState(), "55P03")) {
                throw new TooManyRequestsException(e.getMessage(), e);
            }
            throw e;
        }

        if (needNewTransaction(ctx, performers)) {
            TransactionWrapper.INSTANCE.execute(() -> createPersonalProjects(ctx, performers));
        } else {
            createPersonalProjects(ctx, performers);
        }

        return ConversionUtils.convert(id2operation, ctx);
    }

    private void readRealProjects(@NotNull final Context ctx, @NotNull final Collection<DiPerformer> performers) {
        final Set<String> publicProjectKeys = performers.stream().map(DiPerformer::getProjectKey).collect(Collectors.toSet());
        ctx.projects = Hierarchy.get().getProjectReader().readByPublicKeys(publicProjectKeys);
        ctx.checkRealLeaf();
    }

    private void createEntities(@NotNull final Context ctx, @NotNull final Collection<DiOperation<?>> operations) {
        final List<Entity> absentEntities = StreamUtils.instances(operations.stream().map(DiOperation::getAction), DiEntity.class)
                .map(e -> ConversionUtils.convert(e, ctx))
                .collect(Collectors.toList());
        entityDao.createAll(absentEntities);

        final Set<Entity.Key> allKeys = new HashSet<>();
        absentEntities.stream().map(Entity::getKey).forEach(allKeys::add);
        operations.stream()
                .map(DiOperation::getAction)
                .map(this::toReference)
                .filter(Objects::nonNull)
                .map(e -> ConversionUtils.convert(e, ctx))
                .forEach(allKeys::add);

        ctx.entities = readEntities(allKeys, ctx.mode);
    }

    @NotNull
    private Map<Entity.Key, Entity> readEntities(@NotNull final Collection<Entity.Key> keys, @NotNull final DiProcessingMode mode) {
        switch (mode) {
            case ROLLBACK_ON_ERROR:
                return entityDao.readAllForUpdate(keys);
            case IGNORE_UNKNOWN_ENTITIES_AND_USAGES:
                return entityDao.readPresentForUpdate(keys);
            default:
                throw new UnsupportedOperationException("Unknown processing mode: " + mode);
        }
    }

    private boolean needNewTransaction(@NotNull final Context ctx, @NotNull final Collection<DiPerformer> performers) {
        final Set<String> logins = CollectionUtils.map(performers, DiPerformer::getLogin);
        final Set<String> presentLogins = Hierarchy.get().getPersonReader().readPresent(logins).keySet();
        if (!Sets.difference(logins, presentLogins).isEmpty()) {
            return true;
        }
        ctx.persons = Hierarchy.get().getPersonReader().readAll(logins);
        final Set<Project.Key> personalProjectKeys = CollectionUtils.keys(ctx.personalize(performers));
        final Set<Project.Key> presentPersonalProjectKeys = Hierarchy.get().getProjectReader().readPresent(personalProjectKeys).keySet();
        return !Sets.difference(personalProjectKeys, presentPersonalProjectKeys).isEmpty();
    }

    private void createPersonalProjects(@NotNull final Context ctx, @NotNull final Collection<DiPerformer> performers) {
        if (ctx.persons == null) {
            final List<String> logins = performers.stream().map(DiPerformer::getLogin).collect(Collectors.toList());
            final Set<Person> persons = Hierarchy.get().getPersonReader().readPersonsByLogins(logins);

            ctx.persons = persons.stream()
                    .collect(Collectors.toMap(Person::getLogin, Function.identity()));
        }
        // validation
        ctx.checkMemberAccess(performers);

        final Set<Project> personalProjects = ctx.personalize(performers);
        ctx.personalProjects = Hierarchy.get().getProjectReader().createAllIfAbsent(personalProjects);
    }

    @Nullable
    private DiEntityReference toReference(@NotNull final Object action) {
        if (action instanceof DiEntity) {
            return null;
        }
        if (action.getClass() == DiEntityReference.class) {
            return (DiEntityReference) action;
        }
        if (action instanceof DiEntityUsage) {
            return ((DiEntityUsage) action).getEntity();
        }
        return null;
    }
}
