package ru.yandex.qe.dispenser.ws.quota.request;

import java.math.BigDecimal;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumMap;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.inject.Inject;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.NotNull;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import ru.yandex.qe.dispenser.api.util.SerializationUtils;
import ru.yandex.qe.dispenser.api.v1.DiQuotaRequestHistoryEventType;
import ru.yandex.qe.dispenser.api.v1.ExpandQuotaChangeRequest;
import ru.yandex.qe.dispenser.domain.MessageHelper;
import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.QuotaChangeRequest;
import ru.yandex.qe.dispenser.domain.QuotaChangeRequestHistoryEvent;
import ru.yandex.qe.dispenser.domain.base_resources.BaseResourceChange;
import ru.yandex.qe.dispenser.domain.base_resources.WithBaseResourceChanges;
import ru.yandex.qe.dispenser.domain.dao.base_resources.BaseResourceChangeDao;
import ru.yandex.qe.dispenser.domain.dao.history.request.QuotaChangeRequestHistoryDao;
import ru.yandex.qe.dispenser.domain.dao.project.ProjectReader;
import ru.yandex.qe.dispenser.domain.dao.quota.request.QuotaChangeRequestDao;
import ru.yandex.qe.dispenser.domain.dao.quota.request.QuotaChangeRequestReader;
import ru.yandex.qe.dispenser.domain.index.LongIndexBase;
import ru.yandex.qe.dispenser.ws.base_resources.impl.BaseResourcesMapper;
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.ChangeOwningCostContext;
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.ChangesWithOwningCostForUpdate;
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.QuotaChangeOwningCostManager;
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.QuotaChangeRequestOwningCostContext;
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.formula.ProviderOwningCostFormula;
import ru.yandex.qe.dispenser.ws.quota.request.ticket.QuotaChangeRequestTicketManager;
import ru.yandex.qe.dispenser.ws.quota.request.unbalanced.QuotaChangeUnbalancedManager;
import ru.yandex.qe.dispenser.ws.quota.request.workflow.context.CreateRequestContext;
import ru.yandex.qe.dispenser.ws.quota.request.workflow.context.PerformerContext;
import ru.yandex.qe.dispenser.ws.quota.request.workflow.context.UpdateRequestContext;

import static ru.yandex.qe.dispenser.domain.QuotaChangeRequestHistoryEvent.DATA_TYPE;

@Component
public class QuotaChangeRequestManager {

    public static final EnumSet<QuotaChangeRequest.Field> STATUS_FIELD = EnumSet.of(QuotaChangeRequest.Field.STATUS);
    public static final EnumSet<QuotaChangeRequest.Field> ISSUE_KEY_FIELD = EnumSet.of(QuotaChangeRequest.Field.ISSUE_KEY);
    public static final EnumSet<QuotaChangeRequest.Field> READY_FOR_ALLOCATION_FIELD = EnumSet.of(QuotaChangeRequest.Field.READY_FOR_ALLOCATION);

    private static final Map<QuotaChangeRequest.Field, DiQuotaRequestHistoryEventType> FIELD_TO_EVENT_TYPE_MAPPING =
            ImmutableMap.<QuotaChangeRequest.Field, DiQuotaRequestHistoryEventType>builder()
                    .put(QuotaChangeRequest.Field.STATUS, DiQuotaRequestHistoryEventType.STATUS_UPDATE)
                    .put(QuotaChangeRequest.Field.ISSUE_KEY, DiQuotaRequestHistoryEventType.ISSUE_KEY_UPDATE)
                    .put(QuotaChangeRequest.Field.CHANGES, DiQuotaRequestHistoryEventType.CHANGES_UPDATE)
                    .put(QuotaChangeRequest.Field.PROJECT, DiQuotaRequestHistoryEventType.PROJECT_UPDATE)
                    .build();

    private final QuotaChangeRequestDao quotaChangeRequestDao;
    private final QuotaChangeRequestTicketManager ticketManager;
    private final ProjectReader projectReader;
    private final MessageHelper messageHelper;
    private final QuotaChangeRequestHistoryDao quotaChangeRequestHistoryDao;
    private final BaseResourcesMapper baseResourcesMapper;
    private final BaseResourceChangeDao baseResourceChangeDao;
    private final QuotaChangeOwningCostManager quotaChangeOwningCostManager;
    private final QuotaChangeUnbalancedManager quotaChangeUnbalancedManager;

    @Inject
    public QuotaChangeRequestManager(final QuotaChangeRequestDao quotaChangeRequestDao,
                                     final QuotaChangeRequestTicketManager ticketManager,
                                     final ProjectReader projectReader,
                                     final MessageHelper messageHelper,
                                     final QuotaChangeRequestHistoryDao quotaChangeRequestHistoryDao,
                                     final BaseResourcesMapper baseResourcesMapper,
                                     final BaseResourceChangeDao baseResourceChangeDao,
                                     final QuotaChangeOwningCostManager quotaChangeOwningCostManager,
                                     final QuotaChangeUnbalancedManager quotaChangeUnbalancedManager) {
        this.quotaChangeRequestDao = quotaChangeRequestDao;
        this.ticketManager = ticketManager;
        this.projectReader = projectReader;
        this.messageHelper = messageHelper;
        this.quotaChangeRequestHistoryDao = quotaChangeRequestHistoryDao;
        this.baseResourcesMapper = baseResourcesMapper;
        this.baseResourceChangeDao = baseResourceChangeDao;
        this.quotaChangeOwningCostManager = quotaChangeOwningCostManager;
        this.quotaChangeUnbalancedManager = quotaChangeUnbalancedManager;
    }

    public QuotaChangeRequest setStatus(final QuotaChangeRequest request, final QuotaChangeRequest.Status status,
                                        final PerformerContext ctx, boolean suppressSummon) {

        final QuotaChangeRequest modifiedRequest = request.copyBuilder()
                .status(status)
                .updated(System.currentTimeMillis())
                .build();

        quotaChangeRequestDao.update(modifiedRequest);

        updateHistory(modifiedRequest, request, STATUS_FIELD, DiQuotaRequestHistoryEventType.STATUS_UPDATE, ctx);

        ticketManager.applyTicketChanges(modifiedRequest, request, ctx, suppressSummon);

        return modifiedRequest;
    }

    public void setStatusBatch(final QuotaChangeRequestReader.QuotaChangeRequestFilter filter,
                               final QuotaChangeRequest.Status status,
                               final PerformerContext ctx, final boolean suppressSummon) {

        final long now = System.currentTimeMillis();

        final List<Pair<QuotaChangeRequest, QuotaChangeRequest>> requestChanges =
                quotaChangeRequestDao.setStatuses(filter, status, now).collect(Collectors.toList());

        updateHistory(requestChanges, STATUS_FIELD, DiQuotaRequestHistoryEventType.BATCH_STATUS_UPDATE, ctx);
        requestChanges.forEach(p -> ticketManager.applyTicketChanges(p.getLeft(), p.getRight(), ctx, suppressSummon));
    }

    public List<QuotaChangeRequest> setStatusBatch(final Collection<QuotaChangeRequest> requests,
                                                   final QuotaChangeRequest.Status status,
                                                   final PerformerContext ctx, final boolean suppressSummon) {
        final long updated = System.currentTimeMillis();

        final List<Pair<QuotaChangeRequest, QuotaChangeRequest>> updatedAndOldRequests = requests.stream()
                .map(request -> Pair.of(request.copyBuilder().status(status).updated(updated).build(), request))
                .collect(Collectors.toList());

        final List<QuotaChangeRequest> updatedRequests = updatedAndOldRequests.stream().map(Pair::getLeft).collect(Collectors.toList());
        quotaChangeRequestDao.updateAll(updatedRequests);

        updateHistory(updatedAndOldRequests, STATUS_FIELD, DiQuotaRequestHistoryEventType.BATCH_STATUS_UPDATE, ctx);

        updatedAndOldRequests.forEach(p -> ticketManager
                .applyTicketChanges(p.getLeft(), p.getRight(), ctx, suppressSummon));

        return updatedRequests;

    }

    public List<MoveResult> moveToProject(final List<QuotaChangeRequest> requests, final int toProjectAbcId,
                                          final PerformerContext ctx, final boolean suppressSummon) {
        final Map<Long, QuotaChangeRequest> requestById = requests.stream()
                .collect(Collectors.toMap(LongIndexBase::getId, Function.identity()));

        final Project projectTo = projectReader.readByAbcServiceId(toProjectAbcId);

        final Set<Long> requestIds = requestById.keySet();
        quotaChangeRequestDao.moveToProject(requestIds, projectTo);

        final Map<Long, QuotaChangeRequest> updatedRequests = quotaChangeRequestDao.read(requestIds);

        final List<Pair<QuotaChangeRequest, QuotaChangeRequest>> updatedAndOldRequests = updatedWithOriginalRequests(requestById, updatedRequests);

        updateHistory(updatedAndOldRequests, EnumSet.of(QuotaChangeRequest.Field.PROJECT), DiQuotaRequestHistoryEventType.PROJECT_UPDATE, ctx);

        return requests.stream()
                .map(originalReq -> {
                    final long reqId = originalReq.getId();
                    final QuotaChangeRequest updatedRequest = updatedRequests.get(reqId);

                    if (updatedRequest == null) {
                        return new MoveResult(originalReq, "Error: cannot read request after update");
                    }
                    if (updatedRequest.getProject().equals(originalReq.getProject())) {
                        return new MoveResult(originalReq, "Error: project not updated");
                    }
                    if (originalReq.getTrackerIssueKey() == null) {
                        return new MoveResult(originalReq, "No Tracker issue");
                    }

                    final boolean res = ticketManager.applyTicketChanges(updatedRequest, originalReq, ctx, suppressSummon);
                    if (!res) {
                        return new MoveResult(originalReq, "Error: cannot update Tracker issue");
                    }
                    return new MoveResult(originalReq, "OK");
                })
                .collect(Collectors.toList());
    }

    public void addRequestGoalDataComments(final QuotaChangeRequest request, final UpdateRequestContext ctx) {
        final String commentHead = messageHelper.format("quota.request.ticket.requestGoalChanged", ctx.getPerson()) + "\n\n";
        final String comment = commentHead + ticketManager.getAnswersForComment(request);
        ticketManager.tryAddIssueComment(request, comment);
    }

    public void setReadyForAllocationState(final Collection<Long> requestIds, final boolean isReadyForAllocation, final PerformerContext ctx) {
        if (requestIds.isEmpty()) {
            return;
        }
        final Map<Long, QuotaChangeRequest> originalRequests = quotaChangeRequestDao.read(requestIds);
        quotaChangeRequestDao.setReadyForAllocationState(requestIds, isReadyForAllocation);
        final Map<Long, QuotaChangeRequest> updatedRequests = quotaChangeRequestDao.read(requestIds);

        updateHistory(updatedWithOriginalRequests(originalRequests, updatedRequests), READY_FOR_ALLOCATION_FIELD,
                DiQuotaRequestHistoryEventType.READY_FOR_ALLOCATION_STATE_UPDATE, ctx);
    }

    public void setShowAllocationNote(final Collection<Long> requestIds, final boolean showAllocationNote) {
        if (requestIds.isEmpty()) {
            return;
        }
        quotaChangeRequestDao.setShowAllocationNote(requestIds, showAllocationNote);
    }

    public void setChangesReadyAmount(final Collection<QuotaChangeRequest> affectedRequests, final Map<Long, Long> readyByChangeId, final PerformerContext ctx) {
        quotaChangeRequestDao.setChangesReadyAmount(readyByChangeId);
        updateChangesHistory(affectedRequests, DiQuotaRequestHistoryEventType.CHANGES_READY_AMOUNT_UPDATE, ctx);
    }

    public void setChangesAllocatedAmount(final Collection<QuotaChangeRequest> affectedRequests, final Map<Long, Long> readyByChangeId, final PerformerContext ctx) {
        quotaChangeRequestDao.setChangesAllocatedAmount(readyByChangeId);
        updateChangesHistory(affectedRequests, DiQuotaRequestHistoryEventType.CHANGES_ALLOCATED_AMOUNT_UPDATE, ctx);
    }

    public void setChangesAllocatingAmount(final Collection<QuotaChangeRequest> affectedRequests, final Map<Long, Long> allocatingByChangeId, final PerformerContext ctx) {
        quotaChangeRequestDao.setChangesAllocatingAmount(allocatingByChangeId);
        updateChangesHistory(affectedRequests, DiQuotaRequestHistoryEventType.CHANGES_ALLOCATING_AMOUNT_UPDATE, ctx);
    }

    private void updateChangesHistory(final Collection<QuotaChangeRequest> affectedRequests, final DiQuotaRequestHistoryEventType eventType, final PerformerContext ctx) {
        final Map<Long, QuotaChangeRequest> requestById = affectedRequests.stream()
                .collect(Collectors.toMap(LongIndexBase::getId, Function.identity()));
        final Map<Long, QuotaChangeRequest> updatedById = quotaChangeRequestDao.read(requestById.keySet());

        final List<Pair<QuotaChangeRequest, QuotaChangeRequest>> requests = updatedWithOriginalRequests(requestById, updatedById);

        final Instant now = Instant.now().truncatedTo(ChronoUnit.MILLIS);

        quotaChangeRequestHistoryDao.create(historyEvents(requests, EnumSet.of(QuotaChangeRequest.Field.CHANGES), eventType, now, ctx));
    }

    public WithBaseResourceChanges<QuotaChangeRequest> create(final QuotaChangeRequest request,
                                                              final CreateRequestContext ctx) {
        final QuotaChangeRequest created = quotaChangeRequestDao.create(request);
        quotaChangeRequestHistoryDao.create(createdEvent(created, ctx));
        Set<BaseResourceChange> baseChanges = baseResourceChangeDao
                .create(baseResourcesMapper.mapQuotaChangeRequestAsBaseChanges(created, true));
        return new WithBaseResourceChanges<>(created, Map.of(created.getId(), baseChanges));
    }

    public WithBaseResourceChanges<List<QuotaChangeRequest>> create(final Collection<QuotaChangeRequest> requests,
                                                                    final CreateRequestContext ctx) {
        final List<QuotaChangeRequest> createdRequests = requests.stream()
                .map(quotaChangeRequestDao::create)
                .collect(Collectors.toList());

        final List<QuotaChangeRequestHistoryEvent> events = createdRequests.stream()
                .map(req -> createdEvent(req, ctx))
                .collect(Collectors.toList());

        quotaChangeRequestHistoryDao.create(events);

        List<BaseResourceChange.Builder> baseChangeBuilders = createdRequests.stream()
                .flatMap(r -> baseResourcesMapper.mapQuotaChangeRequestAsBaseChanges(r, true).stream())
                .collect(Collectors.toList());
        Map<Long, Set<BaseResourceChange>> baseChanges = baseResourceChangeDao.create(baseChangeBuilders).stream()
                .collect(Collectors.groupingBy(BaseResourceChange::getQuotaRequestId, Collectors.toSet()));
        return new WithBaseResourceChanges<>(createdRequests, baseChanges);
    }

    private static QuotaChangeRequestHistoryEvent createdEvent(final QuotaChangeRequest created, final CreateRequestContext ctx) {
        return new QuotaChangeRequestHistoryEvent(
                ctx.getPerson().getId(),
                null,
                Instant.ofEpochMilli(created.getCreated()),
                null,
                created.getId(),
                DiQuotaRequestHistoryEventType.CREATE,
                Collections.emptyMap(),
                SerializationUtils.convertValue(created.toBasicView(), DATA_TYPE)
        );
    }

    public void updateChanges(final QuotaChangeRequest request,
                                                         final List<? extends QuotaChangeRequest.ChangeAmount> newChanges,
                                                         final PerformerContext ctx,
                                                         final QuotaChangeRequest.Builder updateBuilder) {
        final QuotaChangeRequest lockedRequest = quotaChangeRequestDao.readForUpdate(Collections.singleton(request.getId())).values().iterator().next();
        final List<QuotaChangeRequest.Change> updatedChanges = quotaChangeRequestDao.updateChangesAmount(lockedRequest, newChanges);

        ChangesWithOwningCostForUpdate changesWithCalculatedOwningCostForUpdate =
                quotaChangeOwningCostManager.getChangesWithCalculatedOwningCostForUpdate(
                QuotaChangeRequestOwningCostContext.builder()
                        .changeOwningCostContexts(getChangeOwningCostContexts(request, updatedChanges))
                        .originalChangeOwningCostContexts(getChangeOwningCostContexts(request,
                                request.getChanges()))
                        .mode(QuotaChangeRequestOwningCostContext.Mode.UPDATE)
                        .build());
        quotaChangeRequestDao.updateChangesOwningCost(
                changesWithCalculatedOwningCostForUpdate.getChangesWithCalculatedOwningCostForUpdateOwningCost());
        List<QuotaChangeRequest.Change> changesWithCalculatedOwningCost =
                changesWithCalculatedOwningCostForUpdate.getChangesWithCalculatedOwningCost();
        quotaChangeRequestHistoryDao.create(updateChangesEvent(request, changesWithCalculatedOwningCost, ctx));
        Long requestOwningCost = changesWithCalculatedOwningCost.stream()
                .map(change -> ProviderOwningCostFormula.owningCostToOutputFormat(change.getOwningCost()))
                .reduce(BigDecimal::add)
                .map(ProviderOwningCostFormula::owningCostToOutputFormat)
                .map(BigDecimal::longValueExact)
                .orElse(0L);
        updateBuilder.changes(changesWithCalculatedOwningCost)
                .requestOwningCost(requestOwningCost);
    }

    @NotNull
    private List<ChangeOwningCostContext> getChangeOwningCostContexts(QuotaChangeRequest request,
                                                                      List<QuotaChangeRequest.Change> changes) {
        return changes.stream()
                .map(change -> ChangeOwningCostContext.builder()
                        .change(change)
                        .abcServiceIdFromRequest(request)
                        .campaignFromRequest(request)
                        .build())
                .collect(Collectors.toList());
    }

    @Transactional(propagation = Propagation.REQUIRES_NEW)
    public void refreshBaseResourceChanges(List<Long> quotaRequestIds) {
        Map<Long, QuotaChangeRequest> lockedRequests = quotaChangeRequestDao.readForUpdate(quotaRequestIds);
        Set<BaseResourceChange> currentBaseChanges = baseResourceChangeDao
                .getByQuotaRequestIds(lockedRequests.keySet());
        List<BaseResourceChange.Builder> newBaseChanges = baseResourcesMapper
                .mapQuotaChangeRequestsAsBaseChanges(lockedRequests.values(), false);
        List<BaseResourceChange.Builder> baseChangesToCreate = new ArrayList<>();
        List<BaseResourceChange.Update> baseChangesToUpdate = new ArrayList<>();
        Set<BaseResourceChange> unmodifiedBaseChanges = new HashSet<>();
        Set<Long> baseChangesToDelete = new HashSet<>();
        baseResourcesMapper.splitBaseChangesUpdate(currentBaseChanges, newBaseChanges, baseChangesToCreate,
                baseChangesToUpdate, unmodifiedBaseChanges, baseChangesToDelete);
        baseResourceChangeDao.deleteByIds(baseChangesToDelete);
        baseResourceChangeDao.update(baseChangesToUpdate);
        baseResourceChangeDao.create(baseChangesToCreate);
    }

    private static QuotaChangeRequestHistoryEvent updateChangesEvent(final QuotaChangeRequest request,
                                                                     final List<QuotaChangeRequest.Change> newChanges,
                                                                     final PerformerContext ctx) {
        final Map<String, Object> oldData = new HashMap<>();
        final QuotaChangeRequest.Field changesField = QuotaChangeRequest.Field.CHANGES;
        oldData.put(changesField.getFieldName(), changesField.getView(request));

        final Map<String, Object> newData = new HashMap<>();
        newData.put(changesField.getFieldName(), newChanges.stream()
                .map(QuotaChangeRequest.Change::toView).collect(Collectors.toList()));

        return new QuotaChangeRequestHistoryEvent(ctx.getPerson().getId(), ctx.getTvmId(), Instant.now().truncatedTo(ChronoUnit.MILLIS), ctx.getComment(), request.getId(),
                DiQuotaRequestHistoryEventType.CHANGES_UPDATE, oldData, newData);
    }

    public WithBaseResourceChanges<QuotaChangeRequest> updateIssueKey(final QuotaChangeRequest request,
                                                                      final String issueKey,
                                                                      final PerformerContext ctx,
                                                                      final boolean alwaysLoadBaseChanges) {
        final QuotaChangeRequest.Builder updatedBuilder = request.copyBuilder()
                .trackerIssueKey(issueKey);
        return update(updatedBuilder, request, ISSUE_KEY_FIELD, ctx, alwaysLoadBaseChanges);
    }

    public WithBaseResourceChanges<QuotaChangeRequest> updateWithChanges(
            final QuotaChangeRequest.Builder updateBuilder,
            final QuotaChangeRequest oldRequest,
            final List<? extends QuotaChangeRequest.ChangeAmount> newChanges,
            final Set<QuotaChangeRequest.Field> updatedFields,
            final PerformerContext ctx,
            final boolean alwaysLoadBaseChanges) {
        updateChanges(oldRequest, newChanges, ctx, updateBuilder);

        boolean unbalanced = quotaChangeUnbalancedManager.calculateForUpdateRequest(oldRequest, newChanges);
        updateBuilder.unbalanced(unbalanced);

        return update(updateBuilder, oldRequest, updatedFields, ctx, alwaysLoadBaseChanges);
    }

    public WithBaseResourceChanges<QuotaChangeRequest> update(final QuotaChangeRequest.Builder updateBuilder,
                                                              final QuotaChangeRequest oldRequest,
                                                              final Set<QuotaChangeRequest.Field> updatedFields,
                                                              final PerformerContext ctx,
                                                              final boolean alwaysLoadBaseChanges) {
        updateBuilder.updated(System.currentTimeMillis());
        final QuotaChangeRequest updatedRequest = updateBuilder.build();
        final boolean updated = quotaChangeRequestDao.update(updatedRequest);

        final Sets.SetView<QuotaChangeRequest.Field> fieldsWithoutChanges = Sets.difference(updatedFields, Collections.singleton(QuotaChangeRequest.Field.CHANGES));
        if (updated && !fieldsWithoutChanges.isEmpty()) {

            final Map<DiQuotaRequestHistoryEventType, Set<QuotaChangeRequest.Field>> fieldsByType = new EnumMap<>(DiQuotaRequestHistoryEventType.class);

            for (final QuotaChangeRequest.Field field : fieldsWithoutChanges) {
                final DiQuotaRequestHistoryEventType type = FIELD_TO_EVENT_TYPE_MAPPING.getOrDefault(field, DiQuotaRequestHistoryEventType.FIELDS_UPDATE);
                final Set<QuotaChangeRequest.Field> fields = fieldsByType.computeIfAbsent(type, t -> EnumSet.noneOf(QuotaChangeRequest.Field.class));
                fields.add(field);
            }

            final List<QuotaChangeRequestHistoryEvent> history = new ArrayList<>();
            fieldsByType.forEach((t, fs) -> historyEvent(updatedRequest, oldRequest, fs, t, ctx).ifPresent(history::add));

            quotaChangeRequestHistoryDao.create(history);
        }
        Map<Long, Set<BaseResourceChange>> actualBaseChanges;
        if (updatedFields.contains(QuotaChangeRequest.Field.CHANGES)) {
            Set<BaseResourceChange> currentBaseChanges = baseResourceChangeDao
                    .getByQuotaRequestId(updatedRequest.getId());
            List<BaseResourceChange.Builder> newBaseChanges = baseResourcesMapper
                    .mapQuotaChangeRequestAsBaseChanges(updatedRequest, true);
            List<BaseResourceChange.Builder> baseChangesToCreate = new ArrayList<>();
            List<BaseResourceChange.Update> baseChangesToUpdate = new ArrayList<>();
            Set<BaseResourceChange> unmodifiedBaseChanges = new HashSet<>();
            Set<Long> baseChangesToDelete = new HashSet<>();
            baseResourcesMapper.splitBaseChangesUpdate(currentBaseChanges, newBaseChanges, baseChangesToCreate,
                    baseChangesToUpdate, unmodifiedBaseChanges, baseChangesToDelete);
            baseResourceChangeDao.deleteByIds(baseChangesToDelete);
            Set<BaseResourceChange> updatedChanges = baseResourceChangeDao.update(baseChangesToUpdate);
            Set<BaseResourceChange> createdChanges = baseResourceChangeDao.create(baseChangesToCreate);
            actualBaseChanges = Map.of(updatedRequest.getId(), Sets.union(Sets.union(createdChanges, updatedChanges),
                    unmodifiedBaseChanges));
        } else if (alwaysLoadBaseChanges) {
            actualBaseChanges = Map.of(updatedRequest.getId(),
                    baseResourceChangeDao.getByQuotaRequestId(updatedRequest.getId()));
        } else {
            actualBaseChanges = Map.of();
        }
        return new WithBaseResourceChanges<>(updatedRequest, actualBaseChanges);
    }

    @Transactional(propagation = Propagation.REQUIRED)
    public WithBaseResourceChanges<QuotaChangeRequest> getBaseResourceChanges(QuotaChangeRequest request) {
        return new WithBaseResourceChanges<>(request, Map.of(request.getId(),
                baseResourceChangeDao.getByQuotaRequestId(request.getId())));
    }

    @Transactional(propagation = Propagation.REQUIRED)
    public Set<BaseResourceChange> getBaseResourceChanges(long requestId, Set<ExpandQuotaChangeRequest> expand) {
        if (expand == null || !expand.contains(ExpandQuotaChangeRequest.BASE_RESOURCES)) {
            return Set.of();
        }
        return baseResourceChangeDao.getByQuotaRequestId(requestId);
    }

    @Transactional(propagation = Propagation.REQUIRED)
    public Map<Long, Set<BaseResourceChange>> getBaseResourceChanges(Set<Long> requestIds,
                                                                     Set<ExpandQuotaChangeRequest> expand) {
        if (expand == null || !expand.contains(ExpandQuotaChangeRequest.BASE_RESOURCES)) {
            return Map.of();
        }
        return baseResourceChangeDao.getByQuotaRequestIds(requestIds).stream()
                .collect(Collectors.groupingBy(BaseResourceChange::getQuotaRequestId, Collectors.toSet()));
    }

    private void updateHistory(final QuotaChangeRequest updated,
                               final QuotaChangeRequest old,
                               final Set<QuotaChangeRequest.Field> updatedFields,
                               final DiQuotaRequestHistoryEventType type,
                               final PerformerContext ctx) {
        final Optional<QuotaChangeRequestHistoryEvent> optionalEvent = historyEvent(updated, old, updatedFields, type, ctx);
        optionalEvent.ifPresent(quotaChangeRequestHistoryDao::create);
    }

    /**
     * @param requestChanges Pair.of(updatedRequest, oldRequest)
     **/
    private void updateHistory(final Collection<Pair<QuotaChangeRequest, QuotaChangeRequest>> requestChanges,
                               final Set<QuotaChangeRequest.Field> updatedFields,
                               final DiQuotaRequestHistoryEventType type,
                               final PerformerContext ctx) {
        quotaChangeRequestHistoryDao.create(historyEvents(requestChanges, updatedFields, type, ctx));
    }

    private static List<QuotaChangeRequestHistoryEvent> historyEvents(final Collection<Pair<QuotaChangeRequest, QuotaChangeRequest>> requestChanges,
                                                                      final Set<QuotaChangeRequest.Field> updatedFields,
                                                                      final DiQuotaRequestHistoryEventType type,
                                                                      final PerformerContext ctx) {
        return requestChanges.stream()
                .map(pair -> historyEvent(pair.getLeft(), pair.getRight(), updatedFields, type, ctx))
                .filter(Optional::isPresent)
                .map(Optional::get)
                .collect(Collectors.toList());
    }

    private static List<QuotaChangeRequestHistoryEvent> historyEvents(final Collection<Pair<QuotaChangeRequest, QuotaChangeRequest>> requestChanges,
                                                                      final Set<QuotaChangeRequest.Field> updatedFields,
                                                                      final DiQuotaRequestHistoryEventType type,
                                                                      final Instant updated,
                                                                      final PerformerContext ctx) {
        return requestChanges.stream()
                .map(pair -> historyEvent(pair.getLeft(), pair.getRight(), updatedFields, type, updated, ctx))
                .filter(Optional::isPresent)
                .map(Optional::get)
                .collect(Collectors.toList());
    }


    private static Optional<QuotaChangeRequestHistoryEvent> historyEvent(final QuotaChangeRequest updatedRequest,
                                                                         final QuotaChangeRequest oldRequest,
                                                                         final Set<QuotaChangeRequest.Field> updatedFields,
                                                                         final DiQuotaRequestHistoryEventType type,
                                                                         final PerformerContext ctx) {
        return historyEvent(updatedRequest, oldRequest, updatedFields, type, Instant.ofEpochMilli(updatedRequest.getUpdated()), ctx);
    }

    @SuppressWarnings("MethodWithTooManyParameters")
    private static Optional<QuotaChangeRequestHistoryEvent> historyEvent(final QuotaChangeRequest updatedRequest,
                                                                         final QuotaChangeRequest oldRequest,
                                                                         final Set<QuotaChangeRequest.Field> updatedFields,
                                                                         final DiQuotaRequestHistoryEventType type,
                                                                         final Instant updated,
                                                                         final PerformerContext ctx) {
        final Optional<Pair<Map<String, Object>, Map<String, Object>>> optionalDiff = requestDiff(oldRequest, updatedRequest, updatedFields);
        if (!optionalDiff.isPresent()) {
            return Optional.empty();
        }
        final Pair<Map<String, Object>, Map<String, Object>> diff = optionalDiff.get();
        return Optional.of(new QuotaChangeRequestHistoryEvent(ctx.getPerson().getId(), ctx.getTvmId(), updated, ctx.getComment(),
                oldRequest.getId(), type, diff.getLeft(), diff.getRight()));
    }

    private static Optional<Pair<Map<String, Object>, Map<String, Object>>> requestDiff(final QuotaChangeRequest oldRequest,
                                                                                        final QuotaChangeRequest updatedRequest,
                                                                                        final Set<QuotaChangeRequest.Field> updatedFields) {
        final Map<String, Object> oldFields = new HashMap<>();
        final Map<String, Object> newFields = new HashMap<>();
        for (final QuotaChangeRequest.Field updatedField : updatedFields) {
            final String fieldName = updatedField.getFieldName();
            final Object oldFieldView = updatedField.getView(oldRequest);
            final Object newFieldView = updatedField.getView(updatedRequest);
            if (!Objects.equals(oldFieldView, newFieldView)) {
                oldFields.put(fieldName, oldFieldView);
                newFields.put(fieldName, newFieldView);
            }
        }
        if (!newFields.isEmpty()) {
            return Optional.of(Pair.of(oldFields, newFields));
        } else {
            return Optional.empty();
        }
    }

    private static List<Pair<QuotaChangeRequest, QuotaChangeRequest>> updatedWithOriginalRequests(final Map<Long, QuotaChangeRequest> originalById,
                                                                                                  final Map<Long, QuotaChangeRequest> updateById) {
        return updateById.entrySet().stream()
                .map(e -> Pair.of(e.getValue(), originalById.get(e.getKey())))
                .collect(Collectors.toList());
    }

    public static class MoveResult {
        private final long requestId;
        private final String ticketId;
        private final String result;

        @JsonCreator
        public MoveResult(
                @JsonProperty("requestId") final long requestId,
                @JsonProperty("ticketId") final String ticketId,
                @JsonProperty("result") final String result) {
            this.requestId = requestId;
            this.ticketId = ticketId;
            this.result = result;
        }

        private MoveResult(final QuotaChangeRequest request, final String result) {
            this.requestId = request.getId();
            this.ticketId = request.getTrackerIssueKey();
            this.result = result;
        }

        public long getRequestId() {
            return requestId;
        }

        public String getTicketId() {
            return ticketId;
        }

        public String getResult() {
            return result;
        }
    }
}
