package ru.yandex.qe.dispenser.ws.quota.request;

import java.time.Clock;
import java.util.List;
import java.util.concurrent.TimeUnit;

import javax.inject.Inject;

import com.google.common.base.Stopwatch;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Profile;
import org.springframework.stereotype.Component;

import ru.yandex.qe.dispenser.domain.Person;
import ru.yandex.qe.dispenser.domain.QuotaChangeRequest;
import ru.yandex.qe.dispenser.domain.dao.quota.request.QuotaChangeRequestDao;
import ru.yandex.qe.dispenser.domain.dao.quota.request.QuotaChangeRequestFilterImpl;
import ru.yandex.qe.dispenser.domain.hierarchy.HierarchySupplier;
import ru.yandex.qe.dispenser.ws.quota.request.ticket.QuotaChangeRequestTicketManager;
import ru.yandex.qe.dispenser.ws.quota.request.workflow.context.PerformerContext;
import ru.yandex.qe.dispenser.solomon.SolomonHolder;
import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.Histogram;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

@Component
@Profile("secondary")
@SuppressWarnings("UnstableApiUsage")
public class ReCreateRequestTicketsTask {

    private static final Logger LOG = LoggerFactory.getLogger(ReCreateRequestTicketsTask.class);

    public static final String SENSOR_PREFIX = "request_tickets_recreate_task.";
    public static final String ELAPSED_TIME_SENSOR = SENSOR_PREFIX + "elapsed_time";
    public static final String ERROR_RATE_SENSOR = SENSOR_PREFIX + "error_rate";
    public static final String LAST_START_SENSOR = SENSOR_PREFIX + "time_since_last_start";
    public static final String LAST_SUCCESS_SENSOR = SENSOR_PREFIX + "time_since_last_success_end";

    private final QuotaChangeRequestDao requestDao;
    private final QuotaChangeRequestTicketManager ticketManager;
    private final QuotaChangeRequestManager quotaChangeRequestManager;
    private final HierarchySupplier hierarchySupplier;

    private volatile Person performer;
    private final String performerLogin;

    private final Histogram elapsedTime;
    private final Rate errorRate;
    private volatile long lastStart;
    private volatile long lastSuccessEnd;
    private final Clock clock = Clock.systemDefaultZone();
    private final Stopwatch stopwatch = Stopwatch.createUnstarted();

    @Inject
    public ReCreateRequestTicketsTask(final QuotaChangeRequestDao requestDao,
                                      final QuotaChangeRequestTicketManager ticketManager,
                                      final SolomonHolder solomonHolder,
                                      final QuotaChangeRequestManager quotaChangeRequestManager,
                                      final HierarchySupplier hierarchySupplier,
                                      @Value("${quota.request.tickets.recreate.performer}") final String performerLogin) {
        this.requestDao = requestDao;
        this.ticketManager = ticketManager;
        this.quotaChangeRequestManager = quotaChangeRequestManager;

        this.hierarchySupplier = hierarchySupplier;
        this.performerLogin = performerLogin;

        final MetricRegistry rootRegistry = solomonHolder.getRootRegistry();
        elapsedTime = rootRegistry.histogramRate(ELAPSED_TIME_SENSOR, Labels.of(), Histograms.exponential(22, 2, 1.0d));
        errorRate = rootRegistry.rate(ERROR_RATE_SENSOR, Labels.of());
        lastStart = clock.millis();
        rootRegistry.lazyGaugeInt64(LAST_START_SENSOR, Labels.of(), () -> clock.millis() - lastStart);
        lastSuccessEnd = clock.millis();
        rootRegistry.lazyGaugeInt64(LAST_SUCCESS_SENSOR, Labels.of(), () -> clock.millis() - lastSuccessEnd);

    }

    public void createTickets() {
        LOG.info("Status request tickets re-creating");
        stopwatch.start();
        lastStart = clock.millis();
        boolean success = false;

        try {
            process();
            success = true;
        } finally {
            if (!success) {
                errorRate.inc();
            } else {
                lastSuccessEnd = clock.millis();
            }
            final long elapsed = stopwatch.elapsed(TimeUnit.MILLISECONDS);
            elapsedTime.record(elapsed);
            stopwatch.reset();
        }

        LOG.info("Finish request tickets re-creating");

    }

    private void setPerformerIfNotExists() {
        if (performer == null) {
            hierarchySupplier.renewThreadHierarchy();
            this.performer = hierarchySupplier.get().getPersonReader().tryReadPersonByLogin(performerLogin)
                    .orElseThrow(() -> new IllegalArgumentException(String.format("Performer for task is required, but there's no person with login '%s'", performerLogin)));
        }
    }

    private void process() {
        setPerformerIfNotExists();
        final QuotaChangeRequestFilterImpl filter = new QuotaChangeRequestFilterImpl();
        filter.setWithoutTicket(true);
        final List<QuotaChangeRequest> requests = requestDao.read(filter);

        for (final QuotaChangeRequest request : requests) {
            if (request.getType() != QuotaChangeRequest.Type.RESOURCE_PREORDER) {
                continue;
            }
            final String trackerIssueKey = ticketManager.createTicketForQuotaChangeRequest(request);
            if (trackerIssueKey == null) {
                LOG.error("Can't create ticket for request id={}", request.getId());
                continue;
            }

            quotaChangeRequestManager.updateIssueKey(request, trackerIssueKey, new PerformerContext(performer), false);

            LOG.info("Update request id={}, ticket={}", request.getId(), trackerIssueKey);
        }
    }
}
