package ru.yandex.qe.dispenser.ws.quota.request.owning_cost;

import java.math.BigDecimal;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.OptionalLong;
import java.util.Set;
import java.util.stream.Collectors;

import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import ru.yandex.qe.dispenser.domain.QuotaChangeRequest;
import ru.yandex.qe.dispenser.domain.dao.quota.request.QuotaChangeRequestDao;
import ru.yandex.qe.dispenser.domain.index.LongIndexBase;
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.formula.ProviderOwningCostFormula;

/**
 * Transaction wrapper for page of request.
 */
@Component
public class QuotaChangeOwningCostRefreshTransactionWrapper {
    public static final long LIMIT = 100L;

    private final QuotaChangeRequestDao quotaChangeRequestDao;
    private final QuotaChangeOwningCostManager quotaChangeOwningCostManager;

    public QuotaChangeOwningCostRefreshTransactionWrapper(QuotaChangeRequestDao quotaChangeRequestDao,
                                                          QuotaChangeOwningCostManager quotaChangeOwningCostManager) {
        this.quotaChangeRequestDao = quotaChangeRequestDao;
        this.quotaChangeOwningCostManager = quotaChangeOwningCostManager;
    }

    @Transactional(propagation = Propagation.REQUIRES_NEW)
    public Long execute(Long fromId) {
        Set<Long> campaignIds = quotaChangeOwningCostManager.getCampaignIds();
        List<QuotaChangeRequest> list = getRequests(fromId, campaignIds);
        OptionalLong max = list.stream().mapToLong(LongIndexBase::getId).max();

        if (max.isEmpty()) {
            return null;
        }

        List<QuotaChangeRequest> requestsWithCalculatedOwningCost =
                quotaChangeOwningCostManager.getRequestsWithCalculatedOwningCost(list,
                        QuotaChangeRequestOwningCostContext.Mode.REFRESH);

        calculateAndUpdateChangeOwningCost(list, requestsWithCalculatedOwningCost);
        calculateAndUpdateRequestOwningCost(list, requestsWithCalculatedOwningCost);

        return max.getAsLong();
    }

    private void calculateAndUpdateChangeOwningCost(List<QuotaChangeRequest> list,
                                                    List<QuotaChangeRequest> requestsWithCalculatedOwningCost) {
        List<QuotaChangeRequest.Change> updatedChanges =
                requestsWithCalculatedOwningCost.stream()
                        .flatMap(request -> request.getChanges().stream())
                        .collect(Collectors.toList());

        Map<Long, BigDecimal> oldOwningCostByChangeId = list.stream()
                .flatMap(request -> request.getChanges().stream())
                .collect(Collectors.toMap(QuotaChangeRequest.Change::getId, QuotaChangeRequest.Change::getOwningCost));

        List<QuotaChangeRequest.Change> changesWithUpdatedOwningCost = updatedChanges.stream()
                .filter(change -> oldOwningCostByChangeId.containsKey(change.getId())
                        && !oldOwningCostByChangeId.get(change.getId()).equals(change.getOwningCost()))
                .collect(Collectors.toList());

        updateChangesOwningCost(changesWithUpdatedOwningCost);
    }

    private List<QuotaChangeRequest> getRequests(Long fromId, Set<Long> campaignIds) {
        return quotaChangeRequestDao.readRequestsByCampaignsForUpdate(campaignIds, fromId, LIMIT);
    }

    private void updateChangesOwningCost(List<QuotaChangeRequest.Change> changesWithCalculatedOwningCost) {
        quotaChangeRequestDao.updateChangesOwningCost(changesWithCalculatedOwningCost);
    }

    private void calculateAndUpdateRequestOwningCost(List<QuotaChangeRequest> list,
                                                     List<QuotaChangeRequest> requestsWithCalculatedOwningCost) {
        Map<Long, Long> oldRequestOwningCost = list.stream()
                .collect(Collectors.toMap(QuotaChangeRequest::getId, QuotaChangeRequest::getRequestOwningCost));
        Map<Long, Long> requestOwningCostMap = new HashMap<>();

        requestsWithCalculatedOwningCost
                .forEach(request -> request.getChanges().stream()
                        .map(change -> ProviderOwningCostFormula.owningCostToOutputFormat(change.getOwningCost()))
                        .reduce(BigDecimal::add)
                        .map(ProviderOwningCostFormula::owningCostToOutputFormat)
                        .ifPresent(requestOwningCost -> {
                            long id = request.getId();
                            if (oldRequestOwningCost.containsKey(id)) {
                                long requestOwningCostLong = requestOwningCost.longValueExact();
                                if (oldRequestOwningCost.get(id) != requestOwningCostLong) {
                                    requestOwningCostMap.put(id, requestOwningCostLong);
                                }
                            }
                        }));

        updateRequestChangesOwningCost(requestOwningCostMap);
    }

    private void updateRequestChangesOwningCost(Map<Long, Long> requestOwningCostMap) {
        quotaChangeRequestDao.updateRequestsOwningCost(requestOwningCostMap);
    }
}
