package ru.yandex.qe.dispenser.ws.quota.request.owning_cost.formula;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableMap;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import ru.yandex.qe.dispenser.api.util.EnumUtils;
import ru.yandex.qe.dispenser.api.v1.DiUnit;
import ru.yandex.qe.dispenser.domain.QuotaChangeRequest;
import ru.yandex.qe.dispenser.domain.dao.segment.SegmentUtils;
import ru.yandex.qe.dispenser.ws.bot.Provider;
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.ChangeOwningCostContext;
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.pricing.PricingModel;
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.pricing.QuotaChangeOwningCostTariffManager;

/**
 * Distbuild owning cost formula (DISPENSER-4384).
 *
 * @author Ruslan Kadriev <aqru@yandex-team.ru>
 */
@Component
public class DistbuildOwningCostFormula implements ProviderOwningCostFormula {
    private static final Logger LOG = LoggerFactory.getLogger(DistbuildOwningCostFormula.class);

    private static final Provider provider = Provider.DISTBUILD;
    private static final String DISTBUILD_COMPUTE_GUARANTEE_SLOTS_AUTOCHECK_SEGMENT = "distbuild.compute.guarantee.slots_autocheck_segment";
    private static final String DISTBUILD_COMPUTE_GUARANTEE_SLOTS_USER_SEGMENT = "distbuild.compute.guarantee.slots_user_segment";

    private static final Map<Segment, String> CPU_SKU_BY_SEGMENT = new EnumMap<>(Map.of(
            Segment.DISTBUILD_AUTOCHECK,  DISTBUILD_COMPUTE_GUARANTEE_SLOTS_AUTOCHECK_SEGMENT,
            Segment.DISTBUILD_USER,  DISTBUILD_COMPUTE_GUARANTEE_SLOTS_USER_SEGMENT
    ));

    private final QuotaChangeOwningCostTariffManager quotaChangeOwningCostTariffManager;
    private final String distbuildSegmentationKey;

    public DistbuildOwningCostFormula(@Value("${dispenser.distbuild.segmentation.key}") final String distbuildSegmentationKey,
                                      QuotaChangeOwningCostTariffManager quotaChangeOwningCostTariffManager) {
        this.quotaChangeOwningCostTariffManager = quotaChangeOwningCostTariffManager;
        this.distbuildSegmentationKey = distbuildSegmentationKey;
    }

    @Override
    public @NotNull String getProviderKey() {
        return provider.getServiceKey();
    }

    @Override
    public @NotNull Map<QuotaChangeRequest.ChangeKey, BigDecimal> calculateOwningCostFromContext(
            @NotNull Collection<ChangeOwningCostContext> changes) {
        Map<QuotaChangeRequest.ChangeKey, BigDecimal> result = new HashMap<>();
        Map<CampaignKey, List<ChangeOwningCostContext>> changesByCampaign = ProviderOwningCostFormula
                .groupByCampaign(changes, result);
        changesByCampaign.forEach((campaignKey, campaignChanges) -> {
            Map<String, ? extends PricingModel> pricingBySKU =
                    quotaChangeOwningCostTariffManager.getByProviderCampaign(provider, campaignKey.getKey()).stream()
                            .collect(Collectors.toMap(PricingModel::getSKU, Function.identity()));
            Map<QuotaChangeRequest.ChangeKey, BigDecimal> campaignCosts = campaignChanges.stream()
                    .map(ChangeOwningCostContext::getChange)
                    .collect(Collectors.toMap(QuotaChangeRequest.ChangeAmount::getKey,
                            change -> toOwningCost(change, pricingBySKU)));
            result.putAll(campaignCosts);
        });
        return result;
    }

    private BigDecimal toOwningCost(QuotaChangeRequest.Change change, Map<String, ? extends PricingModel> pricingBySKU) {
        BigDecimal result = DEFAULT_OWNING_COST;

        if (pricingBySKU.isEmpty()) {
            return result;
        }

        String publicKey = change.getResource().getPublicKey();
        Optional<Resource> resourcesO = Resource.byKey(publicKey);

        if (resourcesO.isPresent()) {
            switch (resourcesO.get()) {
                case CPU:
                    result = calculateForCPU(change, pricingBySKU);
                    break;
            }
        }

        return result;
    }

    private BigDecimal calculateForCPU(QuotaChangeRequest.Change change,
                                       Map<String, ? extends PricingModel> pricingBySKU) {
        BigDecimal result = DEFAULT_OWNING_COST;

        Optional<Segment> segmentO = getSegmentFromChange(change);
        if (segmentO.isPresent()) {
            Segment segment = segmentO.get();

            if (CPU_SKU_BY_SEGMENT.containsKey(segment)) {
                String SKU = CPU_SKU_BY_SEGMENT.get(segment);

                if (pricingBySKU.containsKey(SKU)) {
                    PricingModel pricingModel = pricingBySKU.get(SKU);
                    result = convert(change, DiUnit.CORES, LOG)
                            .multiply(pricingModel.getPrice(), MATH_CONTEXT);
                }
            }
        }

        return result;
    }

    private Optional<Segment> getSegmentFromChange(QuotaChangeRequest.Change change) {
        return getDBSegment(change)
                .flatMap(segment -> Segment.byKey(segment.getPublicKey()));
    }

    private Optional<ru.yandex.qe.dispenser.domain.Segment> getDBSegment(final QuotaChangeRequest.Change change) {
        return SegmentUtils.getSegmentBySegmentationKey(change.getSegments(), distbuildSegmentationKey);
    }

    private enum Resource implements EnumUtils.StringKey {
        CPU("cpu"),
        ;

        private static Map<String, Resource> resourceByKey;
        private final String key;

        Resource(String key) {
            this.key = key;
        }

        @Override
        public String getKey() {
            return key;
        }

        public static Optional<Resource> byKey(String key) {
            if (resourceByKey == null) {
                resourceByKey = ImmutableMap.copyOf(EnumUtils.prepareKeysMap(Resource.values()));
            }

            return Optional.ofNullable(resourceByKey.get(key));
        }
    }

    private enum Segment implements EnumUtils.StringKey {
        DISTBUILD_AUTOCHECK("distbuild_autocheck"),
        DISTBUILD_USER("distbuild_user"),
        ;

        private static Map<String, Segment> segmentByKey;
        private final String key;

        Segment(String key) {
            this.key = key;
        }

        @Override
        public String getKey() {
            return key;
        }

        public static Optional<Segment> byKey(String key) {
            if (segmentByKey == null) {
                segmentByKey = ImmutableMap.copyOf(EnumUtils.prepareKeysMap(Segment.values()));
            }

            return Optional.ofNullable(segmentByKey.get(key));
        }
    }
}
