package ru.yandex.qe.dispenser.ws.quota.request.owning_cost.formula;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableMap;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import ru.yandex.qe.dispenser.api.util.EnumUtils;
import ru.yandex.qe.dispenser.api.v1.DiUnit;
import ru.yandex.qe.dispenser.domain.QuotaChangeRequest;
import ru.yandex.qe.dispenser.domain.Segment;
import ru.yandex.qe.dispenser.domain.dao.segment.SegmentUtils;
import ru.yandex.qe.dispenser.ws.bot.Provider;
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.ChangeOwningCostContext;
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.pricing.PricingModel;
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.pricing.QuotaChangeOwningCostTariffManager;

@Component
public class LogbrokerOwningCostFormula implements ProviderOwningCostFormula {

    private static final Logger LOG = LoggerFactory.getLogger(LogbrokerOwningCostFormula.class);

    private static final Provider PROVIDER = Provider.LOGBROKER;
    private static final Set<String> SKIPPED_SEGMENTS = Set.of("logbroker-yc", "logbroker-yc-preprod");
    private static final Set<String> LBKX_SEGMENTS = Set.of("lbkx");
    private static final String CPU_SKU = "yp.cpu.quota";
    private static final String RAM_SKU = "yp.memory.quota";
    private static final String HDD_SKU = "yp.hdd_storage.quota";
    private static final String SSD_SKU = "yp.ssd_storage.quota";
    private static final BigDecimal BYTES_PER_MB = BigDecimal.valueOf(1_000_000L);
    private static final BigDecimal BYTES_PER_MIB = BigDecimal.valueOf(1024L).pow(2);
    private static final BigDecimal BYTES_PER_TB = BigDecimal.valueOf(1_000_000_000_000L);
    private static final BigDecimal BYTES_PER_GIB = BigDecimal.valueOf(1024L).pow(3);
    private static final BigDecimal BYTES_PER_GB = BigDecimal.valueOf(1_000_000_000L);
    private static final BigDecimal CORES_PER_HOST = BigDecimal.valueOf(128L);
    private static final BigDecimal RAM_GIB_PER_HOST = BigDecimal.valueOf(1024L);
    private static final BigDecimal SSD_TB_PER_HOST = new BigDecimal("38.4", MATH_CONTEXT);
    private static final BigDecimal HDD_TB_PER_HOST = BigDecimal.valueOf(8L);
    private static final BigDecimal HOSTS_PER_GBPS = BigDecimal.valueOf(5L);
    private static final BigDecimal HOSTS_PER_GBPS_LBKX = BigDecimal.valueOf(16L);

    private final String logbrokerDbSegmentationKey;
    private final QuotaChangeOwningCostTariffManager quotaChangeOwningCostTariffManager;

    public LogbrokerOwningCostFormula(
            @Value("${dispenser.logbroker.segmentation.key}") final String logbrokerDbSegmentationKey,
            QuotaChangeOwningCostTariffManager quotaChangeOwningCostTariffManager) {
        this.logbrokerDbSegmentationKey = logbrokerDbSegmentationKey;
        this.quotaChangeOwningCostTariffManager = quotaChangeOwningCostTariffManager;
    }

    @Override
    public @NotNull String getProviderKey() {
        return PROVIDER.getServiceKey();
    }

    @Override
    public @NotNull Map<QuotaChangeRequest.ChangeKey, BigDecimal> calculateOwningCostFromContext(
            @NotNull Collection<ChangeOwningCostContext> changes) {
        Map<QuotaChangeRequest.ChangeKey, BigDecimal> result = new HashMap<>();
        Map<CampaignKey, List<ChangeOwningCostContext>> changesByCampaign = ProviderOwningCostFormula
                .groupByCampaign(changes, result);
        changesByCampaign.forEach((campaignKey, campaignChanges) -> {
            if (QuotaChangeOwningCostTariffManager.CAMPAIGN_KEYS_2022.contains(campaignKey.getKey())) {
                Map<QuotaChangeRequest.ChangeKey, BigDecimal> campaignCosts = calculate2022Costs(campaignKey, campaignChanges);
                result.putAll(campaignCosts);
            } else {
                Map<String, ? extends PricingModel> pricingBySKU =
                        quotaChangeOwningCostTariffManager.getByProviderCampaign(Provider.YP, campaignKey.getKey()).stream()
                                .collect(Collectors.toMap(PricingModel::getSKU, Function.identity()));
                Optional<BigDecimal> flowPrice = getFlowPrice(pricingBySKU);
                Optional<BigDecimal> lbkxFlowPrice = getLbkxFlowPrice(pricingBySKU);
                Map<QuotaChangeRequest.ChangeKey, BigDecimal> campaignCosts = campaignChanges.stream()
                        .map(ChangeOwningCostContext::getChange)
                        .collect(Collectors.toMap(QuotaChangeRequest.ChangeAmount::getKey,
                                change -> toOwningCost(change, flowPrice.orElse(null), lbkxFlowPrice.orElse(null))));
                result.putAll(campaignCosts);
            }
        });
        return result;
    }

    private Map<QuotaChangeRequest.ChangeKey, BigDecimal> calculate2022Costs(
            CampaignKey campaignKey, List<ChangeOwningCostContext> campaignChanges) {
        Map<String, ? extends PricingModel> pricingBySKU =
                quotaChangeOwningCostTariffManager.getByProviderCampaign(Provider.YP, campaignKey.getKey()).stream()
                        .collect(Collectors.toMap(PricingModel::getSKU, Function.identity()));
        Optional<BigDecimal> flowPrice = getFlowPrice(pricingBySKU);
        Optional<BigDecimal> lbkxFlowPrice = getLbkxFlowPrice(pricingBySKU);
        return campaignChanges.stream()
                .map(ChangeOwningCostContext::getChange)
                .collect(Collectors.toMap(QuotaChangeRequest.ChangeAmount::getKey,
                        change -> toOwningCost2022(change, flowPrice.orElse(null), lbkxFlowPrice.orElse(null))));
    }

    private BigDecimal toOwningCost2022(QuotaChangeRequest.Change change,
                                        BigDecimal flowPrice, BigDecimal lbkxFlowPrice) {
        if (flowPrice == null || lbkxFlowPrice == null) {
            return DEFAULT_OWNING_COST;
        }
        BigDecimal result = DEFAULT_OWNING_COST;
        Optional<LogbrokerSegment> segmentO = getSegmentFromChange(change);
        if (segmentO.isPresent()) {
            LogbrokerSegment segment = segmentO.get();
            if (!SKIPPED_SEGMENTS.contains(segment.getKey())) {
                BigDecimal selectedFlowPrice = LBKX_SEGMENTS.contains(segment.getKey()) ? lbkxFlowPrice : flowPrice;
                String publicKey = change.getResource().getPublicKey();
                Optional<Resource> resourcesO = Resource.byKey(publicKey);
                if (resourcesO.isPresent()) {
                    if (resourcesO.get().equals(Resource.DATA_FLOW_BINARY)) {
                        result = calculateForDataFlowBinary(change, selectedFlowPrice);
                    }
                }
            }
        }
        return result;
    }

    private BigDecimal toOwningCost(QuotaChangeRequest.Change change,
                                    BigDecimal flowPrice, BigDecimal lbkxFlowPrice) {
        if (flowPrice == null || lbkxFlowPrice == null) {
            return DEFAULT_OWNING_COST;
        }
        BigDecimal result = DEFAULT_OWNING_COST;
        Optional<LogbrokerSegment> segmentO = getSegmentFromChange(change);
        if (segmentO.isPresent()) {
            LogbrokerSegment segment = segmentO.get();
            if (!SKIPPED_SEGMENTS.contains(segment.getKey())) {
                BigDecimal selectedFlowPrice = LBKX_SEGMENTS.contains(segment.getKey()) ? lbkxFlowPrice : flowPrice;
                String publicKey = change.getResource().getPublicKey();
                Optional<Resource> resourcesO = Resource.byKey(publicKey);
                if (resourcesO.isPresent()) {
                    switch (resourcesO.get()) {
                        case DATA_FLOW_BINARY:
                            result = calculateForDataFlowBinary(change, selectedFlowPrice);
                            break;
                        case DATA_FLOW:
                            result = calculateForDataFlow(change, selectedFlowPrice);
                            break;
                    }
                }
            }
        }
        return result;
    }

    private Optional<BigDecimal> getFlowPrice(Map<String, ? extends PricingModel> pricingModelsBySKU) {
        Optional<BigDecimal> hostPriceO = getHostPrice(pricingModelsBySKU);
        if (hostPriceO.isEmpty()) {
            return hostPriceO;
        }
        BigDecimal pricePerMibps = HOSTS_PER_GBPS.multiply(hostPriceO.get(), MATH_CONTEXT)
                .multiply(BYTES_PER_MIB, MATH_CONTEXT).divide(BYTES_PER_GB, MATH_CONTEXT);
        return Optional.of(pricePerMibps);
    }

    private Optional<BigDecimal> getLbkxFlowPrice(Map<String, ? extends PricingModel> pricingModelsBySKU) {
        Optional<BigDecimal> hostPriceO = getHostPrice(pricingModelsBySKU);
        if (hostPriceO.isEmpty()) {
            return hostPriceO;
        }
        BigDecimal pricePerMibps = HOSTS_PER_GBPS_LBKX.multiply(hostPriceO.get(), MATH_CONTEXT)
                .multiply(BYTES_PER_MIB, MATH_CONTEXT).divide(BYTES_PER_GB, MATH_CONTEXT);
        return Optional.of(pricePerMibps);
    }

    private Optional<BigDecimal> getHostPrice(Map<String, ? extends PricingModel> pricingModelsBySKU) {
        if (pricingModelsBySKU.containsKey(CPU_SKU) && pricingModelsBySKU.containsKey(RAM_SKU)
                && pricingModelsBySKU.containsKey(HDD_SKU) && pricingModelsBySKU.containsKey(SSD_SKU)) {
            BigDecimal cpuPricePerCoreMonth = pricingModelsBySKU.get(CPU_SKU).getPrice();
            BigDecimal ramPricePerGibMonth = pricingModelsBySKU.get(RAM_SKU).getPrice();
            BigDecimal hddPricePerGibMonth = pricingModelsBySKU.get(HDD_SKU).getPrice();
            BigDecimal ssdPricePerGibMonth = pricingModelsBySKU.get(SSD_SKU).getPrice();
            BigDecimal cpuCostPerMonth = CORES_PER_HOST.multiply(cpuPricePerCoreMonth, MATH_CONTEXT);
            BigDecimal ramCostPerMonth = RAM_GIB_PER_HOST.multiply(ramPricePerGibMonth, MATH_CONTEXT);
            BigDecimal ssdCostPerMonth = SSD_TB_PER_HOST.multiply(ssdPricePerGibMonth, MATH_CONTEXT)
                    .multiply(BYTES_PER_TB, MATH_CONTEXT).divide(BYTES_PER_GIB, MATH_CONTEXT);
            BigDecimal hddCostPerMonth = HDD_TB_PER_HOST.multiply(hddPricePerGibMonth, MATH_CONTEXT)
                    .multiply(BYTES_PER_TB, MATH_CONTEXT).divide(BYTES_PER_GIB, MATH_CONTEXT);
            BigDecimal result = cpuCostPerMonth.add(ramCostPerMonth, MATH_CONTEXT)
                    .add(ssdCostPerMonth, MATH_CONTEXT).add(hddCostPerMonth, MATH_CONTEXT);
            return Optional.of(result);
        }
        return Optional.empty();
    }
    private Optional<LogbrokerSegment> getSegmentFromChange(QuotaChangeRequest.Change change) {
        return getClusterSegment(change)
                .flatMap(segment -> LogbrokerSegment.byKey(segment.getPublicKey()));

    }

    private Optional<Segment> getClusterSegment(final QuotaChangeRequest.Change change) {
        return SegmentUtils.getSegmentBySegmentationKey(change.getSegments(), logbrokerDbSegmentationKey);
    }

    private BigDecimal calculateForDataFlowBinary(QuotaChangeRequest.Change change,
                                                  BigDecimal flowPrice) {
        return calculateForDataFlowMibps(convert(change, DiUnit.MIBPS, LOG), flowPrice);
    }

    private BigDecimal calculateForDataFlow(QuotaChangeRequest.Change change,
                                            BigDecimal flowPrice) {
        BigDecimal mbps = convert(change, DiUnit.MBPS, LOG);
        BigDecimal mibps = mbps.multiply(BYTES_PER_MB, MATH_CONTEXT).divide(BYTES_PER_MIB, MATH_CONTEXT);
        return calculateForDataFlowMibps(mibps, flowPrice);
    }

    private BigDecimal calculateForDataFlowMibps(BigDecimal dataFlowMibps,
                                                 BigDecimal flowPrice) {
        return dataFlowMibps.multiply(flowPrice, MATH_CONTEXT);
    }

    private enum Resource implements EnumUtils.StringKey {
        DATA_FLOW("data_flow"),
        DATA_FLOW_BINARY("data_flow_binary"),
        ;

        private static Map<String, Resource> resourceByKey;
        private final String key;

        Resource(String key) {
            this.key = key;
        }

        public String getKey() {
            return key;
        }

        public static Optional<Resource> byKey(String key) {
            if (Resource.resourceByKey == null) {
                Resource.resourceByKey = ImmutableMap.copyOf(EnumUtils.prepareKeysMap(Resource.values()));
            }

            return Optional.ofNullable(Resource.resourceByKey.get(key));
        }

    }

    private enum LogbrokerSegment implements EnumUtils.StringKey {
        LBKX("lbkx"),
        LOGBROKER_SAS("logbroker_SAS"),
        LOGBROKER_VLA("logbroker_VLA"),
        LOGBROKER_MAN("logbroker_MAN"),
        LOGBROKER_MYT("logbroker_MYT"),
        LOGBROKER_IVA("logbroker_IVA"),
        LOGBROKER_PRESTABLE_SAS("logbroker-prestable_SAS"),
        LOGBROKER_PRESTABLE_VLA("logbroker-prestable_VLA"),
        LOGBROKER_PRESTABLE_MAN("logbroker-prestable_MAN"),
        LOGBROKER_PRESTABLE_MYT("logbroker-prestable_MYT"),
        LOGBROKER_PRESTABLE_IVA("logbroker-prestable_IVA"),
        LOGBROKER_YC("logbroker-yc"),
        LOGBROKER_YC_PREPROD("logbroker-yc-preprod"),
        LOGBROKER_KLG("logbroker_KLG"),
        LOGBROKER_PRESTABLE_KLG("logbroker-prestable_KLG"),
        ;

        private static Map<String, LogbrokerSegment> segmentByKey;
        private final String key;

        LogbrokerSegment(String key) {
            this.key = key;
        }

        @Override
        public String getKey() {
            return key;
        }

        public static Optional<LogbrokerSegment> byKey(String key) {
            if (segmentByKey == null) {
                segmentByKey = ImmutableMap.copyOf(EnumUtils.prepareKeysMap(LogbrokerSegment.values()));
            }
            return Optional.ofNullable(segmentByKey.get(key));
        }

    }

}
