package ru.yandex.qe.dispenser.ws.quota.request.owning_cost.formula;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableMap;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import ru.yandex.qe.dispenser.api.util.EnumUtils;
import ru.yandex.qe.dispenser.domain.QuotaChangeRequest;
import ru.yandex.qe.dispenser.ws.bot.Provider;
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.ChangeOwningCostContext;
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.pricing.PricingModel;
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.pricing.QuotaChangeOwningCostTariffManager;

/**
 * Logfeller owning cost formula (DISPENSER-4434).
 */
@Component
public class LogfellerOwningCostFormula implements ProviderOwningCostFormula {
    public static final Provider PROVIDER = Provider.LOGFELLER;
    private final String ytClusterSegmentationKey;
    private final QuotaChangeOwningCostTariffManager quotaChangeOwningCostTariffManager;

    public LogfellerOwningCostFormula(@Value("${dispenser.yt.segmentation.key}") final String ytClusterSegmentationKey,
                                      QuotaChangeOwningCostTariffManager quotaChangeOwningCostTariffManager) {
        this.ytClusterSegmentationKey = ytClusterSegmentationKey;
        this.quotaChangeOwningCostTariffManager = quotaChangeOwningCostTariffManager;
    }

    @Override
    public @NotNull String getProviderKey() {
        return PROVIDER.getServiceKey();
    }

    @Override
    public @NotNull Map<QuotaChangeRequest.ChangeKey, BigDecimal> calculateOwningCostFromContext(
            @NotNull Collection<ChangeOwningCostContext> changes) {
        Map<QuotaChangeRequest.ChangeKey, BigDecimal> result = new HashMap<>();
        Map<CampaignKey, List<ChangeOwningCostContext>> changesByCampaign = ProviderOwningCostFormula
                .groupByCampaign(changes, result);
        changesByCampaign.forEach((campaignKey, campaignChanges) -> {
            Map<String, ? extends PricingModel> pricingBySKU =
                    quotaChangeOwningCostTariffManager.getByProviderCampaign(Provider.YT, campaignKey.getKey()).stream()
                            .collect(Collectors.toMap(PricingModel::getSKU, Function.identity()));
            Map<QuotaChangeRequest.ChangeKey, BigDecimal> campaignCosts = campaignChanges.stream()
                    .collect(Collectors.toMap(change -> change.getChange().getKey(),
                            change -> toOwningCost(change, pricingBySKU)));
            result.putAll(campaignCosts);
        });
        return result;
    }

    private BigDecimal toOwningCost(ChangeOwningCostContext change,
                                    Map<String, ? extends PricingModel> pricingBySKU) {
        BigDecimal result = DEFAULT_OWNING_COST;

        Optional<CampaignKey> campaignKeyO = CampaignKey.byKey(change.getCampaign().getKey());

        if (campaignKeyO.isPresent()) {
            switch (campaignKeyO.get()) {
                case AUG2020:
                case FEB2021:
                    result = toOwningCostOld(change.getChange(), pricingBySKU);
                    break;
                case AUG2021:
                case AUG_2022_DRAFT:
                case AUG_2022_AGGREGATED:
                    // No Logfeller in 2022 campaign
                    break;
            }
        }

        return result;
    }

    private BigDecimal toOwningCostOld(QuotaChangeRequest.Change change,
                                       Map<String, ? extends PricingModel> pricingBySKU) {
        BigDecimal result = DEFAULT_OWNING_COST;

        if (pricingBySKU.isEmpty()) {
            return result;
        }

        Optional<YtOwningCostFormula.Cluster> clusterO = YtOwningCostFormula.getClusterFromChange(change, ytClusterSegmentationKey);
        if (clusterO.isPresent()) {
            YtOwningCostFormula.Cluster cluster = clusterO.get();

            String publicKey = change.getResource().getPublicKey();
            Optional<Resource> resourcesO = Resource.byKey(publicKey);
            if (resourcesO.isPresent()) {
                switch (resourcesO.get()) {
                    case CPU:
                        result = YtOwningCostFormula.calculateForStrongGuaranteeCpu(change, cluster, pricingBySKU);
                        break;
                    case HDD:
                        result = YtOwningCostFormula.calculateForHDD(change, cluster, pricingBySKU);
                        break;
                    case SSD:
                        result = YtOwningCostFormula.calculateForSSD(change, cluster, pricingBySKU);
                        break;
                }
            }
        }

        return result;
    }

    private enum Resource implements EnumUtils.StringKey {
        CPU("cpu"),
        HDD("hdd"),
        SSD("ssd"),
        ;

        private static Map<String, Resource> resourceByKey;
        private final String key;

        Resource(String key) {
            this.key = key;
        }

        @Override
        public String getKey() {
            return key;
        }

        public static Optional<Resource> byKey(String key) {
            if (Resource.resourceByKey == null) {
                Resource.resourceByKey = ImmutableMap.copyOf(EnumUtils.prepareKeysMap(Resource.values()));
            }

            return Optional.ofNullable(Resource.resourceByKey.get(key));
        }
    }
}
