package ru.yandex.qe.dispenser.ws.quota.request.owning_cost.formula;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableMap;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import ru.yandex.qe.dispenser.api.util.EnumUtils;
import ru.yandex.qe.dispenser.api.v1.DiUnit;
import ru.yandex.qe.dispenser.domain.QuotaChangeRequest;
import ru.yandex.qe.dispenser.domain.Segment;
import ru.yandex.qe.dispenser.domain.dao.segment.SegmentUtils;
import ru.yandex.qe.dispenser.ws.bot.Provider;
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.ChangeOwningCostContext;
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.pricing.PricingModel;
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.pricing.QuotaChangeOwningCostTariffManager;

/**
 * MDB owning cost formula (DISPENSER-4381).
 *
 * @author Ruslan Kadriev <aqru@yandex-team.ru>
 */
@Component
public class MDBOwningCostFormula implements ProviderOwningCostFormula {
    private static final Logger LOG = LoggerFactory.getLogger(MDBOwningCostFormula.class);

    private static final Provider provider = Provider.MDB;
    private static final BigDecimal FORMULA_COEFFICIENT = new BigDecimal("720", MATH_CONTEXT); // (DISPENSER-4381)
    public static final BigDecimal S3_PRICE = new BigDecimal("0.1047", MATH_CONTEXT); // (DISPENSER-4381)
    private static final String DATABASE_PLACEHOLDER = "{database}";
    private static final String CPU_100_TEMPLATE = "mdb.cluster.{database}.v3.cpu.c100";
    private static final String RAM_TEMPLATE = "mdb.cluster.{database}.v3.ram";
    private static final String SSD_TEMPLATE = "mdb.cluster.local-ssd.{database}";
    private static final BigDecimal CPU_COST_MULTIPLIER_2022 = new BigDecimal("1.3", MATH_CONTEXT);
    private static final BigDecimal RAM_COST_MULTIPLIER_2022 = new BigDecimal("1.3", MATH_CONTEXT);
    private static final BigDecimal HDD_COST_MULTIPLIER_2022 = new BigDecimal("2.0", MATH_CONTEXT);
    private static final BigDecimal SSD_COST_MULTIPLIER_2022 = new BigDecimal("2.0", MATH_CONTEXT);
    private static final BigDecimal HOURS_PER_MONTH = BigDecimal.valueOf(720L);
    private static final String HDD_X3_SKU = "s3.quota_space.hdd.x3";
    private static final String YP_CPU_SKU = "yp.cpu.quota";
    private static final String YP_RAM_SKU = "yp.memory.quota";
    private static final String YP_HDD_SKU = "yp.hdd_storage.quota";
    private static final String YP_SSD_SKU = "yp.ssd_storage.quota";

    public static final String MDB_CLUSTER_CLICKHOUSE_V_3_CPU_C_100 = "mdb.cluster.clickhouse.v3.cpu.c100";
    public static final String MDB_CLUSTER_CLICKHOUSE_V_3_RAM = "mdb.cluster.clickhouse.v3.ram";
    public static final String MDB_CLUSTER_LOCAL_SSD_CH = "mdb.cluster.local-ssd.ch";
    public static final String MDB_CLUSTER_LOCAL_HDD_CH = "mdb.cluster.local-hdd.ch";
    /**
     * SKU's by dataBase for MDB resources
     */
    private static final Map<DataBase, String> CPU_SKU_BY_DATABASE = new EnumMap<>(Map.of(
            DataBase.DBAAS_KAFKA, DataBase.DBAAS_KAFKA.toDataBaseSKU(CPU_100_TEMPLATE),
            DataBase.DBAAS_ELASTICSEARCH, DataBase.DBAAS_ELASTICSEARCH.toDataBaseSKU(CPU_100_TEMPLATE),
            DataBase.DBAAS_MONGODB, DataBase.DBAAS_MONGODB.toDataBaseSKU(CPU_100_TEMPLATE),
            DataBase.DBAAS_MYSQL, DataBase.DBAAS_MYSQL.toDataBaseSKU(CPU_100_TEMPLATE),
            DataBase.DBAAS_PGSQL, DataBase.DBAAS_PGSQL.toDataBaseSKU(CPU_100_TEMPLATE),
            DataBase.DBAAS_REDIS, DataBase.DBAAS_REDIS.toDataBaseSKU(CPU_100_TEMPLATE),

            DataBase.DBAAS_CLICKHOUSE, MDB_CLUSTER_CLICKHOUSE_V_3_CPU_C_100,
            DataBase.DBAAS_GREENPLUM, DataBase.DBAAS_PGSQL.toDataBaseSKU(CPU_100_TEMPLATE) // (DISPENSER-4381)
    ));

    private static final Map<DataBase, String> RAM_SKU_BY_DATABASE = new EnumMap<>(Map.of(
            DataBase.DBAAS_KAFKA, DataBase.DBAAS_KAFKA.toDataBaseSKU(RAM_TEMPLATE),
            DataBase.DBAAS_ELASTICSEARCH, DataBase.DBAAS_ELASTICSEARCH.toDataBaseSKU(RAM_TEMPLATE),
            DataBase.DBAAS_MONGODB, DataBase.DBAAS_MONGODB.toDataBaseSKU(RAM_TEMPLATE),
            DataBase.DBAAS_MYSQL, DataBase.DBAAS_MYSQL.toDataBaseSKU(RAM_TEMPLATE),
            DataBase.DBAAS_PGSQL, DataBase.DBAAS_PGSQL.toDataBaseSKU(RAM_TEMPLATE),
            DataBase.DBAAS_REDIS, DataBase.DBAAS_REDIS.toDataBaseSKU(RAM_TEMPLATE),

            DataBase.DBAAS_CLICKHOUSE, MDB_CLUSTER_CLICKHOUSE_V_3_RAM,
            DataBase.DBAAS_GREENPLUM, DataBase.DBAAS_PGSQL.toDataBaseSKU(RAM_TEMPLATE) // (DISPENSER-4381)
    ));

    private static final Map<DataBase, String> SSD_SKU_BY_DATABASE = new EnumMap<>(Map.of(
            DataBase.DBAAS_KAFKA, DataBase.DBAAS_KAFKA.toDataBaseSKU(SSD_TEMPLATE),
            DataBase.DBAAS_ELASTICSEARCH, DataBase.DBAAS_ELASTICSEARCH.toDataBaseSKU(SSD_TEMPLATE),
            DataBase.DBAAS_MONGODB, DataBase.DBAAS_MONGODB.toDataBaseSKU(SSD_TEMPLATE),
            DataBase.DBAAS_MYSQL, DataBase.DBAAS_MYSQL.toDataBaseSKU(SSD_TEMPLATE),
            DataBase.DBAAS_PGSQL, DataBase.DBAAS_PGSQL.toDataBaseSKU(SSD_TEMPLATE),
            DataBase.DBAAS_REDIS, DataBase.DBAAS_REDIS.toDataBaseSKU(SSD_TEMPLATE),

            DataBase.DBAAS_CLICKHOUSE, MDB_CLUSTER_LOCAL_SSD_CH,
            DataBase.DBAAS_GREENPLUM, DataBase.DBAAS_PGSQL.toDataBaseSKU(SSD_TEMPLATE) // (DISPENSER-4381)
    ));

    private static final Map<DataBase, String> HDD_SKU_BY_DATABASE = new EnumMap<>(Map.of(
            DataBase.DBAAS_CLICKHOUSE, MDB_CLUSTER_LOCAL_HDD_CH
    ));

    private final String mdbDbSegmentationKey;
    private final QuotaChangeOwningCostTariffManager quotaChangeOwningCostTariffManager;

    public MDBOwningCostFormula(@Value("${dispenser.mdb.segmentation.key}") final String mdbDbSegmentationKey,
                                QuotaChangeOwningCostTariffManager quotaChangeOwningCostTariffManager) {
        this.mdbDbSegmentationKey = mdbDbSegmentationKey;
        this.quotaChangeOwningCostTariffManager = quotaChangeOwningCostTariffManager;
    }

    @Override
    public @NotNull String getProviderKey() {
        return provider.getServiceKey();
    }

    public @NotNull Map<QuotaChangeRequest.ChangeKey, BigDecimal> calculateOwningCostFromContext(
            @NotNull Collection<ChangeOwningCostContext> changes) {
        Map<QuotaChangeRequest.ChangeKey, BigDecimal> result = new HashMap<>();
        Map<CampaignKey, List<ChangeOwningCostContext>> changesByCampaign = ProviderOwningCostFormula
                .groupByCampaign(changes, result);
        changesByCampaign.forEach((campaignKey, campaignChanges) -> {
            if (QuotaChangeOwningCostTariffManager.CAMPAIGN_KEYS_2022.contains(campaignKey.getKey())) {
                Map<QuotaChangeRequest.ChangeKey, BigDecimal> campaignCosts = calculate2022Costs(campaignKey, campaignChanges);
                result.putAll(campaignCosts);
            } else {
                Map<String, ? extends PricingModel> pricingBySKU =
                        quotaChangeOwningCostTariffManager.getByProviderCampaign(provider, campaignKey.getKey()).stream()
                                .collect(Collectors.toMap(PricingModel::getSKU, Function.identity()));
                Map<QuotaChangeRequest.ChangeKey, BigDecimal> campaignCosts = campaignChanges.stream()
                        .map(ChangeOwningCostContext::getChange)
                        .collect(Collectors.toMap(QuotaChangeRequest.ChangeAmount::getKey,
                        change -> toOwningCost(change, pricingBySKU)));
                result.putAll(campaignCosts);
            }
        });
        return result;
    }

    private Map<QuotaChangeRequest.ChangeKey, BigDecimal> calculate2022Costs(
            CampaignKey campaignKey, List<ChangeOwningCostContext> campaignChanges) {
        Map<String, ? extends PricingModel> pricingBySKU =
                quotaChangeOwningCostTariffManager.getByProvidersCampaign(List.of(Provider.YP, Provider.S3),
                                campaignKey.getKey())
                        .stream().collect(Collectors.toMap(PricingModel::getSKU, Function.identity()));
        return campaignChanges.stream()
                .map(ChangeOwningCostContext::getChange)
                .collect(Collectors.toMap(QuotaChangeRequest.ChangeAmount::getKey,
                        change -> toOwningCost2022(change, pricingBySKU)));
    }

    private BigDecimal toOwningCost2022(QuotaChangeRequest.Change change,
                                        Map<String, ? extends PricingModel> pricingBySKU) {
        BigDecimal result = DEFAULT_OWNING_COST;
        if (pricingBySKU.isEmpty()) {
            return result;
        }
        String publicKey = change.getResource().getPublicKey();
        Optional<Resource> resourcesO = Resource.byKey(publicKey);
        if (resourcesO.isPresent()) {
            switch (resourcesO.get()) {
                case CPU_ALL -> result = calculateForCpu2022(change, pricingBySKU);
                case RAM_ALL -> result = calculateForRam2022(change, pricingBySKU);
                case HDD_ALL -> result = calculateForHdd2022(change, pricingBySKU);
                case SSD_ALL -> result = calculateForSsd2022(change, pricingBySKU);
                case S3_HDD_X3 -> result = calculateForS3X3Hdd2022(change, pricingBySKU);
            }
        }
        return result;
    }

    private BigDecimal calculateForCpu2022(QuotaChangeRequest.Change change,
                                           Map<String, ? extends PricingModel> pricingBySKU) {
        PricingModel pricing = pricingBySKU.get(YP_CPU_SKU);
        if (pricing == null) {
            return ProviderOwningCostFormula.DEFAULT_OWNING_COST;
        }
        return convert(change, DiUnit.CORES, LOG)
                .multiply(CPU_COST_MULTIPLIER_2022, MATH_CONTEXT)
                .multiply(pricing.getPrice(), MATH_CONTEXT);
    }

    private BigDecimal calculateForRam2022(QuotaChangeRequest.Change change,
                                           Map<String, ? extends PricingModel> pricingBySKU) {
        PricingModel pricing = pricingBySKU.get(YP_RAM_SKU);
        if (pricing == null) {
            return ProviderOwningCostFormula.DEFAULT_OWNING_COST;
        }
        return convert(change, DiUnit.GIBIBYTE, LOG)
                .multiply(RAM_COST_MULTIPLIER_2022, MATH_CONTEXT)
                .multiply(pricing.getPrice(), MATH_CONTEXT);
    }

    private BigDecimal calculateForHdd2022(QuotaChangeRequest.Change change,
                                           Map<String, ? extends PricingModel> pricingBySKU) {
        PricingModel pricing = pricingBySKU.get(YP_HDD_SKU);
        if (pricing == null) {
            return ProviderOwningCostFormula.DEFAULT_OWNING_COST;
        }
        return convert(change, DiUnit.GIBIBYTE, LOG)
                .multiply(HDD_COST_MULTIPLIER_2022, MATH_CONTEXT)
                .multiply(pricing.getPrice(), MATH_CONTEXT);
    }

    private BigDecimal calculateForSsd2022(QuotaChangeRequest.Change change,
                                           Map<String, ? extends PricingModel> pricingBySKU) {
        PricingModel pricing = pricingBySKU.get(YP_SSD_SKU);
        if (pricing == null) {
            return ProviderOwningCostFormula.DEFAULT_OWNING_COST;
        }
        return convert(change, DiUnit.GIBIBYTE, LOG)
                .multiply(SSD_COST_MULTIPLIER_2022, MATH_CONTEXT)
                .multiply(pricing.getPrice(), MATH_CONTEXT);
    }

    private BigDecimal calculateForS3X3Hdd2022(QuotaChangeRequest.Change change,
                                               Map<String, ? extends PricingModel> pricingBySKU) {
        PricingModel pricing = pricingBySKU.get(HDD_X3_SKU);
        if (pricing == null) {
            return ProviderOwningCostFormula.DEFAULT_OWNING_COST;
        }
        return convert(change, DiUnit.GIBIBYTE, LOG)
                .multiply(HOURS_PER_MONTH, ProviderOwningCostFormula.MATH_CONTEXT)
                .multiply(pricing.getPrice(), ProviderOwningCostFormula.MATH_CONTEXT);
    }

    private BigDecimal toOwningCost(QuotaChangeRequest.Change change,
                                    Map<String, ? extends PricingModel> pricingBySKU) {
        BigDecimal result = DEFAULT_OWNING_COST;

        if (pricingBySKU.isEmpty()) {
            return result;
        }

        String publicKey = change.getResource().getPublicKey();
        Optional<Resource> resourcesO = Resource.byKey(publicKey);
        if (resourcesO.isPresent()) {
            switch (resourcesO.get()) {
                case SSD:
                    result = calculateForSSD(change, pricingBySKU);
                    break;
                case HDD:
                    result = calculateForHDD(change, pricingBySKU);
                    break;
                case CPU:
                    result = calculateForCPU(change, pricingBySKU);
                    break;
                case RAM:
                    result = calculateForRAM(change, pricingBySKU);
                    break;
                case STORAGE_S3:
                    result = calculateForS3(change);
                    break;
            }
        }

        return result;
    }

    private BigDecimal calculateForCPU(QuotaChangeRequest.Change change,
                                       Map<String, ? extends PricingModel> pricingBySKU) {
        BigDecimal result = DEFAULT_OWNING_COST;

        Optional<DataBase> dataBaseO = getDataBaseFromChange(change);
        if (dataBaseO.isPresent()) {
            DataBase dataBase = dataBaseO.get();

            if (CPU_SKU_BY_DATABASE.containsKey(dataBase)) {
                String SKU = CPU_SKU_BY_DATABASE.get(dataBase);

                if (pricingBySKU.containsKey(SKU)) {
                    PricingModel pricingModel = pricingBySKU.get(SKU);

                    result = convert(change, DiUnit.CORES, LOG)
                            .multiply(pricingModel.getPrice(), MATH_CONTEXT)
                            .multiply(FORMULA_COEFFICIENT, MATH_CONTEXT);
                }
            }
        }

        return result;
    }

    private BigDecimal calculateForRAM(QuotaChangeRequest.Change change,
                                       Map<String, ? extends PricingModel> pricingBySKU) {
        BigDecimal result = DEFAULT_OWNING_COST;

        Optional<DataBase> dataBaseO = getDataBaseFromChange(change);
        if (dataBaseO.isPresent()) {
            DataBase dataBase = dataBaseO.get();

            if (RAM_SKU_BY_DATABASE.containsKey(dataBase)) {
                String SKU = RAM_SKU_BY_DATABASE.get(dataBase);

                if (pricingBySKU.containsKey(SKU)) {
                    PricingModel pricingModel = pricingBySKU.get(SKU);

                    result = convert(change, DiUnit.GIBIBYTE, LOG)
                            .multiply(pricingModel.getPrice(), MATH_CONTEXT)
                            .multiply(FORMULA_COEFFICIENT, MATH_CONTEXT);
                }
            }
        }

        return result;
    }

    private BigDecimal calculateForSSD(QuotaChangeRequest.Change change,
                                       Map<String, ? extends PricingModel> pricingBySKU) {
        BigDecimal result = DEFAULT_OWNING_COST;

        Optional<DataBase> dataBaseO = getDataBaseFromChange(change);
        if (dataBaseO.isPresent()) {
            DataBase dataBase = dataBaseO.get();

            if (SSD_SKU_BY_DATABASE.containsKey(dataBase)) {
                String SKU = SSD_SKU_BY_DATABASE.get(dataBase);

                if (pricingBySKU.containsKey(SKU)) {
                    PricingModel pricingModel = pricingBySKU.get(SKU);

                    result = convert(change, DiUnit.GIBIBYTE, LOG)
                            .multiply(pricingModel.getPrice(), MATH_CONTEXT)
                            .multiply(FORMULA_COEFFICIENT, MATH_CONTEXT);
                }
            }
        }

        return result;
    }

    private BigDecimal calculateForHDD(QuotaChangeRequest.Change change,
                                       Map<String, ? extends PricingModel> pricingBySKU) {
        BigDecimal result = DEFAULT_OWNING_COST;

        Optional<DataBase> dataBaseO = getDataBaseFromChange(change);
        if (dataBaseO.isPresent()) {
            DataBase dataBase = dataBaseO.get();

            if (HDD_SKU_BY_DATABASE.containsKey(dataBase)) {
                String SKU = HDD_SKU_BY_DATABASE.get(dataBase);

                if (pricingBySKU.containsKey(SKU)) {
                    PricingModel pricingModel = pricingBySKU.get(SKU);

                    result = convert(change, DiUnit.GIBIBYTE, LOG)
                            .multiply(pricingModel.getPrice(), MATH_CONTEXT)
                            .multiply(FORMULA_COEFFICIENT, MATH_CONTEXT);
                }
            }
        }

        return result;
    }

    private BigDecimal calculateForS3(QuotaChangeRequest.Change change) {
        return convert(change, DiUnit.GIBIBYTE, LOG)
                .multiply(S3_PRICE, MATH_CONTEXT);
    }

    private Optional<DataBase> getDataBaseFromChange(QuotaChangeRequest.Change change) {
        return getDBSegment(change)
                .flatMap(segment -> DataBase.byKey(segment.getPublicKey()));

    }

    private enum DataBase implements EnumUtils.StringKey {
        DBAAS_KAFKA("dbaas_kafka", "kafka"),
        DBAAS_CLICKHOUSE("dbaas_clickhouse"),
        DBAAS_ELASTICSEARCH("dbaas_elasticsearch", "elasticsearch"),
        DBAAS_MONGODB("dbaas_mongodb", "mongodb"),
        DBAAS_MYSQL("dbaas_mysql", "mysql"),
        DBAAS_PGSQL("dbaas_pgsql", "pg"),
        DBAAS_REDIS("dbaas_redis", "redis"),
        DBAAS_GREENPLUM("dbaas_greenplum"),
        ;

        private static Map<String, DataBase> dbByKey;
        private final String key;
        private final String skuKey;

        DataBase(String key) {
            this.key = key;
            this.skuKey = null;
        }

        DataBase(String key, String skuKey) {
            this.key = key;
            this.skuKey = skuKey;
        }

        @Override
        public String getKey() {
            return key;
        }

        public static Optional<DataBase> byKey(String key) {
            if (dbByKey == null) {
                dbByKey = ImmutableMap.copyOf(EnumUtils.prepareKeysMap(DataBase.values()));
            }

            return Optional.ofNullable(dbByKey.get(key));
        }

        public String toDataBaseSKU(String SKU) {
            if (skuKey == null) {
                throw new IllegalStateException("skuKey must be provided!");
            }

            return SKU.replace(DATABASE_PLACEHOLDER, this.skuKey);
        }
    }

    private enum Resource implements EnumUtils.StringKey {
        SSD("ssd"),
        HDD("hdd"),
        CPU("cpu"),
        RAM("ram"),
        STORAGE_S3("storage-s3"),
        CPU_ALL("cpu_all"),
        RAM_ALL("ram_all"),
        HDD_ALL("hdd_all"),
        SSD_ALL("ssd_all"),
        S3_HDD_X3("s3_hdd_x3"),
        ;

        private static Map<String, Resource> resourceByKey;
        private final String key;

        Resource(String key) {
            this.key = key;
        }

        @Override
        public String getKey() {
            return key;
        }

        public static Optional<Resource> byKey(String key) {
            if (resourceByKey == null) {
                resourceByKey = ImmutableMap.copyOf(EnumUtils.prepareKeysMap(Resource.values()));
            }

            return Optional.ofNullable(resourceByKey.get(key));
        }
    }

    private Optional<Segment> getDBSegment(final QuotaChangeRequest.Change change) {
        return SegmentUtils.getSegmentBySegmentationKey(change.getSegments(), mdbDbSegmentationKey);
    }
}
