package ru.yandex.qe.dispenser.ws.quota.request.owning_cost.formula

import mu.KotlinLogging
import org.springframework.stereotype.Component
import ru.yandex.qe.dispenser.api.v1.DiUnit
import ru.yandex.qe.dispenser.domain.QuotaChangeRequest
import ru.yandex.qe.dispenser.ws.bot.Provider
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.ChangeOwningCostContext
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.formula.ProviderOwningCostFormula.CampaignKey
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.pricing.PricingModel
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.pricing.QuotaChangeOwningCostTariffManager
import java.math.BigDecimal

private val logger = KotlinLogging.logger {}

@Component
class MdsOwningCostFormula(
    private val tariffManager: QuotaChangeOwningCostTariffManager
): ProviderOwningCostFormula {

    private val provider: Provider = Provider.MDS_NEW
    private val hoursPerMonth: BigDecimal = BigDecimal.valueOf(720L)
    private val x2HddSku = "mds.quota_space.hdd.x2"
    private val x2SsdSku = "mds.quota_space.ssd.x2"
    private val x3HddSku = "mds.quota_space.hdd.x3"
    private val x3SsdSku = "mds.quota_space.ssd.x3"
    private val x15HddSku = "mds.quota_space.hdd.lrc"

    override fun getProviderKey(): String {
        return provider.serviceKey
    }

    override fun calculateOwningCostFromContext(
        changes: MutableCollection<ChangeOwningCostContext>
    ): MutableMap<QuotaChangeRequest.ChangeKey, BigDecimal> {
        val result = mutableMapOf<QuotaChangeRequest.ChangeKey, BigDecimal>()
        val changesByCampaign = ProviderOwningCostFormula.groupByCampaign(changes, result)
        changesByCampaign.forEach { (campaignKey, campaignChanges) ->
            val pricingBySKU = tariffManager.getByProviderCampaign(provider, campaignKey.key).associateBy { it.sku }
            if (pricingBySKU.isNotEmpty()) {
                result.putAll(campaignChanges.associateBy { it.change.key }.mapValues { toOwningCost(it.value, pricingBySKU) })
            }
        }
        return result
    }

    private fun toOwningCost(change: ChangeOwningCostContext, pricingBySKU: Map<String, PricingModel>): BigDecimal {
        return when (CampaignKey.byKey(change.campaign.key).orElse(null)) {
            CampaignKey.AUG_2022_DRAFT, CampaignKey.AUG_2022_AGGREGATED -> toOwningCost2022(change, pricingBySKU)
            else -> ProviderOwningCostFormula.DEFAULT_OWNING_COST
        }
    }

    private fun toOwningCost2022(change: ChangeOwningCostContext, pricingBySKU: Map<String, PricingModel>): BigDecimal {
        return when (Resource.byKey(change.change.resource.publicKey)) {
            Resource.REPLICAS_X2_HDD -> cost(x2HddSku, change, pricingBySKU)
            Resource.REPLICAS_X2_SSD -> cost(x2SsdSku, change, pricingBySKU)
            Resource.REPLICAS_X3_HDD -> cost(x3HddSku, change, pricingBySKU)
            Resource.REPLICAS_X3_SSD -> cost(x3SsdSku, change, pricingBySKU)
            Resource.LRC_X1_5_HDD -> cost(x15HddSku, change, pricingBySKU)
            else -> ProviderOwningCostFormula.DEFAULT_OWNING_COST
        }
    }

    private fun cost(sku: String, change: ChangeOwningCostContext, pricingBySKU: Map<String, PricingModel>): BigDecimal {
        val pricing = pricingBySKU[sku] ?: return ProviderOwningCostFormula.DEFAULT_OWNING_COST
        return convert(change.change, DiUnit.GIBIBYTE, logger)
            .multiply(hoursPerMonth, ProviderOwningCostFormula.MATH_CONTEXT)
            .multiply(pricing.price, ProviderOwningCostFormula.MATH_CONTEXT)
    }

    private enum class Resource(private val key: String) {
        REPLICAS_X2_HDD("replicas_x2_hdd"),
        REPLICAS_X2_SSD("replicas_x2_ssd"),
        REPLICAS_X3_HDD("replicas_x3_hdd"),
        REPLICAS_X3_SSD("replicas_x3_ssd"),
        LRC_X1_5_HDD("lrc_x1_5_hdd");

        companion object {
            private val resourceByKey: Map<String, Resource> = values().associateBy { it.key }

            fun byKey(key: String): Resource? {
                return resourceByKey[key]
            }
        }
    }

}
