package ru.yandex.qe.dispenser.ws.quota.request.owning_cost.formula;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableMap;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import ru.yandex.qe.dispenser.api.util.EnumUtils;
import ru.yandex.qe.dispenser.api.v1.DiUnit;
import ru.yandex.qe.dispenser.domain.QuotaChangeRequest;
import ru.yandex.qe.dispenser.domain.Segment;
import ru.yandex.qe.dispenser.domain.dao.segment.SegmentUtils;
import ru.yandex.qe.dispenser.ws.bot.Provider;
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.ChangeOwningCostContext;
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.pricing.PricingModel;
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.pricing.QuotaChangeOwningCostTariffManager;

/**
 * Nirvana owning cost formula (DISPENSER-4379).
 *
 * @author Ruslan Kadriev <aqru@yandex-team.ru>
 */
@Component
public class NirvanaOwningCostFormula implements ProviderOwningCostFormula {
    private static final Logger LOG = LoggerFactory.getLogger(NirvanaOwningCostFormula.class);

    private static final Provider provider = Provider.NIRVANA;

    private final BigDecimal S3_STORAGE_TARIFF_BY_GIB = new BigDecimal("0.1047", MATH_CONTEXT);

    private static final String GPU_STRONG_GUARANTEE_ARNOLD_SKU = "yt.arnold.gpu.tesla_a100_80g.strong_guarantee.gpu";
    private static final String GPU_USAGE_ARNOLD_SKU = "yt.arnold.gpu.tesla_a100_80g.usage.gpu";
    private static final String GPU_STRONG_GUARANTEE_HAHN_SKU = "yt.hahn.gpu.tesla_a100_80g.strong_guarantee.gpu";
    private static final String GPU_USAGE_HAHN_SKU = "yt.hahn.gpu.tesla_a100_80g.usage.gpu";
    private static final String CPU_STRONG_GUARANTEE_HAHN_SKU = "yt.hahn.compute.strong_guarantee.cpu";
    private static final String CPU_USAGE_HAHN_SKU = "yt.hahn.compute.usage.cpu";
    private static final String YT_HAHN_COMPUTE_USAGE_MEMORY = "yt.hahn.compute.usage.memory";
    private static final BigDecimal HOURS_PER_MONTH = BigDecimal.valueOf(720L);
    private static final String HDD_X2_SKU = "s3.quota_space.hdd.x2";

    /**
     * SKU's by location for Nirvana resources.
     */
    private static final Map<Location, List<String>> GPU_SKU_BY_LOCATION = new EnumMap<>(Map.of(
            Location.SAS, List.of(GPU_STRONG_GUARANTEE_HAHN_SKU, GPU_USAGE_HAHN_SKU),
            Location.VLA, List.of(GPU_STRONG_GUARANTEE_ARNOLD_SKU, GPU_USAGE_ARNOLD_SKU)
    ));
    private static final Map<Location, List<String>> CPU_SKU_BY_LOCATION = new EnumMap<>(Map.of(
            Location.SAS, List.of(CPU_STRONG_GUARANTEE_HAHN_SKU, CPU_USAGE_HAHN_SKU)
    ));
    private static final Map<Location, String> RAM_SKU_BY_LOCATION = new EnumMap<>(Map.of(
            Location.SAS, YT_HAHN_COMPUTE_USAGE_MEMORY
    ));

    private final QuotaChangeOwningCostTariffManager quotaChangeOwningCostTariffManager;
    private final String locationSegmentationKey;

    public NirvanaOwningCostFormula(QuotaChangeOwningCostTariffManager quotaChangeOwningCostTariffManager,
                                    @Value("${dispenser.location.segmentation.key}") final String locationSegmentationKey) {
        this.quotaChangeOwningCostTariffManager = quotaChangeOwningCostTariffManager;
        this.locationSegmentationKey = locationSegmentationKey;
    }

    @Override
    public @NotNull String getProviderKey() {
        return provider.getServiceKey();
    }

    public @NotNull Map<QuotaChangeRequest.ChangeKey, BigDecimal> calculateOwningCostFromContext(
            @NotNull Collection<ChangeOwningCostContext> changes) {
        Map<QuotaChangeRequest.ChangeKey, BigDecimal> result = new HashMap<>();
        Map<CampaignKey, List<ChangeOwningCostContext>> changesByCampaign = ProviderOwningCostFormula
                .groupByCampaign(changes, result);
        changesByCampaign.forEach((campaignKey, campaignChanges) -> {
            if (QuotaChangeOwningCostTariffManager.CAMPAIGN_KEYS_2022.contains(campaignKey.getKey())) {
                Map<String, ? extends PricingModel> pricingBySKU =
                        quotaChangeOwningCostTariffManager.getByProviderCampaign(Provider.S3, campaignKey.getKey()).stream()
                                .collect(Collectors.toMap(PricingModel::getSKU, Function.identity()));
                Map<QuotaChangeRequest.ChangeKey, BigDecimal> campaignCosts = campaignChanges.stream()
                        .map(ChangeOwningCostContext::getChange)
                        .collect(Collectors.toMap(QuotaChangeRequest.ChangeAmount::getKey,
                                change -> toOwningCost2022(change, pricingBySKU)));
                result.putAll(campaignCosts);
            } else {
                Map<String, ? extends PricingModel> pricingBySKU =
                        quotaChangeOwningCostTariffManager.getByProviderCampaign(Provider.YT, campaignKey.getKey()).stream()
                                .collect(Collectors.toMap(PricingModel::getSKU, Function.identity()));
                Map<QuotaChangeRequest.ChangeKey, BigDecimal> campaignCosts = campaignChanges.stream()
                        .map(ChangeOwningCostContext::getChange)
                        .collect(Collectors.toMap(QuotaChangeRequest.ChangeAmount::getKey,
                                change -> toOwningCost(change, pricingBySKU)));
                result.putAll(campaignCosts);
            }
        });
        return result;
    }

    private BigDecimal toOwningCost2022(QuotaChangeRequest.Change change,
                                        Map<String, ? extends PricingModel> pricingBySKU) {
        BigDecimal result = DEFAULT_OWNING_COST;
        if (pricingBySKU.isEmpty()) {
            return result;
        }
        String publicKey = change.getResource().getPublicKey();
        Optional<Resource> resourcesO = Resource.byKey(publicKey);
        if (resourcesO.isPresent()) {
            if (resourcesO.get() == Resource.S3_HDD_X2) {
                result = calculate2022ForS3HddX2(change, pricingBySKU);
            }
        }
        return result;
    }

    private BigDecimal calculate2022ForS3HddX2(QuotaChangeRequest.Change change,
                                               Map<String, ? extends PricingModel> pricingBySKU) {
        PricingModel pricing = pricingBySKU.get(HDD_X2_SKU);
        if (pricing == null) {
            return ProviderOwningCostFormula.DEFAULT_OWNING_COST;
        }
        return convert(change, DiUnit.GIBIBYTE, LOG)
                .multiply(HOURS_PER_MONTH, ProviderOwningCostFormula.MATH_CONTEXT)
                .multiply(pricing.getPrice(), ProviderOwningCostFormula.MATH_CONTEXT);
    }

    private BigDecimal toOwningCost(QuotaChangeRequest.Change change,
                                    Map<String, ? extends PricingModel> pricingBySKU) {
        BigDecimal result = DEFAULT_OWNING_COST;

        if (pricingBySKU.isEmpty()) {
            return result;
        }

        String publicKey = change.getResource().getPublicKey();
        Optional<Resource> resourcesO = Resource.byKey(publicKey);
        if (resourcesO.isPresent()) {
            switch (resourcesO.get()) {
                case CPU:
                    result = calculateForCpu(change, pricingBySKU);
                    break;
                case GPU:
                    result = calculateForGpu(change, pricingBySKU);
                    break;
                case S3_STORAGE:
                    result = calculateForS3Storage(change);
                    break;
                case RAM_GIG:
                    result = calculateForRam(change, pricingBySKU);
                    break;
            }
        }

        return result;
    }

    private BigDecimal calculateForGpu(QuotaChangeRequest.Change change,
                                       Map<String, ? extends PricingModel> prisingModelsBySKU) {
        BigDecimal result = DEFAULT_OWNING_COST;

        Optional<Location> locationO = getLocationFromChange(change);
        if (locationO.isPresent()) {
            Location location = locationO.get();

            if (GPU_SKU_BY_LOCATION.containsKey(location)) {
                List<String> SKUList = GPU_SKU_BY_LOCATION.get(location);

                if (SKUList.size() == 2 && prisingModelsBySKU.keySet().containsAll(SKUList)) {
                    PricingModel pricingModel1 = prisingModelsBySKU.get(SKUList.get(0));
                    PricingModel pricingModel2 = prisingModelsBySKU.get(SKUList.get(1));

                    result = convert(change, DiUnit.COUNT, LOG)
                            .multiply(pricingModel1.add(pricingModel2, MATH_CONTEXT), MATH_CONTEXT);
                }
            }
        }

        return result;
    }

    private BigDecimal calculateForCpu(QuotaChangeRequest.Change change,
                                       Map<String, ? extends PricingModel> prisingModelsBySKU) {
        BigDecimal result = DEFAULT_OWNING_COST;

        Optional<Location> locationO = getLocationFromChange(change);
        if (locationO.isPresent()) {
            Location location = locationO.get();

            if (CPU_SKU_BY_LOCATION.containsKey(location)) {
                List<String> SKUList = CPU_SKU_BY_LOCATION.get(location);

                if (SKUList.size() == 2 && prisingModelsBySKU.keySet().containsAll(SKUList)) {
                    PricingModel pricingModel1 = prisingModelsBySKU.get(SKUList.get(0));
                    PricingModel pricingModel2 = prisingModelsBySKU.get(SKUList.get(1));

                    result = convert(change, DiUnit.CORES, LOG)
                            .multiply(pricingModel1.add(pricingModel2, MATH_CONTEXT), MATH_CONTEXT);
                }
            }
        }

        return result;
    }

    private BigDecimal calculateForRam(QuotaChangeRequest.Change change,
                                       Map<String, ? extends PricingModel> prisingModelsBySKU) {
        BigDecimal result = DEFAULT_OWNING_COST;

        Optional<Location> locationO = getLocationFromChange(change);
        if (locationO.isPresent()) {
            Location location = locationO.get();

            if (RAM_SKU_BY_LOCATION.containsKey(location)) {
                String SKU = RAM_SKU_BY_LOCATION.get(location);

                if (prisingModelsBySKU.containsKey(SKU)) {
                    PricingModel pricingModel = prisingModelsBySKU.get(SKU);

                    result = convert(change, DiUnit.GIBIBYTE, LOG)
                            .multiply(pricingModel.getPrice(), MATH_CONTEXT);
                }
            }
        }

        return result;
    }

    private BigDecimal calculateForS3Storage(QuotaChangeRequest.Change change) {
        return convert(change, DiUnit.GIBIBYTE, LOG)
                .multiply(S3_STORAGE_TARIFF_BY_GIB, MATH_CONTEXT);
    }

    private enum Location implements EnumUtils.StringKey {
        VLA("VLA"),
        SAS("SAS"),
        MAN("MAN"),
        MYT("MYT"),
        IVA("IVA"),
        ;

        private static Map<String, Location> locationByKey;
        private final String key;

        Location(String key) {
            this.key = key;
        }

        public String getKey() {
            return key;
        }

        public static Optional<Location> byKey(String key) {
            if (locationByKey == null) {
                locationByKey = ImmutableMap.copyOf(EnumUtils.prepareKeysMap(Location.values()));
            }

            return Optional.ofNullable(locationByKey.get(key));
        }
    }

    private enum Resource implements EnumUtils.StringKey {
        CPU("cpu"),
        GPU("gpu"),
        S3_STORAGE("s3-storage"),
        RAM_GIG("ram_gig"),
        S3_HDD_X2("s3_hdd_x2"),
        ;

        private static Map<String, Resource> resourceByKey;
        private final String key;

        Resource(String key) {
            this.key = key;
        }

        public String getKey() {
            return key;
        }

        public static Optional<Resource> byKey(String key) {
            if (Resource.resourceByKey == null) {
                Resource.resourceByKey = ImmutableMap.copyOf(EnumUtils.prepareKeysMap(Resource.values()));
            }

            return Optional.ofNullable(Resource.resourceByKey.get(key));
        }
    }

    private Optional<Location> getLocationFromChange(QuotaChangeRequest.Change change) {
        return getLocationSegment(change)
                .flatMap(segment -> Location.byKey(segment.getPublicKey()));
    }

    private Optional<Segment> getLocationSegment(final QuotaChangeRequest.Change change) {
        return SegmentUtils.getSegmentBySegmentationKey(change.getSegments(), locationSegmentationKey);
    }

}
