package ru.yandex.qe.dispenser.ws.quota.request.owning_cost.formula;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.math.MathContext;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableMap;
import org.apache.commons.lang.NotImplementedException;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;

import ru.yandex.qe.dispenser.api.util.EnumUtils;
import ru.yandex.qe.dispenser.api.v1.DiUnit;
import ru.yandex.qe.dispenser.domain.QuotaChangeRequest;
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.ChangeOwningCostContext;

/**
 * Providers owning cost formula interface.
 * Formula used for calculating owning cost, if implemented for provider, or formula by default used instead.
 *
 * @author Ruslan Kadriev <aqru@yandex-team.ru>
 */
public interface ProviderOwningCostFormula {
    String DEFAULT_FORMULA_KEY = "default";
    BigDecimal DEFAULT_OWNING_COST = BigDecimal.ZERO;
    String ZERO = "0";
    /**
     * Math context, that should be used for all calculation in implemented formulas.
     */
    MathContext MATH_CONTEXT = new MathContext(18, RoundingMode.HALF_UP);

    /**
     * Used for mapping changes of provider to its formula.
     * @return formulas provider key
     */
    @NotNull
    String getProviderKey();

    /**
     * Override this method, if formula just need change without context.
     * @param changes from one request that are intended for the provider.
     * @return changes with calculated owning cost
     */
    @NotNull
    default Map<QuotaChangeRequest.ChangeKey, BigDecimal> calculateOwningCost(@NotNull Collection<QuotaChangeRequest.Change> changes) {
        throw new NotImplementedException();
    }

    /**
     * Override this method, if formula need change context.
     * @param changes with context from one request that are intended for the provider.
     * @return changes with calculated owning cost
     */
    @NotNull
    default Map<QuotaChangeRequest.ChangeKey, BigDecimal> calculateOwningCostFromContext(
            @NotNull Collection<ChangeOwningCostContext> changes) {
        return calculateOwningCost(changes.stream()
                .map(ChangeOwningCostContext::getChange)
                .collect(Collectors.toList()));
    }

    /**
     * Converting change to unit with math context and fail-safe.
     * @param change to convert
     * @param to unit to convert change
     * @param logger for logging fail, if occurred
     * @return {@link BigDecimal} with converted value, or {@link ProviderOwningCostFormula#DEFAULT_OWNING_COST} on fail
     */
    default BigDecimal convert(QuotaChangeRequest.Change change, final DiUnit to, Logger logger) {
        return ProviderOwningCostFormula.convertS(change, to, logger);
    }

    /**
     * Converting change to unit with math context and fail-safe.
     * @param change to convert
     * @param to unit to convert change
     * @param logger for logging fail, if occurred
     * @return {@link BigDecimal} with converted value, or {@link ProviderOwningCostFormula#DEFAULT_OWNING_COST} on fail
     */
    static BigDecimal convertS(QuotaChangeRequest.Change change, final DiUnit to, Logger logger) {
        DiUnit baseUnit = change.getResource().getType().getBaseUnit();
        BigDecimal result = BigDecimal.ZERO;

        try {
            result = to.convert(new BigDecimal(change.getAmount(), MATH_CONTEXT), baseUnit, MATH_CONTEXT);
        } catch (Exception e) {
            logger.warn("Failed convert " + change + " to " + to + " unit!", e);
        }

        return result;
    }

    /**
     * Convert owning cost to output format.
     * @param owningCost to convert
     * @return converted to output format owning cost
     */
    @NotNull
    static BigDecimal owningCostToOutputFormat(@NotNull BigDecimal owningCost) {
        return owningCost.setScale(0, RoundingMode.HALF_UP);
    }

    /**
     * Convert owning cost to output string.
     * @param owningCost to convert
     * @return converted to string in output format owning cost
     */
    @NotNull
    static String owningCostToOutputString(@NotNull BigDecimal owningCost) {
        return owningCostToOutputFormat(owningCost).toString();
    }

    /**
     * Calculating relative cost of request to provider quota price.
     * @param requestOwningCost to calculating and convert
     * @param quotaPrice for calculating relative cost
     * @return calculate relative cost of request to provider quota price, converted to string in output format
     */
    @NotNull
    static String relativeCostToProviderQuotaPrice(long requestOwningCost, @NotNull BigDecimal quotaPrice) {
        if (requestOwningCost <= 0 || quotaPrice.compareTo(BigDecimal.ZERO) <= 0) {
            return ZERO;
        }
        return new BigDecimal(requestOwningCost, MATH_CONTEXT)
                .divide(quotaPrice, MATH_CONTEXT)
                .setScale(1, RoundingMode.HALF_UP).stripTrailingZeros().toPlainString();
    }


    /**
     * Calculating percentage of request from campaign owning cost and convert it to output string.
     * @param requestOwningCost to calculating and convert
     * @param campaignOwningCost for calculating percentage
     * @return percentage of request from campaign owning cost, converted to string in output format
     */
    @NotNull
    static String percentageOfCampaignOwningCostOutputString(long requestOwningCost,
                                                             @NotNull BigInteger campaignOwningCost) {
        if (requestOwningCost <= 0 || campaignOwningCost.compareTo(BigInteger.ZERO) <= 0) {
            return ZERO;
        }

        return percentageOfCampaignOwningCostOutputString(new BigDecimal(requestOwningCost, MATH_CONTEXT),
                new BigDecimal(campaignOwningCost));
    }

    /**
     * Calculating percentage of request from campaign owning cost and convert it to output string.
     * @param requestsOwningCost to calculating and convert
     * @param campaignOwningCost for calculating percentage
     * @return percentage of request from campaign owning cost, converted to string in output format
     */
    @NotNull
    static String percentageOfCampaignOwningCostOutputForStatisticString(@NotNull BigInteger requestsOwningCost,
                                                                         @NotNull BigInteger campaignOwningCost) {
        if (requestsOwningCost.compareTo(BigInteger.ZERO) <= 0 || campaignOwningCost.compareTo(BigInteger.ZERO) <= 0) {
            return ZERO;
        }

        return percentageOfCampaignOwningCostOutputString(new BigDecimal(requestsOwningCost, MATH_CONTEXT),
                new BigDecimal(campaignOwningCost));
    }

    /**
     * Calculating percentage of request from campaign owning cost and convert it to output string.
     * @param requestOwningCost to calculating and convert, expected to be with {@link ProviderOwningCostFormula#MATH_CONTEXT}
     * @param campaignOwningCost  for calculating percentage, expected to be with {@link ProviderOwningCostFormula#MATH_CONTEXT}
     * @return percentage of request from campaign owning cost, converted to string in output format
     */
    @NotNull
    static String percentageOfCampaignOwningCostOutputString(@NotNull BigDecimal requestOwningCost,
                                                             @NotNull BigDecimal campaignOwningCost) {
        return requestOwningCost
                .divide(campaignOwningCost, MATH_CONTEXT)
                .movePointRight(2).setScale(1, RoundingMode.HALF_UP).stripTrailingZeros().toPlainString();
    }

    /**
     * Calculating percentage of change from request owning cost and convert it to output string.
     * @param owningCost of change to calculate and convert
     * @param requestOwningCost for calculating percentage
     * @return percentage of change from request owning cost, converted to string in output format
     */
    @NotNull
    static String percentageOfRequestOwningCostOutputString(@NotNull BigDecimal owningCost, long requestOwningCost) {
        if (owningCost.compareTo(BigDecimal.ZERO) <= 0 || requestOwningCost <= 0) {
            return ZERO;
        }

        return owningCost.divide(new BigDecimal(requestOwningCost), MATH_CONTEXT)
                .movePointRight(2).setScale(1, RoundingMode.HALF_UP).stripTrailingZeros().toPlainString();
    }

    @NotNull
    static Map<CampaignKey, List<ChangeOwningCostContext>> groupByCampaign(
            @NotNull Collection<ChangeOwningCostContext> changes, Map<QuotaChangeRequest.ChangeKey, BigDecimal> result) {
        Map<CampaignKey, List<ChangeOwningCostContext>> changesByCampaign = new HashMap<>();
        changes.forEach(change -> {
            Optional<CampaignKey> campaignKeyO = CampaignKey.byKey(change.getCampaign().getKey());
            if (campaignKeyO.isPresent()) {
                changesByCampaign.computeIfAbsent(campaignKeyO.get(), k -> new ArrayList<>()).add(change);
            } else {
                result.put(change.getChange().getKey(), DEFAULT_OWNING_COST);
            }
        });
        return changesByCampaign;
    }

    enum CampaignKey implements EnumUtils.StringKey {
        AUG2020("aug2020"),
        FEB2021("feb2021"),
        AUG2021("aug2021"),
        AUG_2022_AGGREGATED("aug2022aggregated"),
        AUG_2022_DRAFT("aug2022draft")
        ;

        private static Map<String, CampaignKey> campaignByKey;
        private final String key;

        CampaignKey(String key) {
            this.key = key;
        }

        @Override
        public String getKey() {
            return key;
        }

        public static Optional<CampaignKey> byKey(String key) {
            if (campaignByKey == null) {
                campaignByKey = ImmutableMap.copyOf(EnumUtils.prepareKeysMap(CampaignKey.values()));
            }

            return Optional.ofNullable(campaignByKey.get(key));
        }
    }
}
