package ru.yandex.qe.dispenser.ws.quota.request.owning_cost.formula;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableMap;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;
import ru.yandex.qe.dispenser.api.util.EnumUtils;
import ru.yandex.qe.dispenser.api.v1.DiUnit;
import ru.yandex.qe.dispenser.domain.QuotaChangeRequest;
import ru.yandex.qe.dispenser.ws.bot.Provider;
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.ChangeOwningCostContext;
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.pricing.PricingModel;
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.pricing.QuotaChangeOwningCostTariffManager;

/**
 * STRM owning cost formula(DISPENSER-4450).
 *
 * @author Ruslan Kadriev <aqru@yandex-team.ru>
 */
@Component
public class STRMOwningCostFormula implements ProviderOwningCostFormula {

    private static final Logger LOG = LoggerFactory.getLogger(STRMOwningCostFormula.class);

    private static final Provider PROVIDER = Provider.STRM;
    public static final String TRAFFIC_EGRESS = "strm.cdn.traffic.egress";
    private final QuotaChangeOwningCostTariffManager quotaChangeOwningCostTariffManager;

    public STRMOwningCostFormula(QuotaChangeOwningCostTariffManager quotaChangeOwningCostTariffManager) {
        this.quotaChangeOwningCostTariffManager = quotaChangeOwningCostTariffManager;
    }

    @Override
    public @NotNull String getProviderKey() {
        return PROVIDER.getServiceKey();
    }

    @Override
    public @NotNull Map<QuotaChangeRequest.ChangeKey, BigDecimal> calculateOwningCostFromContext(
            @NotNull Collection<ChangeOwningCostContext> changes) {
        Map<QuotaChangeRequest.ChangeKey, BigDecimal> result = new HashMap<>();
        Map<CampaignKey, List<ChangeOwningCostContext>> changesByCampaign = ProviderOwningCostFormula
                .groupByCampaign(changes, result);
        changesByCampaign.forEach((campaignKey, campaignChanges) -> {
            Map<String, ? extends PricingModel> pricingBySKU =
                    quotaChangeOwningCostTariffManager.getByProviderCampaign(PROVIDER, campaignKey.getKey()).stream()
                            .collect(Collectors.toMap(PricingModel::getSKU, Function.identity()));
            Map<QuotaChangeRequest.ChangeKey, BigDecimal> campaignCosts = campaignChanges.stream()
                    .map(ChangeOwningCostContext::getChange)
                    .collect(Collectors.toMap(QuotaChangeRequest.ChangeAmount::getKey,
                            change -> toOwningCost(change, pricingBySKU)));
            result.putAll(campaignCosts);
        });
        return result;
    }

    private BigDecimal toOwningCost(QuotaChangeRequest.Change change,
                                    Map<String, ? extends PricingModel> pricingBySKU) {
        if (pricingBySKU.isEmpty()) {
            return DEFAULT_OWNING_COST;
        }

        BigDecimal result = DEFAULT_OWNING_COST;
        String publicKey = change.getResource().getPublicKey();
        Optional<Resource> resourcesO = Resource.byKey(publicKey);
        if (resourcesO.isPresent()) {
            switch (resourcesO.get()) {
                case STRM_OUTBOUND_EXTERNAL_TRAFFIC:
                    result = calculateForTraffic(change, pricingBySKU);
                    break;
                case STRM_SIMULTANEOUS_LIVE_CASTS:
                    break;
            }
        }

        return result;
    }

    private BigDecimal calculateForTraffic(QuotaChangeRequest.Change change,
                                           Map<String, ? extends PricingModel> pricingBySKU) {
        BigDecimal result = DEFAULT_OWNING_COST;

        if (pricingBySKU.containsKey(TRAFFIC_EGRESS)) {
            PricingModel pricingModel = pricingBySKU.get(TRAFFIC_EGRESS);

            result = convert(change, DiUnit.TEBIBYTE, LOG)
                    .multiply(pricingModel.getPrice(), MATH_CONTEXT);
        }

        return result;
    }

    private enum Resource implements EnumUtils.StringKey {
        STRM_OUTBOUND_EXTERNAL_TRAFFIC("strm_outbound_external_traffic"),
        STRM_SIMULTANEOUS_LIVE_CASTS("strm_simultaneous_live_casts"),
        ;

        private static Map<String, Resource> resourceByKey;
        private final String key;

        Resource(String key) {
            this.key = key;
        }

        public String getKey() {
            return key;
        }

        public static Optional<Resource> byKey(String key) {
            if (Resource.resourceByKey == null) {
                Resource.resourceByKey = ImmutableMap.copyOf(EnumUtils.prepareKeysMap(Resource.values()));
            }

            return Optional.ofNullable(Resource.resourceByKey.get(key));
        }
    }
}
