package ru.yandex.qe.dispenser.ws.quota.request.owning_cost.formula;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableMap;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import ru.yandex.qe.dispenser.api.util.EnumUtils;
import ru.yandex.qe.dispenser.api.v1.DiUnit;
import ru.yandex.qe.dispenser.domain.QuotaChangeRequest;
import ru.yandex.qe.dispenser.domain.Segment;
import ru.yandex.qe.dispenser.domain.dao.segment.SegmentUtils;
import ru.yandex.qe.dispenser.ws.bot.Provider;
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.ChangeOwningCostContext;
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.pricing.PricingModel;
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.pricing.QuotaChangeOwningCostTariffManager;

/**
 * Solomon owning cost formula (DISPENSER-4404).
 *
 * @author Ruslan Kadriev <aqru@yandex-team.ru>
 */
@Component
public class SolomonOwningCostFormula implements ProviderOwningCostFormula {

    private static final Logger LOG = LoggerFactory.getLogger(SolomonOwningCostFormula.class);
    private static final Provider PROVIDER = Provider.SOLOMON;

    private static final String YP_CPU_QUOTA = "yp.cpu.quota";
    private static final String YP_MEMORY_QUOTA = "yp.memory.quota";
    public static final BigDecimal FLOW_DIVISOR = BigDecimal.valueOf(4_000_000L);
    public static final BigDecimal FLOW_METRICS_MULTIPLICAND = BigDecimal.valueOf(56L);
    public static final BigDecimal COMMON_MULTIPLICAND = BigDecimal.valueOf(2L);
    public static final BigDecimal METRICS_DIVISOR = BigDecimal.valueOf(400_000_000L);

    public static final BigDecimal ALERTS_DIVISOR = BigDecimal.valueOf(18_750L);
    public static final BigDecimal GIB_PER_CORE = BigDecimal.valueOf(8L);

    private final QuotaChangeOwningCostTariffManager quotaChangeOwningCostTariffManager;
    private final String locationSegmentationKey;

    public SolomonOwningCostFormula(QuotaChangeOwningCostTariffManager quotaChangeOwningCostTariffManager,
                                    @Value("${dispenser.location.segmentation.key}") final String locationSegmentationKey) {
        this.quotaChangeOwningCostTariffManager = quotaChangeOwningCostTariffManager;
        this.locationSegmentationKey = locationSegmentationKey;
    }

    @Override
    public @NotNull String getProviderKey() {
        return PROVIDER.getServiceKey();
    }

    @Override
    public @NotNull Map<QuotaChangeRequest.ChangeKey, BigDecimal> calculateOwningCostFromContext(@NotNull Collection<ChangeOwningCostContext> changes) {
        Map<QuotaChangeRequest.ChangeKey, BigDecimal> result = new HashMap<>();
        Map<CampaignKey, List<ChangeOwningCostContext>> changesByCampaign = ProviderOwningCostFormula
                .groupByCampaign(changes, result);
        changesByCampaign.forEach((campaignKey, campaignChanges) -> {
            Map<String, ? extends PricingModel> pricingBySKU =
                    quotaChangeOwningCostTariffManager.getByProviderCampaign(Provider.YP, campaignKey.getKey()).stream()
                            .collect(Collectors.toMap(PricingModel::getSKU, Function.identity()));
            Map<QuotaChangeRequest.ChangeKey, BigDecimal> campaignCosts = calculateCampaignOwningCost(
                    campaignKey, campaignChanges, pricingBySKU);
            result.putAll(campaignCosts);
        });

        return result;
    }

    private Map<QuotaChangeRequest.ChangeKey, BigDecimal> calculateCampaignOwningCost(
            CampaignKey campaignKey,
            List<ChangeOwningCostContext> changes,
            Map<String, ? extends PricingModel> pricingBySKU) {
        return switch (campaignKey) {
            case AUG2020, FEB2021 -> calculate2020CampaignOwningCost(changes, pricingBySKU);
            case AUG2021 -> calculate2021CampaignOwningCost(changes, pricingBySKU);
            case AUG_2022_DRAFT, AUG_2022_AGGREGATED -> calculate2022CampaignOwningCost(changes, pricingBySKU);
        };
    }

    private Map<QuotaChangeRequest.ChangeKey, BigDecimal> calculate2022CampaignOwningCost(
            List<ChangeOwningCostContext> changes, Map<String, ? extends PricingModel> pricingBySKU) {
        Map<QuotaChangeRequest.ChangeKey, BigDecimal> result = new HashMap<>();
        Map<Long, List<ChangeOwningCostContext>> changesByBigOrder = new HashMap<>();
        changes.forEach(change -> {
            if (change.getChange().getBigOrder() != null) {
                changesByBigOrder.computeIfAbsent(change.getChange().getBigOrder().getId(), k -> new ArrayList<>())
                        .add(change);
            } else {
                result.put(change.getChange().getKey(), DEFAULT_OWNING_COST);
            }
        });
        changesByBigOrder.forEach((bigOrderId, bigOrderChanges) -> {
            Map<QuotaChangeRequest.ChangeKey, BigDecimal> bigOrderCosts = calculate2022CampaignOrderOwningCost(
                    bigOrderChanges, pricingBySKU);
            result.putAll(bigOrderCosts);
        });
        return result;
    }

    private Map<QuotaChangeRequest.ChangeKey, BigDecimal> calculate2022CampaignOrderOwningCost(
            List<ChangeOwningCostContext> bigOrderChanges, Map<String, ? extends PricingModel> pricingBySKU) {
        if (!pricingBySKU.containsKey(YP_CPU_QUOTA) || !pricingBySKU.containsKey(YP_MEMORY_QUOTA)) {
            return bigOrderChanges.stream()
                    .collect(Collectors.toMap(k -> k.getChange().getKey(), v -> DEFAULT_OWNING_COST));
        }

        Map<QuotaChangeRequest.ChangeKey, BigDecimal> result = new HashMap<>();
        Map<Resource, List<ChangeOwningCostContext>> changesByResourceKey = new HashMap<>();
        bigOrderChanges.forEach(change -> {
            Optional<Resource> oResource = Resource.byKey(change.getChange().getResource().getPublicKey());
            if (oResource.isPresent()) {
                Resource key = oResource.get();
                changesByResourceKey.computeIfAbsent(key, (k) -> new ArrayList<>()).add(change);
            }
            result.put(change.getChange().getKey(), DEFAULT_OWNING_COST);
        });

        Map<Optional<Location>, ChangeOwningCostContext> flowByLocation =
                changesByResourceKey.getOrDefault(Resource.METRICS_WRITE_FLOW_SEGMENTED, List.of())
                        .stream()
                        .collect(Collectors.toMap(k -> getLocationFromChange(k.getChange()), Function.identity()));

        Optional<QuotaChangeRequest.Change> metricsStoredChangeO =
                Optional.ofNullable(changesByResourceKey.get(Resource.METRICS_STORED_COUNT))
                        .filter(changes -> changes.size() == 1)
                        .map(changes -> changes.get(0).getChange());
        Optional<BigDecimal> metricsStoredCountO =
                metricsStoredChangeO
                        .map(change -> convert(change, DiUnit.COUNT, LOG));

        Optional<QuotaChangeRequest.Change> alertsChange =
                Optional.ofNullable(changesByResourceKey.get(Resource.ALERTS_COUNT))
                .filter(changes -> changes.size() == 1)
                .map(changes -> changes.get(0).getChange());
        BigDecimal alerts = alertsChange
                .map(change -> convert(change, DiUnit.COUNT, LOG))
                .orElse(BigDecimal.ZERO);

        BigDecimal flow = flowByLocation.values()
                .stream()
                .map(change -> convert(change.getChange(), DiUnit.COUNT, LOG))
                .reduce((change1, change2) -> change1.add(change2, MATH_CONTEXT))
                .orElse(BigDecimal.ZERO);

        BigDecimal metrics = metricsStoredCountO.orElse(BigDecimal.ZERO);

        BigDecimal flowMetricsCores = totalCores(flow, metrics);
        BigDecimal flowMetricsGibRam = flowMetricsCores.multiply(GIB_PER_CORE, MATH_CONTEXT);

        BigDecimal alertsCores = alerts.divide(ALERTS_DIVISOR, RoundingMode.UP);
        BigDecimal alertsGibRam = alertsCores.multiply(GIB_PER_CORE, MATH_CONTEXT);

        PricingModel cpuQuotaSku = pricingBySKU.get(YP_CPU_QUOTA);
        PricingModel memoryQuotaSku = pricingBySKU.get(YP_MEMORY_QUOTA);

        BigDecimal flowMetricsCost = flowMetricsCores.multiply(cpuQuotaSku.getPrice(), MATH_CONTEXT)
                .add(flowMetricsGibRam.multiply(memoryQuotaSku.getPrice(), MATH_CONTEXT));
        BigDecimal alertsCost = alertsCores.multiply(cpuQuotaSku.getPrice(), MATH_CONTEXT)
                .add(alertsGibRam.multiply(memoryQuotaSku.getPrice(), MATH_CONTEXT), MATH_CONTEXT);

        alertsChange.ifPresent(change -> result.put(change.getKey(), alertsCost));
        if (flow.equals(BigDecimal.ZERO)) {
            metricsStoredChangeO.ifPresent(metricsStoredChange ->
                    result.put(metricsStoredChange.getKey(), flowMetricsCost));
        } else {
            flowByLocation.forEach((location, flowByLoc) -> {
                BigDecimal weight = convert(flowByLoc.getChange(), DiUnit.COUNT, LOG).divide(flow, MATH_CONTEXT);
                BigDecimal locationCost = flowMetricsCost.multiply(weight, MATH_CONTEXT);

                result.put(flowByLoc.getChange().getKey(), locationCost);
            });
        }

        return result;
    }

    private Map<QuotaChangeRequest.ChangeKey, BigDecimal> calculate2021CampaignOwningCost(
            List<ChangeOwningCostContext> changes, Map<String, ? extends PricingModel> pricingBySKU) {
        Map<QuotaChangeRequest.ChangeKey, BigDecimal> result = new HashMap<>();
        Map<Long, List<ChangeOwningCostContext>> changesByBigOrder = new HashMap<>();
        changes.forEach(change -> {
            if (change.getChange().getBigOrder() != null) {
                changesByBigOrder.computeIfAbsent(change.getChange().getBigOrder().getId(), k -> new ArrayList<>())
                        .add(change);
            } else {
                result.put(change.getChange().getKey(), DEFAULT_OWNING_COST);
            }
        });
        changesByBigOrder.forEach((bigOrderId, bigOrderChanges) -> {
            Map<QuotaChangeRequest.ChangeKey, BigDecimal> bigOrderCosts = calculate2021CampaignOrderOwningCost(
                    bigOrderChanges, pricingBySKU);
            result.putAll(bigOrderCosts);
        });
        return result;
    }

    private Map<QuotaChangeRequest.ChangeKey, BigDecimal> calculate2021CampaignOrderOwningCost(
            List<ChangeOwningCostContext> bigOrderChanges, Map<String, ? extends PricingModel> pricingBySKU) {
        if (!pricingBySKU.containsKey(YP_CPU_QUOTA) || !pricingBySKU.containsKey(YP_MEMORY_QUOTA)) {
            return bigOrderChanges.stream()
                    .collect(Collectors.toMap(k -> k.getChange().getKey(), v -> DEFAULT_OWNING_COST));
        }

        Map<QuotaChangeRequest.ChangeKey, BigDecimal> result = new HashMap<>();
        Map<Resource, List<ChangeOwningCostContext>> changesByResourceKey = new HashMap<>();
        bigOrderChanges.forEach(change -> {
            Optional<Resource> oResource = Resource.byKey(change.getChange().getResource().getPublicKey());
            if (oResource.isPresent()) {
                Resource key = oResource.get();
                changesByResourceKey.computeIfAbsent(key, (k) -> new ArrayList<>()).add(change);
            }
            result.put(change.getChange().getKey(), DEFAULT_OWNING_COST);
        });

        Map<Optional<Location>, ChangeOwningCostContext> flowByLocation =
                changesByResourceKey.getOrDefault(Resource.METRICS_WRITE_FLOW_SEGMENTED, List.of())
                        .stream()
                        .collect(Collectors.toMap(k -> getLocationFromChange(k.getChange()), Function.identity()));

        Optional<QuotaChangeRequest.Change> metricsStoredChangeO =
                Optional.ofNullable(changesByResourceKey.get(Resource.METRICS_STORED_COUNT))
                        .filter(changes -> changes.size() == 1)
                        .map(changes -> changes.get(0).getChange());
        Optional<BigDecimal> metricsStoredCountO =
                metricsStoredChangeO
                        .map(change -> convert(change, DiUnit.COUNT, LOG));

        BigDecimal flow = flowByLocation.values()
                .stream()
                .map(change -> convert(change.getChange(), DiUnit.COUNT, LOG))
                .reduce((change1, change2) -> change1.add(change2, MATH_CONTEXT))
                .orElse(BigDecimal.ZERO);

        BigDecimal metrics = metricsStoredCountO.orElse(BigDecimal.ZERO);

        BigDecimal totalCores = totalCores(flow, metrics);
        BigDecimal totalGibRam = totalCores.multiply(GIB_PER_CORE, MATH_CONTEXT);

        PricingModel cpuQuotaSku = pricingBySKU.get(YP_CPU_QUOTA);
        PricingModel memoryQuotaSku = pricingBySKU.get(YP_MEMORY_QUOTA);

        BigDecimal cost = totalCores.multiply(cpuQuotaSku.getPrice(), MATH_CONTEXT)
                .add(totalGibRam.multiply(memoryQuotaSku.getPrice(), MATH_CONTEXT));

        if (flow.equals(BigDecimal.ZERO)) {
            metricsStoredChangeO.ifPresent(metricsStoredChange ->
                    result.put(metricsStoredChange.getKey(), cost));
        } else {
            flowByLocation.forEach((location, flowByLoc) -> {
                BigDecimal weight = convert(flowByLoc.getChange(), DiUnit.COUNT, LOG).divide(flow, MATH_CONTEXT);
                BigDecimal locationCost = cost.multiply(weight, MATH_CONTEXT);

                result.put(flowByLoc.getChange().getKey(), locationCost);
            });
        }

        return result;
    }

    private BigDecimal totalCores(BigDecimal flow, BigDecimal metrics) {
        return COMMON_MULTIPLICAND.multiply(
                flow.divide(FLOW_DIVISOR, MATH_CONTEXT).multiply(FLOW_METRICS_MULTIPLICAND, MATH_CONTEXT).add(
                        flow.divide(FLOW_DIVISOR, MATH_CONTEXT)
                                .max(metrics.divide(METRICS_DIVISOR, MATH_CONTEXT))
                                .multiply(FLOW_METRICS_MULTIPLICAND, MATH_CONTEXT)
                ).setScale(0, RoundingMode.UP), MATH_CONTEXT
        );
    }

    private Map<QuotaChangeRequest.ChangeKey, BigDecimal> calculate2020CampaignOwningCost(
            List<ChangeOwningCostContext> changes, Map<String, ? extends PricingModel> pricingBySKU) {
        Map<QuotaChangeRequest.ChangeKey, BigDecimal> result = new HashMap<>();
        Map<Long, List<ChangeOwningCostContext>> changesByBigOrder = new HashMap<>();
        changes.forEach(change -> {
            if (change.getChange().getBigOrder() != null) {
                changesByBigOrder.computeIfAbsent(change.getChange().getBigOrder().getId(), k -> new ArrayList<>())
                        .add(change);
            } else {
                result.put(change.getChange().getKey(), DEFAULT_OWNING_COST);
            }
        });
        changesByBigOrder.forEach((bigOrderId, bigOrderChanges) -> {
            Map<QuotaChangeRequest.ChangeKey, BigDecimal> bigOrderCosts = calculate2020CampaignOrderOwningCost(
                    bigOrderChanges, pricingBySKU);
            result.putAll(bigOrderCosts);
        });
        return result;
    }

    private Map<QuotaChangeRequest.ChangeKey, BigDecimal> calculate2020CampaignOrderOwningCost(
            List<ChangeOwningCostContext> changes, Map<String, ? extends PricingModel> pricingBySKU) {
        if (!pricingBySKU.containsKey(YP_CPU_QUOTA) || !pricingBySKU.containsKey(YP_MEMORY_QUOTA)) {
            return changes.stream()
                    .collect(Collectors.toMap(k -> k.getChange().getKey(), v -> DEFAULT_OWNING_COST));
        }

        Map<QuotaChangeRequest.ChangeKey, BigDecimal> result = new HashMap<>();
        Map<Resource2020, ChangeOwningCostContext> changeByResourceKey = new HashMap<>();
        changes.forEach(change -> {
            Optional<Resource2020> oResource2020 = Resource2020.byKey(change.getChange().getResource().getPublicKey());
            if (oResource2020.isPresent()) {
                Resource2020 key = oResource2020.get();
                changeByResourceKey.put(key, change);
            }
            result.put(change.getChange().getKey(), DEFAULT_OWNING_COST);
        });

        Optional<BigDecimal> metricsWriteFlowO =
                Optional.ofNullable(changeByResourceKey.get(Resource2020.METRICS_WRITE_FLOW))
                        .map(change -> convert(change.getChange(), DiUnit.COUNT, LOG));
        Optional<BigDecimal> metricsStoredCountO =
                Optional.ofNullable(changeByResourceKey.get(Resource2020.METRICS_STORED_COUNT))
                        .map(change -> convert(change.getChange(), DiUnit.COUNT, LOG));

        BigDecimal flow = metricsWriteFlowO.orElse(BigDecimal.ZERO);
        BigDecimal metrics = metricsStoredCountO.orElse(BigDecimal.ZERO);

        BigDecimal totalCores = totalCores(flow, metrics);

        BigDecimal totalGibRam = totalCores.multiply(GIB_PER_CORE, MATH_CONTEXT);

        PricingModel cpuQuotaSku = pricingBySKU.get(YP_CPU_QUOTA);
        PricingModel memoryQuotaSku = pricingBySKU.get(YP_MEMORY_QUOTA);

        BigDecimal cost = totalCores.multiply(cpuQuotaSku.getPrice(), MATH_CONTEXT)
                .add(totalGibRam.multiply(memoryQuotaSku.getPrice(), MATH_CONTEXT));

        if (changeByResourceKey.containsKey(Resource2020.METRICS_WRITE_FLOW)
                && !flow.equals(BigDecimal.ZERO)) {
            result.put(changeByResourceKey.get(Resource2020.METRICS_WRITE_FLOW).getChange().getKey(), cost);
        } else if (changeByResourceKey.containsKey(Resource2020.METRICS_STORED_COUNT)) {
            result.put(changeByResourceKey.get(Resource2020.METRICS_STORED_COUNT).getChange().getKey(), cost);
        }

        return result;
    }

    private enum Resource implements EnumUtils.StringKey {
        METRICS_STORED_COUNT("metrics_stored_count"),
        METRICS_WRITE_FLOW_SEGMENTED("metrics_write_flow_segmented"),
        ALERTS_COUNT("alerts_count"),
        ;

        private static Map<String, Resource> resourceByKey;

        private final String key;

        Resource(String key) {
            this.key = key;
        }

        @Override
        public String getKey() {
            return key;
        }

        public static Optional<Resource> byKey(String key) {
            if (Resource.resourceByKey == null) {
                Resource.resourceByKey = ImmutableMap.copyOf(EnumUtils.prepareKeysMap(Resource.values()));
            }

            return Optional.ofNullable(Resource.resourceByKey.get(key));
        }
    }

    private enum Resource2020 implements EnumUtils.StringKey {
        METRICS_STORED_COUNT("metrics_stored_count"),
        METRICS_WRITE_FLOW("metrics_write_flow")
        ;

        private static Map<String, Resource2020> resourceByKey;

        private final String key;

        Resource2020(String key) {
            this.key = key;
        }

        @Override
        public String getKey() {
            return key;
        }

        public static Optional<Resource2020> byKey(String key) {
            if (resourceByKey == null) {
                resourceByKey = ImmutableMap.copyOf(EnumUtils.prepareKeysMap(Resource2020.values()));
            }

            return Optional.ofNullable(resourceByKey.get(key));
        }
    }

    private enum Location implements EnumUtils.StringKey {
        VLA("VLA"),
        SAS("SAS"),
        MAN("MAN"),
        MYT("MYT"),
        IVA("IVA"),
        KLG("KLG"),
        ;

        private static Map<String, Location> locationByKey;
        private final String key;

        Location(String key) {
            this.key = key;
        }

        public String getKey() {
            return key;
        }

        public static Optional<Location> byKey(String key) {
            if (locationByKey == null) {
                locationByKey = ImmutableMap.copyOf(EnumUtils.prepareKeysMap(Location.values()));
            }

            return Optional.ofNullable(locationByKey.get(key));
        }
    }

    private Optional<Location> getLocationFromChange(QuotaChangeRequest.Change change) {
        return getLocationSegment(change)
                .flatMap(segment -> Location.byKey(segment.getPublicKey()));
    }

    private Optional<Segment> getLocationSegment(final QuotaChangeRequest.Change change) {
        return SegmentUtils.getSegmentBySegmentationKey(change.getSegments(), locationSegmentationKey);
    }
}
