package ru.yandex.qe.dispenser.ws.quota.request.owning_cost.formula;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableMap;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.qe.dispenser.api.util.EnumUtils;
import ru.yandex.qe.dispenser.api.v1.DiUnit;
import ru.yandex.qe.dispenser.domain.QuotaChangeRequest;
import ru.yandex.qe.dispenser.ws.bot.Provider;
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.ChangeOwningCostContext;
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.pricing.PricingModel;
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.pricing.QuotaChangeOwningCostTariffManager;

/**
 * YDB owning cost formula (DISPENSER-4400).
 *
 * @author Ruslan Kadriev <aqru@yandex-team.ru>
 */
@Component
public class YDBOwningCostFormula implements ProviderOwningCostFormula {
    private static final Logger LOG = LoggerFactory.getLogger(YDBOwningCostFormula.class);

    private static final Provider PROVIDER = Provider.YDB;
    private static final BigDecimal GROUP_PHYSICAL_SIZE = new BigDecimal("1632", MATH_CONTEXT); // DISPENSER-4400
    private static final BigDecimal GROUP_CAPACITY_FOR_TABLES = new BigDecimal("240", MATH_CONTEXT); // DISPENSER-4400
    private static final BigDecimal TO_GIBIBYTE = new BigDecimal("10", MATH_CONTEXT).pow(9, MATH_CONTEXT)
            .divide(new BigDecimal("1024", MATH_CONTEXT).pow(3, MATH_CONTEXT), MATH_CONTEXT);
    private static final BigDecimal TO_GIGABYTE = new BigDecimal("1024", MATH_CONTEXT).pow(3, MATH_CONTEXT)
            .divide(new BigDecimal("10", MATH_CONTEXT).pow(9, MATH_CONTEXT), MATH_CONTEXT);
    private static final BigDecimal CORE_COEFFICIENT = new BigDecimal("0.46875", MATH_CONTEXT); // DISPENSER-4400
    private static final BigDecimal MIN_CPU_CORE = new BigDecimal("3", MATH_CONTEXT); // DISPENSER-4400
    private static final BigDecimal CPU_CORE_COEFFICIENT = new BigDecimal("12", MATH_CONTEXT); // DISPENSER-4400

    private static final String YP_SSD_STORAGE_QUOTA = "yp.ssd_storage.quota";
    private static final String YP_CPU_QUOTA = "yp.cpu.quota";
    private static final String YDB_SSD_SKU = "ydb.ssd_storage.quota";
    private static final String YDB_CPU_SKU = "ydb.cpu.quota";
    private static final String YDB_RAM_SKU = "ydb.memory.quota";
    private static final BigDecimal GIBIBYTES_PER_STORAGE_GROUP = BigDecimal.valueOf(230L);

    private final QuotaChangeOwningCostTariffManager quotaChangeOwningCostTariffManager;

    public YDBOwningCostFormula(QuotaChangeOwningCostTariffManager quotaChangeOwningCostTariffManager) {
        this.quotaChangeOwningCostTariffManager = quotaChangeOwningCostTariffManager;
    }

    @Override
    public @NotNull String getProviderKey() {
        return PROVIDER.getServiceKey();
    }

    @Override
    public @NotNull Map<QuotaChangeRequest.ChangeKey, BigDecimal> calculateOwningCostFromContext(
            @NotNull Collection<ChangeOwningCostContext> changes) {
        Map<QuotaChangeRequest.ChangeKey, BigDecimal> result = new HashMap<>();
        Map<CampaignKey, List<ChangeOwningCostContext>> changesByCampaign = ProviderOwningCostFormula
                .groupByCampaign(changes, result);
        changesByCampaign.forEach((campaignKey, campaignChanges) -> {
            Map<String, ? extends PricingModel> pricingBySKU =
                    quotaChangeOwningCostTariffManager.getByProvidersCampaign(List.of(Provider.YP, Provider.YDB),
                                    campaignKey.getKey())
                            .stream().collect(Collectors.toMap(PricingModel::getSKU, Function.identity()));
            Map<QuotaChangeRequest.ChangeKey, BigDecimal> campaignCosts = campaignChanges.stream()
                    .collect(Collectors.toMap(change -> change.getChange().getKey(),
                            change -> toOwningCost(change, pricingBySKU)));
            result.putAll(campaignCosts);
        });
        return result;
    }

    private BigDecimal toOwningCost(ChangeOwningCostContext change,
                                    Map<String, ? extends PricingModel> pricingBySKU) {
        BigDecimal result = DEFAULT_OWNING_COST;

        Optional<CampaignKey> campaignKeyO = CampaignKey.byKey(change.getCampaign().getKey());

        if (campaignKeyO.isPresent()) {
            switch (campaignKeyO.get()) {
                case AUG2020:
                case FEB2021:
                    result = toOwningCost2020(change.getChange(), pricingBySKU);
                    break;
                case AUG2021:
                    result = toOwningCost2021(change.getChange(), pricingBySKU);
                    break;
                case AUG_2022_DRAFT:
                case AUG_2022_AGGREGATED:
                    result = toOwningCost2022(change.getChange(), pricingBySKU);
                    break;
            }
        }

        return result;
    }

    private BigDecimal toOwningCost2022(QuotaChangeRequest.Change change,
                                        Map<String, ? extends PricingModel> pricingBySKU) {
        BigDecimal result = DEFAULT_OWNING_COST;
        if (pricingBySKU.isEmpty()) {
            return result;
        }
        String publicKey = change.getResource().getPublicKey();
        Optional<Resource2022> resourceO = Resource2022.byKey(publicKey);
        if (resourceO.isPresent()) {
            switch (resourceO.get()) {
                case STORAGE_GROUPS -> result = calculateForStorageGroups(change, pricingBySKU);
                case TOTAL_CPU -> result = calculateForTotalCpu(change, pricingBySKU);
                case TOTAL_RAM -> result = calculateForTotalRam(change, pricingBySKU);
            }
        }
        return result;
    }

    private BigDecimal calculateForStorageGroups(QuotaChangeRequest.Change change,
                                                 Map<String, ? extends PricingModel> pricingBySKU) {
        PricingModel pricing = pricingBySKU.get(YDB_SSD_SKU);
        if (pricing == null) {
            return ProviderOwningCostFormula.DEFAULT_OWNING_COST;
        }
        return convert(change, DiUnit.COUNT, LOG)
                .multiply(GIBIBYTES_PER_STORAGE_GROUP, MATH_CONTEXT)
                .multiply(pricing.getPrice(), MATH_CONTEXT);
    }

    private BigDecimal calculateForTotalCpu(QuotaChangeRequest.Change change,
                                            Map<String, ? extends PricingModel> pricingBySKU) {
        PricingModel pricing = pricingBySKU.get(YDB_CPU_SKU);
        if (pricing == null) {
            return ProviderOwningCostFormula.DEFAULT_OWNING_COST;
        }
        return convert(change, DiUnit.CORES, LOG)
                .multiply(pricing.getPrice(), MATH_CONTEXT);
    }

    private BigDecimal calculateForTotalRam(QuotaChangeRequest.Change change,
                                            Map<String, ? extends PricingModel> pricingBySKU) {
        PricingModel pricing = pricingBySKU.get(YDB_RAM_SKU);
        if (pricing == null) {
            return ProviderOwningCostFormula.DEFAULT_OWNING_COST;
        }
        return convert(change, DiUnit.GIBIBYTE, LOG)
                .multiply(pricing.getPrice(), MATH_CONTEXT);
    }

    private BigDecimal toOwningCost2020(QuotaChangeRequest.Change change, Map<String, ? extends PricingModel> pricingBySKU) {
        BigDecimal result = DEFAULT_OWNING_COST;

        if (pricingBySKU.isEmpty()) {
            return result;
        }

        String publicKey = change.getResource().getPublicKey();
        Optional<Resource2020> resourceO = Resource2020.byKey(publicKey);

        if (resourceO.isPresent()) {
            switch (resourceO.get()) {
                case ydb_ru_rps:
                    result = calculateForRps(change, pricingBySKU);
                    break;
                case ydb_ru_data_size:
                    result = calculateForDataSize(change, pricingBySKU);
                    break;
            }
        }

        return result;
    }

    private BigDecimal toOwningCost2021(QuotaChangeRequest.Change change, Map<String, ? extends PricingModel> pricingBySKU) {
        BigDecimal result = DEFAULT_OWNING_COST;

        if (pricingBySKU.isEmpty()) {
            return result;
        }

        String publicKey = change.getResource().getPublicKey();
        Optional<Resource> resourceO = Resource.byKey(publicKey);

        if (resourceO.isPresent()) {
            switch (resourceO.get()) {
                case data_size:
                    result = calculateForDataSize(change, pricingBySKU);
                    break;
                case userpool_cores:
                    result = calculateForUserpoolCores(change, pricingBySKU);
                    break;
            }
        }

        return result;
    }

    private BigDecimal calculateForDataSize(QuotaChangeRequest.Change change,
                                            Map<String, ? extends PricingModel> pricingBySKU) {
        BigDecimal result = DEFAULT_OWNING_COST;

        if (pricingBySKU.containsKey(YP_SSD_STORAGE_QUOTA)) {
            PricingModel pricingModel = pricingBySKU.get(YP_SSD_STORAGE_QUOTA);

            BigDecimal data_size_gibibytes = convert(change, DiUnit.GIBIBYTE, LOG);
            BigDecimal data_size_gigabytes = data_size_gibibytes.multiply(TO_GIGABYTE, MATH_CONTEXT);
            BigDecimal bs_group_physical_size = data_size_gigabytes.divide(GROUP_CAPACITY_FOR_TABLES, MATH_CONTEXT).setScale(0, RoundingMode.UP);
            BigDecimal physical_storage_gb = GROUP_PHYSICAL_SIZE.multiply(bs_group_physical_size, MATH_CONTEXT);

            result = pricingModel.getPrice().multiply(physical_storage_gb.multiply(TO_GIBIBYTE, MATH_CONTEXT), MATH_CONTEXT);
        }

        return result;
    }

    private BigDecimal calculateForUserpoolCores(QuotaChangeRequest.Change change,
                                                 Map<String, ? extends PricingModel> pricingBySKU) {
        BigDecimal result = DEFAULT_OWNING_COST;

        if (pricingBySKU.containsKey(YP_CPU_QUOTA)) {
            PricingModel pricingModel = pricingBySKU.get(YP_CPU_QUOTA);
            BigDecimal user_pool_cores = convert(change, DiUnit.CORES, LOG);
            BigDecimal cpu_cores = CPU_CORE_COEFFICIENT.multiply(
                    CORE_COEFFICIENT.multiply(user_pool_cores, MATH_CONTEXT).setScale(0, RoundingMode.UP)
                            .max(MIN_CPU_CORE), MATH_CONTEXT);
            result = pricingModel.getPrice().multiply(cpu_cores, MATH_CONTEXT);
        }

        return result;
    }

    private BigDecimal calculateForRps(QuotaChangeRequest.Change change,
                                       Map<String, ? extends PricingModel> pricingBySKU) {
        BigDecimal result = DEFAULT_OWNING_COST;

        if (pricingBySKU.containsKey(YP_CPU_QUOTA)) {
            BigDecimal user_pool_cores = convert(change, DiUnit.COUNT, LOG)
                    .divide(new BigDecimal("500", MATH_CONTEXT), MATH_CONTEXT)
                    .setScale(0, RoundingMode.UP);

            PricingModel pricingModel = pricingBySKU.get(YP_CPU_QUOTA);
            BigDecimal cpu_cores = CPU_CORE_COEFFICIENT.multiply(
                    CORE_COEFFICIENT.multiply(user_pool_cores, MATH_CONTEXT).setScale(0, RoundingMode.UP)
                            .max(MIN_CPU_CORE), MATH_CONTEXT);
            result = pricingModel.getPrice().multiply(cpu_cores, MATH_CONTEXT);
        }

        return result;
    }

    private enum Resource2022 implements EnumUtils.StringKey {
        STORAGE_GROUPS("storage_groups"),
        TOTAL_CPU("total_cpu"),
        TOTAL_RAM("total_ram")
        ;

        private static Map<String, Resource2022> resourceByKey;
        private final String key;

        Resource2022(String key) {
            this.key = key;
        }

        @Override
        public String getKey() {
            return key;
        }

        public static Optional<Resource2022> byKey(String key) {
            if (resourceByKey == null) {
                resourceByKey = ImmutableMap.copyOf(EnumUtils.prepareKeysMap(Resource2022.values()));
            }

            return Optional.ofNullable(resourceByKey.get(key));
        }
    }

    private enum Resource implements EnumUtils.StringKey {
        data_size("data_size"),
        userpool_cores("userpool_cores"),
        ;

        private static Map<String, Resource> resourceByKey;
        private final String key;

        Resource(String key) {
            this.key = key;
        }

        @Override
        public String getKey() {
            return key;
        }

        public static Optional<Resource> byKey(String key) {
            if (resourceByKey == null) {
                resourceByKey = ImmutableMap.copyOf(EnumUtils.prepareKeysMap(Resource.values()));
            }

            return Optional.ofNullable(resourceByKey.get(key));
        }
    }

    private enum Resource2020 implements EnumUtils.StringKey {
        ydb_ru_rps("ydb_ru-rps"),
        ydb_ru_data_size("ydb_ru-data_size"),
        ;

        private static Map<String, Resource2020> resourceByKey;
        private final String key;

        Resource2020(String key) {
            this.key = key;
        }

        @Override
        public String getKey() {
            return key;
        }

        public static Optional<Resource2020> byKey(String key) {
            if (resourceByKey == null) {
                resourceByKey = ImmutableMap.copyOf(EnumUtils.prepareKeysMap(Resource2020.values()));
            }

            return Optional.ofNullable(resourceByKey.get(key));
        }
    }
}
