package ru.yandex.qe.dispenser.ws.quota.request.owning_cost.formula;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Collection;
import java.util.EnumMap;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableMap;
import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import ru.yandex.bolts.collection.Tuple3;
import ru.yandex.qe.dispenser.api.util.EnumUtils;
import ru.yandex.qe.dispenser.api.v1.DiUnit;
import ru.yandex.qe.dispenser.domain.QuotaChangeRequest;
import ru.yandex.qe.dispenser.domain.Segment;
import ru.yandex.qe.dispenser.domain.dao.segment.SegmentUtils;
import ru.yandex.qe.dispenser.ws.bot.Provider;
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.ChangeOwningCostContext;
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.pricing.PricingModel;
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.pricing.QuotaChangeOwningCostTariffManager;

import static ru.yandex.bolts.collection.Tuple3.tuple;

/**
 * YP owning cost formula (DISPENSER-4375).
 *
 * @author Ruslan Kadriev <aqru@yandex-team.ru>
 */
@Component
public class YtOwningCostFormula implements ProviderOwningCostFormula {
    private static final Logger LOG = LoggerFactory.getLogger(YtOwningCostFormula.class);

    private static final String CLUSTER_PLACEHOLDER = "{cluster_name}";

    private static final String BURST_GUARANTEE_CPU_SKU_TEMPLATE = "yt.{cluster_name}.compute.integral_guarantee.burst.cpu";
    private static final String RELAXED_GUARANTEE_CPU_SKU_TEMPLATE = "yt.{cluster_name}.compute.integral_guarantee.relaxed.cpu";
    private static final String STRONG_GUARANTEE_CPU_SKU_TEMPLATE = "yt.{cluster_name}.compute.strong_guarantee.cpu";
    private static final String CPU_USAGE_SKU_TEMPLATE = "yt.{cluster_name}.compute.usage.cpu";
    private static final String MEMORY_USAGE_SKU_TEMPLATE = "yt.{cluster_name}.compute.usage.memory";
    private static final String HDD_SKU_TEMPLATE = "yt.{cluster_name}.disk_space.hdd";
    private static final String SSD_SKU_TEMPLATE = "yt.{cluster_name}.disk_space.ssd";
    private static final String STRONG_GUARANTEE_GPU_SKU_TEMPLATE = "yt.{cluster_name}.gpu.tesla_a100_80g.strong_guarantee.gpu";
    private static final String GPU_USAGE_SKU_TEMPLATE = "yt.{cluster_name}.gpu.tesla_a100_80g.usage.gpu";
    private static final String TABLE_CELL_SKU_TEMPLATE = "yt.{cluster_name}.dynamic_tables.tablet_cell";
    private static final String TABLE_STATIC_MEMORY_SKU_TEMPLATE = "yt.{cluster_name}.compute.usage.memory";

    /**
     * SKU's by cluster for YT resources
     */
    private static final Map<Cluster, List<String>> BURST_GUARANTEE_CPU_SKU_BY_CLUSTER = new EnumMap<>(Map.of(
            Cluster.ARNOLD, List.of(Cluster.ARNOLD.toClusterSKU(BURST_GUARANTEE_CPU_SKU_TEMPLATE),
                    Cluster.ARNOLD.toClusterSKU(CPU_USAGE_SKU_TEMPLATE)),
            Cluster.BOHR, List.of(Cluster.BOHR.toClusterSKU(BURST_GUARANTEE_CPU_SKU_TEMPLATE),
                    Cluster.BOHR.toClusterSKU(CPU_USAGE_SKU_TEMPLATE)),
            Cluster.FREUD, List.of(Cluster.FREUD.toClusterSKU(BURST_GUARANTEE_CPU_SKU_TEMPLATE),
                    Cluster.FREUD.toClusterSKU(CPU_USAGE_SKU_TEMPLATE)),
            Cluster.HAHN, List.of(Cluster.HAHN.toClusterSKU(BURST_GUARANTEE_CPU_SKU_TEMPLATE),
                    Cluster.HAHN.toClusterSKU(CPU_USAGE_SKU_TEMPLATE)),
            Cluster.HUME, List.of(Cluster.HUME.toClusterSKU(BURST_GUARANTEE_CPU_SKU_TEMPLATE),
                    Cluster.HUME.toClusterSKU(CPU_USAGE_SKU_TEMPLATE)),
            Cluster.LANDAU, List.of(Cluster.LANDAU.toClusterSKU(BURST_GUARANTEE_CPU_SKU_TEMPLATE),
                    Cluster.LANDAU.toClusterSKU(CPU_USAGE_SKU_TEMPLATE))
    ));

    private static final Map<Cluster, List<String>> RELAXED_GUARANTEE_CPU_SKU_BY_CLUSTER = new EnumMap<>(Map.of(
            Cluster.ARNOLD, List.of(Cluster.ARNOLD.toClusterSKU(RELAXED_GUARANTEE_CPU_SKU_TEMPLATE),
                    Cluster.ARNOLD.toClusterSKU(CPU_USAGE_SKU_TEMPLATE)),
            Cluster.BOHR, List.of(Cluster.BOHR.toClusterSKU(RELAXED_GUARANTEE_CPU_SKU_TEMPLATE),
                    Cluster.BOHR.toClusterSKU(CPU_USAGE_SKU_TEMPLATE)),
            Cluster.FREUD, List.of(Cluster.FREUD.toClusterSKU(RELAXED_GUARANTEE_CPU_SKU_TEMPLATE),
                    Cluster.FREUD.toClusterSKU(CPU_USAGE_SKU_TEMPLATE)),
            Cluster.HAHN, List.of(Cluster.HAHN.toClusterSKU(RELAXED_GUARANTEE_CPU_SKU_TEMPLATE),
                    Cluster.HAHN.toClusterSKU(CPU_USAGE_SKU_TEMPLATE)),
            Cluster.HUME, List.of(Cluster.HUME.toClusterSKU(RELAXED_GUARANTEE_CPU_SKU_TEMPLATE),
                    Cluster.HUME.toClusterSKU(CPU_USAGE_SKU_TEMPLATE)),
            Cluster.LANDAU, List.of(Cluster.LANDAU.toClusterSKU(RELAXED_GUARANTEE_CPU_SKU_TEMPLATE),
                    Cluster.LANDAU.toClusterSKU(CPU_USAGE_SKU_TEMPLATE))
    ));

    private static final Map<Cluster, List<String>> STRONG_GUARANTEE_CPU_SKU_BY_CLUSTER = new EnumMap<>(Map.of(
            Cluster.ARNOLD, List.of(Cluster.ARNOLD.toClusterSKU(STRONG_GUARANTEE_CPU_SKU_TEMPLATE),
                    Cluster.ARNOLD.toClusterSKU(CPU_USAGE_SKU_TEMPLATE)),
            Cluster.BOHR, List.of(Cluster.BOHR.toClusterSKU(STRONG_GUARANTEE_CPU_SKU_TEMPLATE),
                    Cluster.BOHR.toClusterSKU(CPU_USAGE_SKU_TEMPLATE)),
            Cluster.FREUD, List.of(Cluster.FREUD.toClusterSKU(STRONG_GUARANTEE_CPU_SKU_TEMPLATE),
                    Cluster.FREUD.toClusterSKU(CPU_USAGE_SKU_TEMPLATE)),
            Cluster.HAHN, List.of(Cluster.HAHN.toClusterSKU(STRONG_GUARANTEE_CPU_SKU_TEMPLATE),
                    Cluster.HAHN.toClusterSKU(CPU_USAGE_SKU_TEMPLATE)),
            Cluster.HUME, List.of(Cluster.HUME.toClusterSKU(STRONG_GUARANTEE_CPU_SKU_TEMPLATE),
                    Cluster.HUME.toClusterSKU(CPU_USAGE_SKU_TEMPLATE)),
            Cluster.LANDAU, List.of(Cluster.LANDAU.toClusterSKU(STRONG_GUARANTEE_CPU_SKU_TEMPLATE),
                    Cluster.LANDAU.toClusterSKU(CPU_USAGE_SKU_TEMPLATE)),
            Cluster.PYTHIA, List.of(Cluster.PYTHIA.toClusterSKU(STRONG_GUARANTEE_CPU_SKU_TEMPLATE),
                    Cluster.PYTHIA.toClusterSKU(CPU_USAGE_SKU_TEMPLATE)),
            Cluster.VANGA, List.of(Cluster.VANGA.toClusterSKU(STRONG_GUARANTEE_CPU_SKU_TEMPLATE),
                    Cluster.VANGA.toClusterSKU(CPU_USAGE_SKU_TEMPLATE))
    ));

    private static final Map<Cluster, String> MEMORY_SKU_BY_CLUSTER = new EnumMap<>(Map.of(
            Cluster.ARNOLD, Cluster.ARNOLD.toClusterSKU(MEMORY_USAGE_SKU_TEMPLATE),
            Cluster.BOHR, Cluster.BOHR.toClusterSKU(MEMORY_USAGE_SKU_TEMPLATE),
            Cluster.FREUD, Cluster.FREUD.toClusterSKU(MEMORY_USAGE_SKU_TEMPLATE),
            Cluster.HAHN, Cluster.HAHN.toClusterSKU(MEMORY_USAGE_SKU_TEMPLATE),
            Cluster.HUME, Cluster.HUME.toClusterSKU(MEMORY_USAGE_SKU_TEMPLATE),
            Cluster.LANDAU, Cluster.LANDAU.toClusterSKU(MEMORY_USAGE_SKU_TEMPLATE),
            Cluster.PYTHIA, Cluster.PYTHIA.toClusterSKU(MEMORY_USAGE_SKU_TEMPLATE),
            Cluster.VANGA, Cluster.VANGA.toClusterSKU(MEMORY_USAGE_SKU_TEMPLATE)
    ));

    private static final Map<Cluster, String> HDD_SKU_BY_CLUSTER = new EnumMap<>(Map.ofEntries(
            Map.entry(Cluster.ARNOLD, Cluster.ARNOLD.toClusterSKU(HDD_SKU_TEMPLATE)),
            Map.entry(Cluster.BOHR, Cluster.BOHR.toClusterSKU(HDD_SKU_TEMPLATE)),
            Map.entry(Cluster.FREUD, Cluster.FREUD.toClusterSKU(HDD_SKU_TEMPLATE)),
            Map.entry(Cluster.HAHN, Cluster.HAHN.toClusterSKU(HDD_SKU_TEMPLATE)),
            Map.entry(Cluster.HUME, Cluster.HUME.toClusterSKU(HDD_SKU_TEMPLATE)),
            Map.entry(Cluster.LANDAU, Cluster.LANDAU.toClusterSKU(HDD_SKU_TEMPLATE)),
            Map.entry(Cluster.PYTHIA, Cluster.PYTHIA.toClusterSKU(HDD_SKU_TEMPLATE)),
            Map.entry(Cluster.SENECA_MAN, Cluster.SENECA_MAN.toClusterSKU(HDD_SKU_TEMPLATE)),
            Map.entry(Cluster.SENECA_SAS, Cluster.SENECA_SAS.toClusterSKU(HDD_SKU_TEMPLATE)),
            Map.entry(Cluster.SENECA_VLA, Cluster.SENECA_VLA.toClusterSKU(HDD_SKU_TEMPLATE)),
            Map.entry(Cluster.SENECA_KLG, Cluster.SENECA_KLG.toClusterSKU(HDD_SKU_TEMPLATE)),
            Map.entry(Cluster.VANGA, Cluster.VANGA.toClusterSKU(HDD_SKU_TEMPLATE)),
            Map.entry(Cluster.ZENO, Cluster.ZENO.toClusterSKU(HDD_SKU_TEMPLATE)),
            Map.entry(Cluster.MARKOV, Cluster.MARKOV.toClusterSKU(SSD_SKU_TEMPLATE)),
            Map.entry(Cluster.NASH, Cluster.NASH.toClusterSKU(HDD_SKU_TEMPLATE))
    ));

    private static final Map<Cluster, String> SSD_SKU_BY_CLUSTER = new EnumMap<>(Map.ofEntries(
            Map.entry(Cluster.ARNOLD, Cluster.ARNOLD.toClusterSKU(SSD_SKU_TEMPLATE)),
            Map.entry(Cluster.BOHR, Cluster.BOHR.toClusterSKU(SSD_SKU_TEMPLATE)),
            Map.entry(Cluster.FREUD, Cluster.FREUD.toClusterSKU(SSD_SKU_TEMPLATE)),
            Map.entry(Cluster.HAHN, Cluster.HAHN.toClusterSKU(SSD_SKU_TEMPLATE)),
            Map.entry(Cluster.HUME, Cluster.HUME.toClusterSKU(SSD_SKU_TEMPLATE)),
            Map.entry(Cluster.LANDAU, Cluster.LANDAU.toClusterSKU(SSD_SKU_TEMPLATE)),
            Map.entry(Cluster.MARKOV, Cluster.MARKOV.toClusterSKU(SSD_SKU_TEMPLATE)),
            Map.entry(Cluster.PYTHIA, Cluster.PYTHIA.toClusterSKU(SSD_SKU_TEMPLATE)),
            Map.entry(Cluster.SENECA_MAN, Cluster.SENECA_MAN.toClusterSKU(SSD_SKU_TEMPLATE)),
            Map.entry(Cluster.SENECA_SAS, Cluster.SENECA_SAS.toClusterSKU(SSD_SKU_TEMPLATE)),
            Map.entry(Cluster.SENECA_VLA, Cluster.SENECA_VLA.toClusterSKU(SSD_SKU_TEMPLATE)),
            Map.entry(Cluster.SENECA_KLG, Cluster.SENECA_KLG.toClusterSKU(SSD_SKU_TEMPLATE)),
            Map.entry(Cluster.VANGA, Cluster.VANGA.toClusterSKU(SSD_SKU_TEMPLATE)),
            Map.entry(Cluster.ZENO, Cluster.ZENO.toClusterSKU(SSD_SKU_TEMPLATE)),
            Map.entry(Cluster.NASH, Cluster.NASH.toClusterSKU(SSD_SKU_TEMPLATE))
    ));

    private static final Map<Cluster, String> TABLE_CELL_SKU_BY_CLUSTER = new EnumMap<>(Map.ofEntries(
            Map.entry(Cluster.ARNOLD, Cluster.ARNOLD.toClusterSKU(TABLE_CELL_SKU_TEMPLATE)),
            Map.entry(Cluster.BOHR, Cluster.BOHR.toClusterSKU(TABLE_CELL_SKU_TEMPLATE)),
            Map.entry(Cluster.FREUD, Cluster.FREUD.toClusterSKU(TABLE_CELL_SKU_TEMPLATE)),
            Map.entry(Cluster.HAHN, Cluster.HAHN.toClusterSKU(TABLE_CELL_SKU_TEMPLATE)),
            Map.entry(Cluster.HUME, Cluster.HUME.toClusterSKU(TABLE_CELL_SKU_TEMPLATE)),
            Map.entry(Cluster.LANDAU, Cluster.LANDAU.toClusterSKU(TABLE_CELL_SKU_TEMPLATE)),
            Map.entry(Cluster.MARKOV, Cluster.MARKOV.toClusterSKU(TABLE_CELL_SKU_TEMPLATE)),
            Map.entry(Cluster.PYTHIA, Cluster.PYTHIA.toClusterSKU(TABLE_CELL_SKU_TEMPLATE)),
            Map.entry(Cluster.SENECA_MAN, Cluster.SENECA_MAN.toClusterSKU(TABLE_CELL_SKU_TEMPLATE)),
            Map.entry(Cluster.SENECA_SAS, Cluster.SENECA_SAS.toClusterSKU(TABLE_CELL_SKU_TEMPLATE)),
            Map.entry(Cluster.SENECA_VLA, Cluster.SENECA_VLA.toClusterSKU(TABLE_CELL_SKU_TEMPLATE)),
            Map.entry(Cluster.VANGA, Cluster.VANGA.toClusterSKU(TABLE_CELL_SKU_TEMPLATE)),
            Map.entry(Cluster.ZENO, Cluster.ZENO.toClusterSKU(TABLE_CELL_SKU_TEMPLATE))
    ));

    private static final Map<Cluster, String> TABLE_STATIC_MEMORY_SKU_BY_CLUSTER = new EnumMap<>(Map.ofEntries(
            Map.entry(Cluster.ARNOLD, Cluster.ARNOLD.toClusterSKU(TABLE_STATIC_MEMORY_SKU_TEMPLATE)),
            Map.entry(Cluster.BOHR, Cluster.BOHR.toClusterSKU(TABLE_STATIC_MEMORY_SKU_TEMPLATE)),
            Map.entry(Cluster.FREUD, Cluster.FREUD.toClusterSKU(TABLE_STATIC_MEMORY_SKU_TEMPLATE)),
            Map.entry(Cluster.HAHN, Cluster.HAHN.toClusterSKU(TABLE_STATIC_MEMORY_SKU_TEMPLATE)),
            Map.entry(Cluster.HUME, Cluster.HUME.toClusterSKU(TABLE_STATIC_MEMORY_SKU_TEMPLATE)),
            Map.entry(Cluster.LANDAU, Cluster.LANDAU.toClusterSKU(TABLE_STATIC_MEMORY_SKU_TEMPLATE)),
            Map.entry(Cluster.PYTHIA, Cluster.PYTHIA.toClusterSKU(TABLE_STATIC_MEMORY_SKU_TEMPLATE)),
            Map.entry(Cluster.VANGA, Cluster.VANGA.toClusterSKU(TABLE_STATIC_MEMORY_SKU_TEMPLATE))
    ));

    private static final Set<Cluster> CLUSTERS_WITH_FIXED_TABLE_STATIC_MEMORY_PRICE = EnumSet.copyOf(Set.of(
            Cluster.MARKOV,
            Cluster.SENECA_MAN,
            Cluster.SENECA_SAS,
            Cluster.SENECA_VLA,
            Cluster.ZENO
    ));

    private static final Map<Cluster, List<String>> GPU_SKU_BY_CLUSTER = new EnumMap<>(Map.of(
            Cluster.ARNOLD, List.of(Cluster.ARNOLD.toClusterSKU(STRONG_GUARANTEE_GPU_SKU_TEMPLATE),
                    Cluster.ARNOLD.toClusterSKU(GPU_USAGE_SKU_TEMPLATE)),
            Cluster.HAHN, List.of(Cluster.HAHN.toClusterSKU(STRONG_GUARANTEE_GPU_SKU_TEMPLATE),
                    Cluster.HAHN.toClusterSKU(GPU_USAGE_SKU_TEMPLATE))
    ));

    private static final Map<Cluster, Tuple3<String, String, String>> RPC_PROXY_SKU_BY_CLUSTER = new EnumMap<>(Map.of(
            Cluster.ARNOLD, tuple(Cluster.ARNOLD.toClusterSKU(STRONG_GUARANTEE_CPU_SKU_TEMPLATE),
                    Cluster.ARNOLD.toClusterSKU(CPU_USAGE_SKU_TEMPLATE), Cluster.ARNOLD.toClusterSKU(MEMORY_USAGE_SKU_TEMPLATE)),
            Cluster.BOHR, tuple(Cluster.BOHR.toClusterSKU(STRONG_GUARANTEE_CPU_SKU_TEMPLATE),
                    Cluster.BOHR.toClusterSKU(CPU_USAGE_SKU_TEMPLATE), Cluster.BOHR.toClusterSKU(MEMORY_USAGE_SKU_TEMPLATE)),
            Cluster.FREUD, tuple(Cluster.FREUD.toClusterSKU(STRONG_GUARANTEE_CPU_SKU_TEMPLATE),
                    Cluster.FREUD.toClusterSKU(CPU_USAGE_SKU_TEMPLATE), Cluster.FREUD.toClusterSKU(MEMORY_USAGE_SKU_TEMPLATE)),
            Cluster.HAHN, tuple(Cluster.HAHN.toClusterSKU(STRONG_GUARANTEE_CPU_SKU_TEMPLATE),
                    Cluster.HAHN.toClusterSKU(CPU_USAGE_SKU_TEMPLATE), Cluster.HAHN.toClusterSKU(MEMORY_USAGE_SKU_TEMPLATE)),
            Cluster.HUME, tuple(Cluster.HUME.toClusterSKU(STRONG_GUARANTEE_CPU_SKU_TEMPLATE),
                    Cluster.HUME.toClusterSKU(CPU_USAGE_SKU_TEMPLATE), Cluster.HUME.toClusterSKU(MEMORY_USAGE_SKU_TEMPLATE)),
            Cluster.LANDAU, tuple(Cluster.LANDAU.toClusterSKU(STRONG_GUARANTEE_CPU_SKU_TEMPLATE),
                    Cluster.LANDAU.toClusterSKU(CPU_USAGE_SKU_TEMPLATE), Cluster.LANDAU.toClusterSKU(MEMORY_USAGE_SKU_TEMPLATE)),
            Cluster.PYTHIA, tuple(Cluster.PYTHIA.toClusterSKU(STRONG_GUARANTEE_CPU_SKU_TEMPLATE),
                    Cluster.PYTHIA.toClusterSKU(CPU_USAGE_SKU_TEMPLATE), Cluster.PYTHIA.toClusterSKU(MEMORY_USAGE_SKU_TEMPLATE)),
            Cluster.VANGA, tuple(Cluster.VANGA.toClusterSKU(STRONG_GUARANTEE_CPU_SKU_TEMPLATE),
                    Cluster.VANGA.toClusterSKU(CPU_USAGE_SKU_TEMPLATE), Cluster.VANGA.toClusterSKU(MEMORY_USAGE_SKU_TEMPLATE))
    ));

    private static final Map<Cluster, List<String>> DYNT_CPU_SKU_BY_CLUSTER = new EnumMap<>(
            generateTable(List.of(Cluster.HAHN, Cluster.ARNOLD, Cluster.HUME, Cluster.FREUD, Cluster.BOHR,
                    Cluster.LANDAU, Cluster.MARKOV, Cluster.SENECA_SAS, Cluster.SENECA_VLA, Cluster.SENECA_KLG,
                    Cluster.ZENO, Cluster.NASH, Cluster.VANGA, Cluster.PYTHIA),
                    List.of(BURST_GUARANTEE_CPU_SKU_TEMPLATE, CPU_USAGE_SKU_TEMPLATE, RELAXED_GUARANTEE_CPU_SKU_TEMPLATE))
    );

    private static final Map<Cluster, String> DYNT_MEMORY_SKU_BY_CLUSTER = new EnumMap<>(
            generateTable(List.of(Cluster.HAHN, Cluster.ARNOLD, Cluster.HUME, Cluster.FREUD, Cluster.BOHR,
                            Cluster.LANDAU, Cluster.MARKOV, Cluster.SENECA_SAS, Cluster.SENECA_VLA, Cluster.SENECA_KLG,
                            Cluster.ZENO, Cluster.NASH, Cluster.VANGA, Cluster.PYTHIA),
                    MEMORY_USAGE_SKU_TEMPLATE)
    );

    private static Map<Cluster, List<String>> generateTable(List<Cluster> clusters, List<String> skuTemplates) {
        return clusters.stream()
                .map(c -> Pair.of(c, skuTemplates.stream().map(c::toClusterSKU).collect(Collectors.toList())))
                .collect(Collectors.toMap(Pair::getKey, Pair::getValue));
    }

    private static Map<Cluster, String> generateTable(List<Cluster> clusters, String skuTemplate) {
        return clusters.stream()
                .map(c -> Pair.of(c, c.toClusterSKU(skuTemplate)))
                .collect(Collectors.toMap(Pair::getKey, Pair::getValue));
    }

    private static final Provider provider = Provider.YT;
    public static final BigDecimal RELAXED_MEMORY_TARIFF_COEFFICIENT = new BigDecimal("0.8", MATH_CONTEXT);
    public static final BigDecimal BURST_MEMORY_TARIFF_COEFFICIENT = new BigDecimal("0.2", MATH_CONTEXT);
    public static final BigDecimal FIXED_STATIC_MEMORY_PRICE = new BigDecimal("29.693", MATH_CONTEXT);
    public static final BigDecimal RPC_PROXY_CPU_MULTIPLIER = new BigDecimal("13", MATH_CONTEXT);
    public static final BigDecimal RPC_PROXY_MEMORY_MULTIPLIER = new BigDecimal("20", MATH_CONTEXT);
    public static final BigDecimal CLUSTER_NETWORK_IN_TABLET_CELL = new BigDecimal("50", MATH_CONTEXT); // DISPENSER-4434
    public static final BigDecimal DYNT_MEMORY_TARIFF_COEFFICIENT = RELAXED_MEMORY_TARIFF_COEFFICIENT
            .add(BURST_MEMORY_TARIFF_COEFFICIENT, MATH_CONTEXT);
    private final String ytClusterSegmentationKey;
    private final QuotaChangeOwningCostTariffManager quotaChangeOwningCostTariffManager;

    public YtOwningCostFormula(@Value("${dispenser.yt.segmentation.key}") final String ytClusterSegmentationKey,
                               QuotaChangeOwningCostTariffManager quotaChangeOwningCostTariffManager) {
        this.ytClusterSegmentationKey = ytClusterSegmentationKey;
        this.quotaChangeOwningCostTariffManager = quotaChangeOwningCostTariffManager;
    }

    @Override
    public @NotNull String getProviderKey() {
        return provider.getServiceKey();
    }

    @Override
    public @NotNull Map<QuotaChangeRequest.ChangeKey, BigDecimal> calculateOwningCostFromContext(
            @NotNull Collection<ChangeOwningCostContext> changes) {
        Map<QuotaChangeRequest.ChangeKey, BigDecimal> result = new HashMap<>();
        Map<CampaignKey, List<ChangeOwningCostContext>> changesByCampaign = ProviderOwningCostFormula
                .groupByCampaign(changes, result);
        changesByCampaign.forEach((campaignKey, campaignChanges) -> {
            Map<String, ? extends PricingModel> pricingBySKU =
                    quotaChangeOwningCostTariffManager.getByProviderCampaign(provider, campaignKey.getKey()).stream()
                            .collect(Collectors.toMap(PricingModel::getSKU, Function.identity()));
            Map<QuotaChangeRequest.ChangeKey, BigDecimal> campaignCosts = campaignChanges.stream()
                    .collect(Collectors.toMap(change -> change.getChange().getKey(),
                            change -> toOwningCost(change, pricingBySKU)));
            result.putAll(campaignCosts);
        });
        return result;
    }

    private BigDecimal toOwningCost(ChangeOwningCostContext change,
                                    Map<String, ? extends PricingModel> pricingBySKU) {
        BigDecimal result = DEFAULT_OWNING_COST;

        Optional<CampaignKey> campaignKeyO = CampaignKey.byKey(change.getCampaign().getKey());

        if (campaignKeyO.isPresent()) {
            result = switch (campaignKeyO.get()) {
                case AUG2020, FEB2021 -> toOwningCostOld(change.getChange(), pricingBySKU);
                case AUG2021 -> toOwningCostAug2021(change.getChange(), pricingBySKU);
                case AUG_2022_DRAFT, AUG_2022_AGGREGATED -> toOwningCostAug2022(change.getChange(), pricingBySKU);
            };
        }

        return result;
    }

    private BigDecimal toOwningCostOld(QuotaChangeRequest.Change change,
                                       Map<String, ? extends PricingModel> pricingBySKU) {
        BigDecimal result = DEFAULT_OWNING_COST;

        if (pricingBySKU.isEmpty()) {
            return result;
        }

        Optional<Cluster> clusterO = getClusterFromChange(change, ytClusterSegmentationKey);
        if (clusterO.isPresent()) {
            Cluster cluster = clusterO.get();

            String publicKey = change.getResource().getPublicKey();
            Optional<Resource2020> resourcesO = Resource2020.byKey(publicKey);
            if (resourcesO.isPresent()) {
                result = switch (resourcesO.get()) {
                    case CPU_CLICKHOUSE -> calculateForStrongGuaranteeCpu(change, cluster, pricingBySKU);
                    case CLUSTER_NETWORK_TRAFFIC -> calculateForClusterNetworkTraffic(change, cluster, pricingBySKU);
                };
            } else {
                result = toOwningCostAug2021(change, pricingBySKU);
            }
        }

        return result;
    }

    private BigDecimal toOwningCostAug2021(QuotaChangeRequest.Change change,
                                           Map<String, ? extends PricingModel> pricingBySKU) {
        BigDecimal result = DEFAULT_OWNING_COST;

        if (pricingBySKU.isEmpty()) {
            return result;
        }

        Optional<Cluster> clusterO = getClusterFromChange(change, ytClusterSegmentationKey);
        if (clusterO.isPresent()) {
            Cluster cluster = clusterO.get();

            String publicKey = change.getResource().getPublicKey();
            Optional<Resource> resourcesO = Resource.byKey(publicKey);
            if (resourcesO.isPresent()) {
                result = switch (resourcesO.get()) {
                    case BURST_GUARANTEE_CPU -> calculateForBurstGuaranteeCpu(change, cluster, pricingBySKU);
                    case CPU -> calculateForStrongGuaranteeCpu(change, cluster, pricingBySKU);
                    case CPU_FLOW -> calculateForRelaxedGuaranteeCpu(change, cluster, pricingBySKU);
                    case GPU -> calculateForGpu(change, cluster, pricingBySKU);
                    case HDD -> calculateForHDD(change, cluster, pricingBySKU);
                    case MEM_BURST -> calculateForBurstGuaranteeMemory(change, cluster, pricingBySKU);
                    case MEM_RELAXED -> calculateForRelaxedGuaranteeMemory(change, cluster, pricingBySKU);
                    case MEM_STRONG -> calculateForStrongGuaranteeMemory(change, cluster, pricingBySKU);
                    case RPC_PROXY -> calculateForRPCProxy(change, cluster, pricingBySKU);
                    case SSD -> calculateForSSD(change, cluster, pricingBySKU);
                    case TABLET_CELL_BUNDLE -> calculateForTableCell(change, cluster, pricingBySKU);
                    case TABLET_STATIC_MEMORY -> calculateForStaticMemory(change, cluster, pricingBySKU);
                    default -> DEFAULT_OWNING_COST;
                };
            }
        }

        return result;
    }

    private BigDecimal toOwningCostAug2022(QuotaChangeRequest.Change change,
                                           Map<String, ? extends PricingModel> pricingBySKU) {
        BigDecimal result = DEFAULT_OWNING_COST;

        if (pricingBySKU.isEmpty()) {
            return result;
        }

        Optional<Cluster> clusterO = getClusterFromChange(change, ytClusterSegmentationKey);
        if (clusterO.isPresent()) {
            Cluster cluster = clusterO.get();

            String publicKey = change.getResource().getPublicKey();
            Optional<Resource> resourcesO = Resource.byKey(publicKey);
            if (resourcesO.isPresent()) {
                result = switch (resourcesO.get()) {
                    case BURST_GUARANTEE_CPU -> calculateForBurstGuaranteeCpu(change, cluster, pricingBySKU);
                    case CPU -> calculateForStrongGuaranteeCpu(change, cluster, pricingBySKU);
                    case CPU_FLOW -> calculateForRelaxedGuaranteeCpu(change, cluster, pricingBySKU);
                    case GPU, GPU_A100 -> calculateForGpu(change, cluster, pricingBySKU);
                    case HDD -> calculateForHDD(change, cluster, pricingBySKU);
                    case MEM_BURST -> calculateForBurstGuaranteeMemory(change, cluster, pricingBySKU);
                    case MEM_RELAXED -> calculateForRelaxedGuaranteeMemory(change, cluster, pricingBySKU);
                    case MEM_STRONG -> calculateForStrongGuaranteeMemory(change, cluster, pricingBySKU);
                    case SSD -> calculateForSSD(change, cluster, pricingBySKU);
                    case CPU_DYNT -> calculateForDyntCpu(change, cluster, pricingBySKU);
                    case MEM_DYNT -> calculateForDyntMemory(change, cluster, pricingBySKU);
                    default -> DEFAULT_OWNING_COST;
                };
            }
        }

        return result;
    }

    private BigDecimal calculateForDyntCpu(QuotaChangeRequest.Change change, Cluster cluster,
                                           Map<String, ? extends PricingModel> prisingModelsBySKU) {
        BigDecimal result = DEFAULT_OWNING_COST;

        if (DYNT_CPU_SKU_BY_CLUSTER.containsKey(cluster)) {
            List<String> SKUList = DYNT_CPU_SKU_BY_CLUSTER.get(cluster);

            if (prisingModelsBySKU.keySet().containsAll(SKUList)) {
                BigDecimal price = SKUList.stream()
                        .map(prisingModelsBySKU::get)
                        .map(PricingModel::getPrice)
                        .reduce(BigDecimal.ZERO, (x, y) -> x.add(y, MATH_CONTEXT));

                result = convert(change, DiUnit.CORES, LOG).multiply(price, MATH_CONTEXT);
            }
        }

        return result;
    }

    private BigDecimal calculateForDyntMemory(QuotaChangeRequest.Change change, Cluster cluster,
                                              Map<String, ? extends PricingModel> prisingModelsBySKU) {
        BigDecimal result = DEFAULT_OWNING_COST;

        if (DYNT_MEMORY_SKU_BY_CLUSTER.containsKey(cluster)) {
            String sku = DYNT_MEMORY_SKU_BY_CLUSTER.get(cluster);

            if (prisingModelsBySKU.containsKey(sku)) {
                PricingModel pricingModel = prisingModelsBySKU.get(sku);

                result = convert(change, DiUnit.GIBIBYTE, LOG).multiply(pricingModel.getPrice()
                        .multiply(DYNT_MEMORY_TARIFF_COEFFICIENT, MATH_CONTEXT), MATH_CONTEXT);
            }
        }

        return result;
    }

    private BigDecimal calculateForBurstGuaranteeCpu(QuotaChangeRequest.Change change, Cluster cluster,
                                                     Map<String, ? extends PricingModel> prisingModelsBySKU) {
        BigDecimal result = DEFAULT_OWNING_COST;

        if (BURST_GUARANTEE_CPU_SKU_BY_CLUSTER.containsKey(cluster)) {
            List<String> SKUList = BURST_GUARANTEE_CPU_SKU_BY_CLUSTER.get(cluster);

            if (SKUList.size() == 2 && prisingModelsBySKU.keySet().containsAll(SKUList)) {
                PricingModel pricingModel1 = prisingModelsBySKU.get(SKUList.get(0));
                PricingModel pricingModel2 = prisingModelsBySKU.get(SKUList.get(1));

                result = convert(change, DiUnit.CORES, LOG)
                        .multiply(pricingModel1.add(pricingModel2, MATH_CONTEXT), MATH_CONTEXT);
            }
        }

        return result;
    }

    private BigDecimal calculateForRelaxedGuaranteeCpu(QuotaChangeRequest.Change change, Cluster cluster,
                                                       Map<String, ? extends PricingModel> prisingModelsBySKU) {
        BigDecimal result = DEFAULT_OWNING_COST;

        if (RELAXED_GUARANTEE_CPU_SKU_BY_CLUSTER.containsKey(cluster)) {
            List<String> SKUList = RELAXED_GUARANTEE_CPU_SKU_BY_CLUSTER.get(cluster);

            if (SKUList.size() == 2 && prisingModelsBySKU.keySet().containsAll(SKUList)) {
                PricingModel pricingModel1 = prisingModelsBySKU.get(SKUList.get(0));
                PricingModel pricingModel2 = prisingModelsBySKU.get(SKUList.get(1));

                result = convert(change, DiUnit.CORES, LOG)
                        .multiply(pricingModel1.add(pricingModel2, MATH_CONTEXT), MATH_CONTEXT);
            }
        }

        return result;
    }

    protected static BigDecimal calculateForStrongGuaranteeCpu(QuotaChangeRequest.Change change,
                                                               Cluster cluster, Map<String, ? extends PricingModel> prisingModelsBySKU) {
        BigDecimal result = DEFAULT_OWNING_COST;

        if (STRONG_GUARANTEE_CPU_SKU_BY_CLUSTER.containsKey(cluster)) {
            List<String> SKUList = STRONG_GUARANTEE_CPU_SKU_BY_CLUSTER.get(cluster);

            if (SKUList.size() == 2 && prisingModelsBySKU.keySet().containsAll(SKUList)) {
                PricingModel pricingModel1 = prisingModelsBySKU.get(SKUList.get(0));
                PricingModel pricingModel2 = prisingModelsBySKU.get(SKUList.get(1));

                result = ProviderOwningCostFormula.convertS(change, DiUnit.CORES, LOG)
                        .multiply(pricingModel1.add(pricingModel2, MATH_CONTEXT), MATH_CONTEXT);
            }
        }

        return result;
    }

    private BigDecimal calculateForStrongGuaranteeMemory(QuotaChangeRequest.Change change, Cluster cluster,
                                                         Map<String, ? extends PricingModel> prisingModelsBySKU) {
        BigDecimal result = DEFAULT_OWNING_COST;

        if (MEMORY_SKU_BY_CLUSTER.containsKey(cluster)) {
            String SKU = MEMORY_SKU_BY_CLUSTER.get(cluster);
            PricingModel pricingModel = prisingModelsBySKU.get(SKU);

            result = convert(change, DiUnit.GIBIBYTE, LOG)
                    .multiply(pricingModel.getPrice(), MATH_CONTEXT);
        }

        return result;
    }

    private BigDecimal calculateForRelaxedGuaranteeMemory(QuotaChangeRequest.Change change, Cluster cluster,
                                                          Map<String, ? extends PricingModel> prisingModelsBySKU) {
        BigDecimal result = DEFAULT_OWNING_COST;

        if (MEMORY_SKU_BY_CLUSTER.containsKey(cluster)) {
            String SKU = MEMORY_SKU_BY_CLUSTER.get(cluster);
            PricingModel pricingModel = prisingModelsBySKU.get(SKU);

            result = convert(change, DiUnit.GIBIBYTE, LOG)
                    .multiply(pricingModel.getPrice().multiply(RELAXED_MEMORY_TARIFF_COEFFICIENT, MATH_CONTEXT), MATH_CONTEXT);
        }

        return result;
    }

    private BigDecimal calculateForBurstGuaranteeMemory(QuotaChangeRequest.Change change, Cluster cluster,
                                                        Map<String, ? extends PricingModel> prisingModelsBySKU) {
        BigDecimal result = DEFAULT_OWNING_COST;

        if (MEMORY_SKU_BY_CLUSTER.containsKey(cluster)) {
            String SKU = MEMORY_SKU_BY_CLUSTER.get(cluster);
            PricingModel pricingModel = prisingModelsBySKU.get(SKU);

            result = convert(change, DiUnit.GIBIBYTE, LOG)
                    .multiply(pricingModel.getPrice().multiply(BURST_MEMORY_TARIFF_COEFFICIENT, MATH_CONTEXT), MATH_CONTEXT);
        }

        return result;
    }

    protected static BigDecimal calculateForHDD(QuotaChangeRequest.Change change, Cluster cluster,
                                       Map<String, ? extends PricingModel> prisingModelsBySKU) {
        BigDecimal result = DEFAULT_OWNING_COST;

        if (HDD_SKU_BY_CLUSTER.containsKey(cluster)) {
            String SKU = HDD_SKU_BY_CLUSTER.get(cluster);
            PricingModel pricingModel = prisingModelsBySKU.get(SKU);

            result = ProviderOwningCostFormula.convertS(change, DiUnit.GIBIBYTE, LOG)
                    .multiply(pricingModel.getPrice(), MATH_CONTEXT);
        }

        return result;
    }

    protected static BigDecimal calculateForSSD(QuotaChangeRequest.Change change, Cluster cluster,
                                       Map<String, ? extends PricingModel> prisingModelsBySKU) {
        BigDecimal result = DEFAULT_OWNING_COST;

        if (SSD_SKU_BY_CLUSTER.containsKey(cluster)) {
            String SKU = SSD_SKU_BY_CLUSTER.get(cluster);
            PricingModel pricingModel = prisingModelsBySKU.get(SKU);

            result = ProviderOwningCostFormula.convertS(change, DiUnit.GIBIBYTE, LOG)
                    .multiply(pricingModel.getPrice(), MATH_CONTEXT);
        }

        return result;
    }

    private BigDecimal calculateForGpu(QuotaChangeRequest.Change change, Cluster cluster,
                                       Map<String, ? extends PricingModel> prisingModelsBySKU) {
        BigDecimal result = DEFAULT_OWNING_COST;

        if (GPU_SKU_BY_CLUSTER.containsKey(cluster)) {
            List<String> SKUList = GPU_SKU_BY_CLUSTER.get(cluster);

            if (SKUList.size() == 2 && prisingModelsBySKU.keySet().containsAll(SKUList)) {
                PricingModel pricingModel1 = prisingModelsBySKU.get(SKUList.get(0));
                PricingModel pricingModel2 = prisingModelsBySKU.get(SKUList.get(1));

                result = convert(change, DiUnit.COUNT, LOG)
                        .multiply(pricingModel1.add(pricingModel2, MATH_CONTEXT), MATH_CONTEXT);
            }
        }

        return result;
    }

    private BigDecimal calculateForTableCell(QuotaChangeRequest.Change change, Cluster cluster,
                                             Map<String, ? extends PricingModel> prisingModelsBySKU) {
        BigDecimal result = DEFAULT_OWNING_COST;

        if (TABLE_CELL_SKU_BY_CLUSTER.containsKey(cluster)) {
            String SKU = TABLE_CELL_SKU_BY_CLUSTER.get(cluster);
            PricingModel pricingModel = prisingModelsBySKU.get(SKU);

            result = convert(change, DiUnit.COUNT, LOG)
                    .multiply(pricingModel.getPrice(), MATH_CONTEXT);
        }

        return result;
    }

    private BigDecimal calculateForStaticMemory(QuotaChangeRequest.Change change, Cluster cluster,
                                                Map<String, ? extends PricingModel> prisingModelsBySKU) {
        BigDecimal result = DEFAULT_OWNING_COST;

        if (TABLE_STATIC_MEMORY_SKU_BY_CLUSTER.containsKey(cluster)) {
            String SKU = TABLE_STATIC_MEMORY_SKU_BY_CLUSTER.get(cluster);
            PricingModel pricingModel = prisingModelsBySKU.get(SKU);

            result = convert(change, DiUnit.GIBIBYTE, LOG)
                    .multiply(pricingModel.getPrice(), MATH_CONTEXT);
        } else if (CLUSTERS_WITH_FIXED_TABLE_STATIC_MEMORY_PRICE.contains(cluster)) {
            result = convert(change, DiUnit.GIBIBYTE, LOG)
                    .multiply(FIXED_STATIC_MEMORY_PRICE, MATH_CONTEXT);
        }

        return result;
    }

    private BigDecimal calculateForRPCProxy(QuotaChangeRequest.Change change, Cluster cluster,
                                            Map<String, ? extends PricingModel> prisingModelsBySKU) {
        BigDecimal result = DEFAULT_OWNING_COST;

        if (RPC_PROXY_SKU_BY_CLUSTER.containsKey(cluster)) {
            Tuple3<String, String, String> SKUTuple = RPC_PROXY_SKU_BY_CLUSTER.get(cluster);

            if (prisingModelsBySKU.containsKey(SKUTuple.get1())
                    && prisingModelsBySKU.containsKey(SKUTuple.get2())
                    && prisingModelsBySKU.containsKey(SKUTuple.get3())) {
                PricingModel strongGuaranteeCpuPrice = prisingModelsBySKU.get(SKUTuple.get1());
                PricingModel cpuUsagePrice = prisingModelsBySKU.get(SKUTuple.get2());
                PricingModel memoryUsagePrice = prisingModelsBySKU.get(SKUTuple.get3());

                result = convert(change, DiUnit.COUNT, LOG)
                        .multiply(
                                strongGuaranteeCpuPrice.add(cpuUsagePrice, MATH_CONTEXT).multiply(RPC_PROXY_CPU_MULTIPLIER)
                                        .add(memoryUsagePrice.getPrice().multiply(RPC_PROXY_MEMORY_MULTIPLIER, MATH_CONTEXT), MATH_CONTEXT)
                                , MATH_CONTEXT);
            }
        }

        return result;
    }

    private BigDecimal calculateForClusterNetworkTraffic(QuotaChangeRequest.Change change, Cluster cluster,
                                                         Map<String, ? extends PricingModel> pricingBySKU) {
        BigDecimal result = DEFAULT_OWNING_COST;

        if (Cluster.MARKOV == cluster && TABLE_CELL_SKU_BY_CLUSTER.containsKey(cluster)) {
            String SKU = TABLE_CELL_SKU_BY_CLUSTER.get(cluster);
            PricingModel pricingModel = pricingBySKU.get(SKU);

            BigDecimal tablet_cells = convert(change, DiUnit.MIBPS, LOG)
                    .divide(CLUSTER_NETWORK_IN_TABLET_CELL, MATH_CONTEXT)
                    .setScale(0, RoundingMode.UP);

            result = tablet_cells.multiply(pricingModel.getPrice(), MATH_CONTEXT);
        }

        return result;
    }

    protected static Optional<Cluster> getClusterFromChange(QuotaChangeRequest.Change change, String ytClusterSegmentationKey) {
        return getClusterSegment(change, ytClusterSegmentationKey)
                .flatMap(segment -> Cluster.byKey(segment.getPublicKey()));

    }

    private enum Resource implements EnumUtils.StringKey {
        BURST_GUARANTEE_CPU("burst_guarantee_cpu"),
        CPU("cpu"),
        CPU_FLOW("cpu_flow"),
        GPU("gpu"),
        HDD("hdd"),
        MEM_BURST("mem_burst"),
        MEM_RELAXED("mem_relaxed"),
        MEM_STRONG("mem_strong"),
        RPC_PROXY("rpc_proxy"),
        SSD("ssd"),
        TABLET_CELL_BUNDLE("tablet_cell_bundle"),
        TABLET_STATIC_MEMORY("tablet_static_memory"),
        GPU_A100("gpu_a100"),
        CPU_DYNT("cpu_dynt"),
        MEM_DYNT("mem_dynt"),
        ;

        private static Map<String, Resource> resourceByKey;
        private final String key;

        Resource(String key) {
            this.key = key;
        }

        @Override
        public String getKey() {
            return key;
        }

        public static Optional<Resource> byKey(String key) {
            if (Resource.resourceByKey == null) {
                Resource.resourceByKey = ImmutableMap.copyOf(EnumUtils.prepareKeysMap(Resource.values()));
            }

            return Optional.ofNullable(Resource.resourceByKey.get(key));
        }
    }

    private enum Resource2020 implements EnumUtils.StringKey {
        CPU_CLICKHOUSE("cpu_clickhouse"),
        CLUSTER_NETWORK_TRAFFIC("cluster_network_traffic"),
        ;

        private static Map<String, Resource2020> resourceByKey;
        private final String key;

        Resource2020(String key) {
            this.key = key;
        }

        @Override
        public String getKey() {
            return key;
        }

        public static Optional<Resource2020> byKey(String key) {
            if (resourceByKey == null) {
                resourceByKey = ImmutableMap.copyOf(EnumUtils.prepareKeysMap(Resource2020.values()));
            }

            return Optional.ofNullable(resourceByKey.get(key));
        }
    }

    protected enum Cluster implements EnumUtils.StringKey {
        ARNOLD("arnold"),
        BOHR("bohr"),
        FREUD("freud"),
        HAHN("hahn"),
        HUME("hume"),
        LANDAU("landau"),
        LOCKE("locke"),
        MARKOV("markov"),
        PYTHIA("pythia"),
        SENECA_MAN("seneca-man"),
        SENECA_SAS("seneca-sas"),
        SENECA_VLA("seneca-vla"),
        SENECA_KLG("seneca-klg"),
        VANGA("vanga"),
        ZENO("zeno"),
        NASH("nash")
        ;

        private static Map<String, Cluster> clusterByKey;
        private final String key;

        Cluster(String key) {
            this.key = key;
        }

        @Override
        public String getKey() {
            return key;
        }

        public static Optional<Cluster> byKey(String key) {
            if (clusterByKey == null) {
                clusterByKey = ImmutableMap.copyOf(EnumUtils.prepareKeysMap(Cluster.values()));
            }

            return Optional.ofNullable(clusterByKey.get(key));
        }

        public String toClusterSKU(String SKU) {
            return SKU.replace(CLUSTER_PLACEHOLDER, this.key);
        }
    }

    private static Optional<Segment> getClusterSegment(final QuotaChangeRequest.Change change, String ytClusterSegmentationKey) {
        return SegmentUtils.getSegmentBySegmentationKey(change.getSegments(), ytClusterSegmentationKey);
    }
}
