package ru.yandex.qe.dispenser.ws.quota.request.owning_cost.pre_calculation_chain;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.EnumMap;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Random;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Sets;
import org.apache.log4j.helpers.LogLog;
import org.springframework.retry.backoff.FixedBackOffPolicy;
import org.springframework.retry.policy.SimpleRetryPolicy;
import org.springframework.retry.support.RetryTemplate;
import org.springframework.stereotype.Component;

import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.qe.dispenser.api.util.EnumUtils;
import ru.yandex.qe.dispenser.domain.QuotaChangeRequest;
import ru.yandex.qe.dispenser.domain.mds.MdsConfigApi;
import ru.yandex.qe.dispenser.domain.mds.MdsMarketingStorage;
import ru.yandex.qe.dispenser.domain.mds.MdsRawStorageByDc;
import ru.yandex.qe.dispenser.domain.mds.StorageType;
import ru.yandex.qe.dispenser.ws.bot.Provider;
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.ChangeOwningCostContext;
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.QuotaChangeRequestOwningCostContext;
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.pricing.QuotaChangeOwningCostTariffManager;

import static ru.yandex.qe.dispenser.ws.quota.request.owning_cost.formula.ProviderOwningCostFormula.MATH_CONTEXT;

/**
 * MDS pre calculation chain.
 * Prepare changes with MDS-storage raw resources in MDS, Sandbox, Nirvana and MDB.
 *
 * @author Ruslan Kadriev <aqru@yandex-team.ru>
 */
@Component
public class MDSPreCalculation implements PreCalculation {
    private final static Random RANDOM = new Random();
    private final static long BACK_OFF_PERIOD = 1000L;
    private static final int MAX_REPEATS = 1000;

    private static final Map<Provider, Set<Resource>> MDS_RESOURCES_BY_PROVIDER = new EnumMap<>(Map.of(
            Provider.MDS, EnumSet.of(Resource.MDS, Resource.AVATARS, Resource.S3_API),
            Provider.NIRVANA, EnumSet.of(Resource.NIRVANA_S3_STORAGE),
            Provider.MDB, EnumSet.of(Resource.MDB_S3_STORAGE),
            Provider.SANDBOX, EnumSet.of(Resource.SANDBOX_S3_STORAGE)
    ));

    private final MdsConfigApi mdsConfigApi;
    private final RetryTemplate retryTemplate;

    public MDSPreCalculation(MdsConfigApi mdsConfigApi) {
        this.mdsConfigApi = mdsConfigApi;
        this.retryTemplate = new RetryTemplate();

        final FixedBackOffPolicy fixedBackOffPolicy = new FixedBackOffPolicy();
        fixedBackOffPolicy.setBackOffPeriod(BACK_OFF_PERIOD);

        final SimpleRetryPolicy retryPolicy = new SimpleRetryPolicy();
        retryPolicy.setMaxAttempts(3);

        retryTemplate.setBackOffPolicy(fixedBackOffPolicy);
        retryTemplate.setRetryPolicy(retryPolicy);
    }

    @Override
    public QuotaChangeRequestOwningCostContext apply(QuotaChangeRequestOwningCostContext quotaChangeRequestOwningCostContext) {
        List<ChangeOwningCostContext> changes =
                quotaChangeRequestOwningCostContext.getChangeOwningCostContexts();
        Map<Boolean, List<ChangeOwningCostContext>> changesWithMdsStorageMap = changes.stream()
                .collect(Collectors.groupingBy(this::filterByChange));

        if (!changesWithMdsStorageMap.containsKey(true)) {
            return quotaChangeRequestOwningCostContext;
        }

        Map<Long, ChangeOwningCostContext> changeOwningCostContextById = new HashMap<>();
        List<ChangeOwningCostContext> changesWithId = changesWithMdsStorageMap.getOrDefault(true, List.of()).stream()
                .map(change -> {
                    ChangeOwningCostContext changeOwningCostContext = change;

                    // if it's request creation, then changes will be with same id = -1
                    if (changeOwningCostContextById.containsKey(changeOwningCostContext.getChange().getId())) {
                        long id;
                        int i = 0;
                        while (changeOwningCostContextById.containsKey(id = RANDOM.nextLong())) {
                            if (++i > MAX_REPEATS) {
                                throw new IllegalStateException("Infinity loop detected!");
                            }
                        }

                        changeOwningCostContext = changeOwningCostContext.copyBuilder()
                                .change(changeOwningCostContext.getChange().copyBuilder()
                                        .id(id)
                                        .build())
                                .build();
                    }

                    changeOwningCostContextById.put(changeOwningCostContext.getChange().getId(), change);

                    return changeOwningCostContext;
                })
                .collect(Collectors.toList());

        List<MdsMarketingStorage> marketingStore = changesWithId.stream()
                .map(this::toMdsMarketStorage)
                .collect(Collectors.toList());

        List<MdsRawStorageByDc> rawStorage = logIfNonConsistent(
                getRawStorage(marketingStore), marketingStore);
        List<Tuple2<Long, Long>> changeIdAmountTupleList = fromMdsRawStorageByDc(rawStorage);

        // return only pre calculated changes, for non pre calculated changes with MDS storage old owning cost will be used

        List<ChangeOwningCostContext> updatedChanges = Stream.concat(changeIdAmountTupleList.stream()
                                .filter(changeIdAmountTuple -> changeOwningCostContextById.containsKey(changeIdAmountTuple.get1()))
                                .map(changeIdAmountTuple -> {
                                    ChangeOwningCostContext changeOwningCostContext =
                                            changeOwningCostContextById.get(changeIdAmountTuple.get1());
                                    return changeOwningCostContext.copyBuilder()
                                            .change(
                                                    changeOwningCostContext.getChange().copyBuilder()
                                                            .amount(changeIdAmountTuple.get2())
                                                            .build()
                                            )
                                            .build();
                                }),
                        // add changes without MDS storage to result list
                        changesWithMdsStorageMap.getOrDefault(false, List.of()).stream())
                .collect(Collectors.toList());

        return quotaChangeRequestOwningCostContext.copyBuilder()
                .changeOwningCostContexts(updatedChanges)
                .build();
    }

    private List<MdsRawStorageByDc> logIfNonConsistent(List<MdsRawStorageByDc> rawStorage,
                                                       List<MdsMarketingStorage> marketingStore) {

        Set<Long> marketingStorageSet = marketingStore.stream().map(MdsMarketingStorage::getId)
                .collect(Collectors.toSet());
        Set<Long> rawStorageSet = rawStorage.stream().map(MdsRawStorageByDc::getId)
                .collect(Collectors.toSet());

        Sets.SetView<Long> differenceMarketingRaw = Sets.difference(marketingStorageSet, rawStorageSet);
        if (!differenceMarketingRaw.isEmpty()) {
            LogLog.warn("None empty diff 'marketing - raw': " + differenceMarketingRaw);
            LogLog.warn("Missed entity's: " + marketingStore.stream()
                    .filter(v -> differenceMarketingRaw.contains(v.getId()))
                    .collect(Collectors.toList()));
        }

        Sets.SetView<Long> differenceRawMarketing = Sets.difference(rawStorageSet, marketingStorageSet);
        if (!differenceRawMarketing.isEmpty()) {
            LogLog.warn("None empty diff 'raw - marketing': " + differenceRawMarketing);
            LogLog.warn("Unknown entity's: " + rawStorage.stream()
                    .filter(v -> differenceRawMarketing.contains(v.getId()))
                    .collect(Collectors.toList()));
        }

        return rawStorage;
    }

    private List<Tuple2<Long, Long>> fromMdsRawStorageByDc(List<MdsRawStorageByDc> rawStorages) {
        List<Tuple2<Long, Long>> result = new ArrayList<>();

        if (rawStorages != null) {
            result = rawStorages.stream()
                    .map(rawStorage -> Tuple2.tuple(rawStorage.getId(),
                            summingDc(rawStorage.getStorage())))
                    .collect(Collectors.toList());
        }

        return result;
    }

    private long summingDc(List<MdsRawStorageByDc.DcStorage> storage) {
        BigDecimal reduce = storage.stream()
                .map(dc -> new BigDecimal(dc.getAmount(), MATH_CONTEXT))
                .reduce(BigDecimal.ZERO, (a, b) -> a.add(b, MATH_CONTEXT));

        return reduce.longValueExact(); // fail on overflow
    }

    private MdsMarketingStorage toMdsMarketStorage(ChangeOwningCostContext changeOwningCostContext) {
        QuotaChangeRequest.Change change = changeOwningCostContext.getChange();
        Resource resource = Resource.byKey(change.getResource().getPublicKey()).orElseThrow();
        switch (resource) {
            case MDS:
                return toMDSMarketingStorage(changeOwningCostContext);
            case AVATARS:
                return toAvatarMarketingStorage(changeOwningCostContext);
            case S3_API:
                return toS3MarketingStorage(changeOwningCostContext);
            case NIRVANA_S3_STORAGE:
                return toNirvanaS3MarketingStorage(changeOwningCostContext);
            case MDB_S3_STORAGE:
                return toMDBS3MarketingStorage(changeOwningCostContext);
            case SANDBOX_S3_STORAGE:
                return toSandboxS3MarketingStorage(changeOwningCostContext);
            default: throw new IllegalStateException("Wrong resource!");
        }
    }

    private MdsMarketingStorage toMDSMarketingStorage(ChangeOwningCostContext changeOwningCostContext) {
        return fromMDSResource(changeOwningCostContext, StorageType.MDS);
    }

    private MdsMarketingStorage toAvatarMarketingStorage(ChangeOwningCostContext changeOwningCostContext) {
        return fromMDSResource(changeOwningCostContext, StorageType.AVATARS);
    }

    private MdsMarketingStorage toS3MarketingStorage(ChangeOwningCostContext changeOwningCostContext) {
        return fromMDSResource(changeOwningCostContext, StorageType.S3);
    }

    private MdsMarketingStorage toNirvanaS3MarketingStorage(ChangeOwningCostContext changeOwningCostContext) {
        QuotaChangeRequest.Change change = changeOwningCostContext.getChange();
        Integer nirvanaAbcServiceId = Objects.requireNonNull(change.getResource()
                .getService()
                .getAbcServiceId());
        return new MdsMarketingStorage(change.getId(), nirvanaAbcServiceId, StorageType.S3, change.getAmount());
    }

    private MdsMarketingStorage toMDBS3MarketingStorage(ChangeOwningCostContext changeOwningCostContext) {
        QuotaChangeRequest.Change change = changeOwningCostContext.getChange();
        Integer mdbAbcServiceId = Objects.requireNonNull(change.getResource()
                .getService()
                .getAbcServiceId());
        return new MdsMarketingStorage(change.getId(), mdbAbcServiceId, StorageType.S3, change.getAmount());
    }

    private MdsMarketingStorage toSandboxS3MarketingStorage(ChangeOwningCostContext changeOwningCostContext) {
        QuotaChangeRequest.Change change = changeOwningCostContext.getChange();
        Integer sandboxAbcServiceId = Objects.requireNonNull(change.getResource()
                .getService()
                .getAbcServiceId());
        return new MdsMarketingStorage(change.getId(), sandboxAbcServiceId, StorageType.S3, change.getAmount());
    }

    private MdsMarketingStorage fromMDSResource(ChangeOwningCostContext changeOwningCostContext, StorageType type) {
        QuotaChangeRequest.Change change = changeOwningCostContext.getChange();
        int projectId = changeOwningCostContext.getAbcServiceId();
        return new MdsMarketingStorage(change.getId(), projectId, type, change.getAmount());
    }

    private boolean filterByChange(ChangeOwningCostContext changeOwningCostContext) {
        if (QuotaChangeOwningCostTariffManager.CAMPAIGN_KEYS_2022
                .contains(changeOwningCostContext.getCampaign().getKey())) {
            // No old MDS pre-calculation for new campaign
            return false;
        }
        QuotaChangeRequest.Change change = changeOwningCostContext.getChange();
        Optional<Provider> providerO = Provider.fromService(change.getResource().getService());
        Optional<Resource> resourceO = Resource.byKey(change.getResource().getPublicKey());

        if (providerO.isPresent() && resourceO.isPresent()) {
            Provider provider = providerO.get();

            if (MDS_RESOURCES_BY_PROVIDER.containsKey(provider)) {
                Set<Resource> resources = MDS_RESOURCES_BY_PROVIDER.get(provider);

                return resources.contains(resourceO.get());
            }
        }

        return false;
    }

    private List<MdsRawStorageByDc> getRawStorage(final List<MdsMarketingStorage> storage) {
        if (storage.isEmpty()) {
            return List.of();
        }
        try {
            return retryTemplate.execute(ctx -> mdsConfigApi.calculateRawStorageByDc(storage));
        } catch (Exception e) {
            throw new RuntimeException("MDS storage convert error", e);
        }
    }

    private enum Resource implements EnumUtils.StringKey {
        MDS("mds"),
        AVATARS("avatars"),
        S3_API("s3-api"),
        NIRVANA_S3_STORAGE("s3-storage"),
        MDB_S3_STORAGE("storage-s3"),
        SANDBOX_S3_STORAGE("s3_storage"),
        ;

        private static Map<String, Resource> resourceByKey;
        private final String key;

        Resource(String key) {
            this.key = key;
        }

        @Override
        public String getKey() {
            return key;
        }

        public static Optional<Resource> byKey(String key) {
            if (Resource.resourceByKey == null) {
                Resource.resourceByKey = ImmutableMap.copyOf(EnumUtils.prepareKeysMap(Resource.values()));
            }

            return Optional.ofNullable(Resource.resourceByKey.get(key));
        }
    }
}
