package ru.yandex.qe.dispenser.ws.quota.request.owning_cost.pricing;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumMap;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.springframework.stereotype.Component;
import ru.yandex.qe.dispenser.ws.bot.Provider;
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.formula.ProviderOwningCostFormula;

/**
 * Manager with tariff's for quota change owning cost calculation.
 *
 * @author Ruslan Kadriev <aqru@yandex-team.ru>
 */
@Component
public class QuotaChangeOwningCostTariffManager {

    public static final Set<String> CAMPAIGN_KEYS_2022 = Set.of(
            ProviderOwningCostFormula.CampaignKey.AUG_2022_AGGREGATED.getKey(),
            ProviderOwningCostFormula.CampaignKey.AUG_2022_DRAFT.getKey()
    );

    final static Map<Provider, List<PricingModel>> skuByProvider;
    final static Map<Provider, List<PricingModel>> skuByProvider2022;

    final static Map<SKUProvider, Provider> providerMapper = new EnumMap<>(Map.ofEntries(
            Map.entry(SKUProvider.YP, Provider.YP),
            Map.entry(SKUProvider.YT, Provider.YT),
            Map.entry(SKUProvider.GENCFG, Provider.GEN_CFG),
            Map.entry(SKUProvider.SAAS, Provider.SAAS),
            Map.entry(SKUProvider.MANAGED_SERVICE_FOR_APACHE_KAFKA, Provider.MDB),
            Map.entry(SKUProvider.MANAGED_SERVICE_FOR_CLICKHOUSE, Provider.MDB),
            Map.entry(SKUProvider.MANAGED_SERVICE_FOR_ELASTICSEARCH, Provider.MDB),
            Map.entry(SKUProvider.MANAGED_SERVICE_FOR_MONGODB, Provider.MDB),
            Map.entry(SKUProvider.MANAGED_SERVICE_FOR_MYSQL, Provider.MDB),
            Map.entry(SKUProvider.MANAGED_SERVICE_FOR_POSTGRESQL, Provider.MDB),
            Map.entry(SKUProvider.MANAGED_SERVICE_FOR_REDIS, Provider.MDB),
            Map.entry(SKUProvider.DISTBUILD, Provider.DISTBUILD),
            Map.entry(SKUProvider.STRM, Provider.STRM),
            Map.entry(SKUProvider.SANDBOX, Provider.SANDBOX),
            Map.entry(SKUProvider.MDS_MEDIA_STORAGE, Provider.MDS_NEW),
            Map.entry(SKUProvider.S3, Provider.S3),
            Map.entry(SKUProvider.AVATARS, Provider.AVATARS),
            Map.entry(SKUProvider.YDB, Provider.YDB)
    ));

    // TODO remove this hardcode
    static {
        skuByProvider = new HashMap<>();
        skuByProvider2022 = new HashMap<>();
        Consumer<PricingModel> consumer = (model) -> skuByProvider.computeIfAbsent(byPricingModel(model),
                (k) -> new ArrayList<>()).add(model);
        Consumer<PricingModel> consumer2022 = (model) -> skuByProvider2022.computeIfAbsent(byPricingModel(model),
                (k) -> new ArrayList<>()).add(model);

        // YP
        SKURawText.toPricingModelStream()
                .filter(model -> model.getProvider().equals(SKUProvider.YP))
                .forEach(consumer);
        SkuRawText2022.toPricingModelStream()
                .filter(model -> model.getProvider().equals(SKUProvider.YP))
                .forEach(consumer2022);

        // YT
        SKURawText.toPricingModelStream()
                .filter(model -> model.getProvider().equals(SKUProvider.YT))
                .forEach(consumer);
        SkuRawText2022.toPricingModelStream()
                .filter(model -> model.getProvider().equals(SKUProvider.YT))
                .forEach(consumer2022);

        // GenCfg
        SKURawText.toPricingModelStream()
                .filter(model -> model.getProvider().equals(SKUProvider.GENCFG))
                .forEach(consumer);

        // SAAS
        SKURawText.toPricingModelStream()
                .filter(model -> model.getProvider().equals(SKUProvider.SAAS))
                .forEach(consumer);
        SkuRawText2022.toPricingModelStream()
                .filter(model -> model.getProvider().equals(SKUProvider.SAAS))
                .forEach(consumer2022);

        Set<SKUProvider> mdbSKUProviders = EnumSet.of(SKUProvider.MANAGED_SERVICE_FOR_APACHE_KAFKA,
                SKUProvider.MANAGED_SERVICE_FOR_CLICKHOUSE,
                SKUProvider.MANAGED_SERVICE_FOR_ELASTICSEARCH,
                SKUProvider.MANAGED_SERVICE_FOR_MONGODB,
                SKUProvider.MANAGED_SERVICE_FOR_MYSQL,
                SKUProvider.MANAGED_SERVICE_FOR_POSTGRESQL,
                SKUProvider.MANAGED_SERVICE_FOR_REDIS);

        // MDB
        SKURawText.toPricingModelStream()
                .filter(model -> mdbSKUProviders.contains(model.getProvider()))
                .forEach(consumer);
        SkuRawText2022.toPricingModelStream()
                .filter(model -> mdbSKUProviders.contains(model.getProvider()))
                .forEach(consumer2022);

        // Distbuild
        SKURawText.toPricingModelStream()
                .filter(model -> model.getProvider().equals(SKUProvider.DISTBUILD))
                .forEach(consumer);
        SkuRawText2022.toPricingModelStream()
                .filter(model -> model.getProvider().equals(SKUProvider.DISTBUILD))
                .forEach(consumer2022);

        // STRM
        SKURawText.toPricingModelStream()
                .filter(model -> model.getProvider().equals(SKUProvider.STRM))
                .forEach(consumer);
        SkuRawText2022.toPricingModelStream()
                .filter(model -> model.getProvider().equals(SKUProvider.STRM))
                .forEach(consumer2022);

        // Sandbox
        SKURawText.toPricingModelStream()
                .filter(model -> model.getProvider().equals(SKUProvider.SANDBOX))
                .forEach(consumer);
        SkuRawText2022.toPricingModelStream()
                .filter(model -> model.getProvider().equals(SKUProvider.SANDBOX))
                .forEach(consumer2022);

        // MDS
        SkuRawText2022.toPricingModelStream()
                .filter(model -> model.getProvider().equals(SKUProvider.MDS_MEDIA_STORAGE))
                .forEach(consumer2022);

        // S3
        SkuRawText2022.toPricingModelStream()
                .filter(model -> model.getProvider().equals(SKUProvider.S3))
                .forEach(consumer2022);

        // Avatars
        SkuRawText2022.toPricingModelStream()
                .filter(model -> model.getProvider().equals(SKUProvider.AVATARS))
                .forEach(consumer2022);

        // YDB
        SkuRawText2022.toPricingModelStream()
                .filter(model -> model.getProvider().equals(SKUProvider.YDB))
                .forEach(consumer2022);
    }

    private static Provider byPricingModel(PricingModel model) {
        return providerMapper.get(model.getProvider());
    }

    /**
     * Get prices for provider.
     * @param provider which pricing will be returned.
     * @return pricing for requested provider, or empty collection, if no such provider
     */
    @NotNull
    public Collection<? extends PricingModel> getByProvider(Provider provider) {
        return skuByProvider.getOrDefault(provider, Collections.emptyList());
    }

    /**
     * Get prices for provider for 2022 campaign.
     * @param provider which pricing will be returned.
     * @return pricing for requested provider, or empty collection, if no such provider
     */
    @NotNull
    public Collection<? extends PricingModel> getByProvider2022(Provider provider) {
        return skuByProvider2022.getOrDefault(provider, Collections.emptyList());
    }

    @NotNull
    public Collection<? extends PricingModel> getByProviderCampaign(Provider provider,
                                                                    @Nullable String campaignKey) {
        if (campaignKey != null && CAMPAIGN_KEYS_2022.contains(campaignKey)) {
            return getByProvider2022(provider);
        }
        return getByProvider(provider);
    }

    /**
     * Get prices for providers.
     * @param providers which pricing will be returned.
     * @return pricing for requested providers, or empty collection, if no such providers
     */
    @NotNull
    public Collection<? extends PricingModel> getByProviders(Collection<? extends Provider> providers) {
        return providers.stream()
                .flatMap(provider -> skuByProvider.getOrDefault(provider, Collections.emptyList()).stream())
                .collect(Collectors.toList());
    }

    /**
     * Get prices for providers for 2022 campaign.
     * @param providers which pricing will be returned.
     * @return pricing for requested providers, or empty collection, if no such providers
     */
    @NotNull
    public Collection<? extends PricingModel> getByProviders2022(Collection<? extends Provider> providers) {
        return providers.stream()
                .flatMap(provider -> skuByProvider2022.getOrDefault(provider, Collections.emptyList()).stream())
                .collect(Collectors.toList());
    }

    @NotNull
    public Collection<? extends PricingModel> getByProvidersCampaign(Collection<? extends Provider> providers,
                                                                     @Nullable String campaignKey) {
        if (campaignKey != null && CAMPAIGN_KEYS_2022.contains(campaignKey)) {
            return getByProviders2022(providers);
        }
        return getByProviders(providers);
    }

}
