package ru.yandex.qe.dispenser.ws.quota.request.report.summary;

import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import javax.inject.Inject;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.HashBasedTable;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;
import com.google.common.collect.Sets;
import com.google.common.collect.Table;
import com.opencsv.CSVWriter;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import ru.yandex.qe.dispenser.api.v1.DiResourcePreorderReasonType;
import ru.yandex.qe.dispenser.domain.BotCampaignGroup;
import ru.yandex.qe.dispenser.domain.CampaignForBot;
import ru.yandex.qe.dispenser.domain.Person;
import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.QuotaChangeRequest;
import ru.yandex.qe.dispenser.domain.Segment;
import ru.yandex.qe.dispenser.domain.Service;
import ru.yandex.qe.dispenser.domain.bot.BigOrder;
import ru.yandex.qe.dispenser.domain.bot.BigOrderConfig;
import ru.yandex.qe.dispenser.domain.bot.ConfigurationComponent;
import ru.yandex.qe.dispenser.domain.bot.ConfigurationWithComponents;
import ru.yandex.qe.dispenser.domain.dao.bot.BotPreOrder;
import ru.yandex.qe.dispenser.domain.dao.bot.CompleteBotConfiguration;
import ru.yandex.qe.dispenser.domain.dao.bot.MappedPreOrder;
import ru.yandex.qe.dispenser.domain.dao.bot.PreOrderBase;
import ru.yandex.qe.dispenser.domain.dao.bot.ServersRemainder;
import ru.yandex.qe.dispenser.domain.dao.bot.configuration.BotConfigurationDao;
import ru.yandex.qe.dispenser.domain.dao.bot.preorder.BotPreOrderDao;
import ru.yandex.qe.dispenser.domain.dao.bot.preorder.MappedPreOrderDao;
import ru.yandex.qe.dispenser.domain.dao.bot.preorder.change.BotPreOrderChangeDao;
import ru.yandex.qe.dispenser.domain.dao.bot.preorder.request.BotPreOrderRequestDao;
import ru.yandex.qe.dispenser.domain.dao.bot.preorder.request.PreOrderRequest;
import ru.yandex.qe.dispenser.domain.dao.bot.remainder.ServersRemainderDao;
import ru.yandex.qe.dispenser.domain.dao.project.ProjectReader;
import ru.yandex.qe.dispenser.domain.dao.quota.request.QuotaChangeRequestDao;
import ru.yandex.qe.dispenser.domain.dao.quota.request.QuotaChangeRequestFilterImpl;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;
import ru.yandex.qe.dispenser.domain.hierarchy.HierarchySupplier;
import ru.yandex.qe.dispenser.domain.hierarchy.Role;
import ru.yandex.qe.dispenser.ws.QuotaChangeRequestExportService;
import ru.yandex.qe.dispenser.ws.bot.BigOrderManager;
import ru.yandex.qe.dispenser.ws.bot.BotResourceType;
import ru.yandex.qe.dispenser.ws.bot.Provider;
import ru.yandex.qe.dispenser.ws.bot.mapping.Location;
import ru.yandex.qe.dispenser.ws.quota.request.ticket.QuotaChangeRequestTicketManager;

@Component
public class ResourcePreorderSummaryReport {
    private static final NumberFormat NUMBER_FORMAT = new DecimalFormat("#0.##");
    private static final Comparator<Map<Column, String>> REQUEST_ROW_COMPARATOR = Comparator
            .<Map<Column, String>, String>comparing(c -> c.get(Column.PROVIDER))
            .thenComparing(c -> c.get(Column.SEGMENT));

    private final QuotaChangeRequestDao quotaChangeRequestDao;
    private final HierarchySupplier hierarchySupplier;
    private final String goalsUrl;
    private final BotPreOrderChangeDao preOrderChangeDao;
    private final BotPreOrderRequestDao preOrderRequestDao;
    private final BotConfigurationDao configurationDao;
    private final MappedPreOrderDao mappedPreOrderDao;
    private final BigOrderManager bigOrderManager;
    private final BotPreOrderDao botPreOrderDao;
    private final ServersRemainderDao serversRemainderDao;

    @Inject
    public ResourcePreorderSummaryReport(final QuotaChangeRequestDao quotaChangeRequestDao,
                                         final HierarchySupplier hierarchySupplier,
                                         @Value("${goal.service.url}") final String goalsUrl,
                                         final BotPreOrderChangeDao preOrderChangeDao,
                                         final BotPreOrderRequestDao preOrderRequestDao,
                                         final BotConfigurationDao configurationDao,
                                         final MappedPreOrderDao mappedPreOrderDao,
                                         final BigOrderManager bigOrderManager,
                                         final BotPreOrderDao botPreOrderDao,
                                         final ServersRemainderDao serversRemainderDao) {
        this.quotaChangeRequestDao = quotaChangeRequestDao;
        this.hierarchySupplier = hierarchySupplier;
        this.goalsUrl = goalsUrl;
        this.preOrderChangeDao = preOrderChangeDao;
        this.preOrderRequestDao = preOrderRequestDao;
        this.configurationDao = configurationDao;
        this.mappedPreOrderDao = mappedPreOrderDao;
        this.bigOrderManager = bigOrderManager;
        this.botPreOrderDao = botPreOrderDao;
        this.serversRemainderDao = serversRemainderDao;
    }

    public enum Column {
        ID,
        ABC_SERVICE,
        BIG_ORDER,
        RESPONSIBLES,
        HEAD_1,
        HEAD_DEPARTMENT_1,
        HEAD_2,
        HEAD_DEPARTMENT_2,
        HEAD_3,
        HEAD_DEPARTMENT_3,
        PROVIDER,
        STATUS,
        SUBTICKET,
        SEGMENT,
        BOT_PREORDER,
        CAMPAIGN,
        JUSTIFICATION,
        GOAL_URL,
        GOAL_NAME,
        SUMMARY,

        TOTAL_PRICE("TOTAL PRICE"),
        SERVERS_TOTAL("Servers TOTAL"),
        STORAGES_TOTAL("Storages TOTAL"),
        STORAGES_PRICE("Storages PRICE"),

        UPGRADES_COUNT("Upgrades Quantity"),

        HW_SERVERS_PRICE("HW Servers PRICE"),
        HW_SERVERS("HW Servers"),
        HW_UPGRADES_PRICE("HW Upgrades PRICE"),

        D_SERVERS_SUM_COMPUTE("D.Servers SUM COMPUTE"),
        D_SERVERS_SUM_GPU("D.Servers SUM GPU"),

        PS_D_SERVERS_SUM_COMPUTE("PS: D.Servers SUM COMPUTE"),
        PS_D_SERVERS_SUM_GPU("PS: D.Servers SUM GPU"),

        VALUE_STREAM("Value Stream"),
        VS_MANAGER("VS Manager"),
        VS_LIDER("VS Lider"),
        VS_CAPACITY_PLANNER("VS Capacity Planner"),

        CPU_SUM("CPU: SUM"),
        RAM_SUM("RAM: SUM"),
        SSD_SUM("SSD: SUM"),
        HDD_SUM("HDD: SUM"),
        NVME_SUM("NVMe: SUM"),
        GPU_SUM("GPU: SUM"),

        C_DISTBUILD("C: DistBuild"),
        C_LOGBROKER("C: Logbroker"),
        C_LOGFELLER("C: Logfeller"),
        C_MDB("C: MDB"),
        C_MDS("C: MDS"),
        C_NIRVANA("C: Nirvana"),
        C_RTC_GENCFG("C: RTC (GenCfg)"),
        C_RTMR_MIRROR("C: RTMR Mirror"),
        C_RTMR_PROCESSING("C: RTMR Processing"),
        C_SQS("C: SQS"),
        C_SAAS("C: SaaS"),
        C_SANDBOX("C: Sandbox"),
        C_SOLOMON("C: Solomon"),
        C_YDB("C: YDB"),
        C_YP("C: YP"),
        C_YT("C: YT"),
        G_NIRVANA("G: Nirvana"),
        G_YP("G: YP"),
        G_YT("G: YT"),

        PS_C_DISTBUILD("PS: C.DistBuild"),
        PS_C_LOGBROKER("PS: C.Logbroker"),
        PS_C_LOGFELLER("PS: C.Logfeller"),
        PS_C_MDB("PS: C.MDB"),
        PS_C_MDS("PS: C.MDS"),
        PS_C_NIRVANA("PS: C.Nirvana"),
        PS_C_RTC_GENCFG("PS: C.RTC (GenCfg)"),
        PS_C_RTMR_MIRROR("PS: C.RTMR Mirror"),
        PS_C_RTMR_PROCESSING("PS: C.RTMR Processing"),
        PS_C_SQS("PS: C.SQS"),
        PS_C_SAAS("PS: C.SaaS"),
        PS_C_SANDBOX("PS: C.Sandbox"),
        PS_C_SOLOMON("PS: C.Solomon"),
        PS_C_YDB("PS: C.YDB"),
        PS_C_YP("PS: C.YP"),
        PS_C_YT("PS: C.YT"),
        PS_G_NIRVANA("PS: G.Nirvana"),
        PS_G_YP("PS: G.YP"),
        PS_G_YT("PS: G.YT"),

        PS_U_DISTBUILD("PS: U.DistBuild"),
        PS_U_LOGBROKER("PS: U.Logbroker"),
        PS_U_LOGFELLER("PS: U.Logfeller"),
        PS_U_MDB("PS: U.MDB"),
        PS_U_MDS("PS: U.MDS"),
        PS_U_NIRVANA("PS: U.Nirvana"),
        PS_U_RTC_GENCFG("PS: U.RTC (GenCfg)"),
        PS_U_RTMR_MIRROR("PS: U.RTMR Mirror"),
        PS_U_RTMR_PROCESSING("PS: U.RTMR Processing"),
        PS_U_SQS("PS: U.SQS"),
        PS_U_SAAS("PS: U.SaaS"),
        PS_U_SANDBOX("PS: U.Sandbox"),
        PS_U_SOLOMON("PS: U.Solomon"),
        PS_U_YDB("PS: U.YDB"),
        PS_U_YP("PS: U.YP"),
        PS_U_YT("PS: U.YT"),

        TOTAL_PRICE_GROWTH("TOTAL PRICE GROWTH"),
        TOTAL_PRICE_GOALS("TOTAL PRICE GOALS"),
        SERVERS_TOTAL_GROWTH("Servers TOTAL GROWTH"),
        SERVERS_TOTAL_GOALS("Servers TOTAL GOALS"),

        VLA,
        SAS,
        MAN,
        MYT,
        IVA,
        ETC,

        DELIVERY_DATE,

        VLA_COMPUTE,
        VLA_COMPUTE_PS,
        VLA_GPU,
        VLA_GPU_PS,
        VLA_UPGRADE_PS,

        SAS_COMPUTE,
        SAS_COMPUTE_PS,
        SAS_GPU,
        SAS_GPU_PS,
        SAS_UPGRADE_PS,

        MAN_COMPUTE,
        MAN_COMPUTE_PS,
        MAN_GPU,
        MAN_GPU_PS,
        MAN_UPGRADE_PS,

        MYT_COMPUTE,
        MYT_COMPUTE_PS,
        MYT_GPU,
        MYT_GPU_PS,
        MYT_UPGRADE_PS,

        IVA_COMPUTE,
        IVA_COMPUTE_PS,
        IVA_GPU,
        IVA_GPU_PS,
        IVA_UPGRADE_PS,

        ETC_COMPUTE,
        ETC_COMPUTE_PS,
        ETC_GPU,
        ETC_GPU_PS,
        ETC_UPGRADE_PS,

        ;

        private final String title;

        Column(final String title) {
            this.title = title;
        }

        Column() {
            this(null);
        }

        public String getTitle() {
            return title != null ? title : name();
        }
    }

    private static final Map<Location, Column> COLUMN_BY_LOCATION = ImmutableMap.of(
            Location.MAN, Column.MAN,
            Location.SAS, Column.SAS,
            Location.VLA, Column.VLA,
            Location.MYT, Column.MYT,
            Location.IVA, Column.IVA
    );

    private static final Map<Location, Column> COMPUTE_COLUMN_BY_LOCATION = ImmutableMap.of(
            Location.MAN, Column.MAN_COMPUTE,
            Location.SAS, Column.SAS_COMPUTE,
            Location.VLA, Column.VLA_COMPUTE,
            Location.MYT, Column.MYT_COMPUTE,
            Location.IVA, Column.IVA_COMPUTE
    );

    private static final Map<Location, Column> GPU_COLUMN_BY_LOCATION = ImmutableMap.of(
            Location.MAN, Column.MAN_GPU,
            Location.SAS, Column.SAS_GPU,
            Location.VLA, Column.VLA_GPU,
            Location.MYT, Column.MYT_GPU,
            Location.IVA, Column.IVA_GPU
    );

    private static final Map<Location, Column> COMPUTE_PS_COLUMN_BY_LOCATION = ImmutableMap.of(
            Location.MAN, Column.MAN_COMPUTE_PS,
            Location.SAS, Column.SAS_COMPUTE_PS,
            Location.VLA, Column.VLA_COMPUTE_PS,
            Location.MYT, Column.MYT_COMPUTE_PS,
            Location.IVA, Column.IVA_COMPUTE_PS
    );

    private static final Map<Location, Column> GPU_PS_COLUMN_BY_LOCATION = ImmutableMap.of(
            Location.MAN, Column.MAN_GPU_PS,
            Location.SAS, Column.SAS_GPU_PS,
            Location.VLA, Column.VLA_GPU_PS,
            Location.MYT, Column.MYT_GPU_PS,
            Location.IVA, Column.IVA_GPU_PS
    );

    private static final Map<Location, Column> UPGRADE_PS_COLUMN_BY_LOCATION = ImmutableMap.of(
            Location.MAN, Column.MAN_UPGRADE_PS,
            Location.SAS, Column.SAS_UPGRADE_PS,
            Location.VLA, Column.VLA_UPGRADE_PS,
            Location.MYT, Column.MYT_UPGRADE_PS,
            Location.IVA, Column.IVA_UPGRADE_PS
    );

    private static final Map<BotResourceType, Column> COLUMN_BY_RESOURCE_TYPE = ImmutableMap.<BotResourceType, Column>builder()
            .put(BotResourceType.CPU, Column.CPU_SUM)
            .put(BotResourceType.RAM, Column.RAM_SUM)
            .put(BotResourceType.SSD, Column.SSD_SUM)
            .put(BotResourceType.HDD, Column.HDD_SUM)
            .put(BotResourceType.NVME, Column.NVME_SUM)
            .put(BotResourceType.GPU, Column.GPU_SUM)
            .build();

    private static final Map<BotResourceType, Long> RESOURCE_TYPE_BASE_UNITS = ImmutableMap.<BotResourceType, Long>builder()
            .put(BotResourceType.CPU, 1_000L)
            .put(BotResourceType.RAM, BotResourceType.RAM_BYTES_IN_GB)
            .put(BotResourceType.SSD, BotResourceType.BYTES_IN_GIGABYTE)
            .put(BotResourceType.HDD, BotResourceType.BYTES_IN_GIGABYTE)
            .put(BotResourceType.NVME, BotResourceType.BYTES_IN_GIGABYTE)
            .put(BotResourceType.GPU, 1_000L)
            .build();

    private static final Map<Provider, Column> COLUMN_BY_PROVIDER_COMPUTE = ImmutableMap.<Provider, Column>builder()
            .put(Provider.DISTBUILD, Column.C_DISTBUILD)
            .put(Provider.LOGBROKER, Column.C_LOGBROKER)
            .put(Provider.LOGFELLER, Column.C_LOGFELLER)
            .put(Provider.MDB, Column.C_MDB)
            .put(Provider.MDS, Column.C_MDS)
            .put(Provider.NIRVANA, Column.C_NIRVANA)
            .put(Provider.GEN_CFG, Column.C_RTC_GENCFG)
            .put(Provider.RTMR_MIRROR, Column.C_RTMR_MIRROR)
            .put(Provider.RTMR_PROCESSING, Column.C_RTMR_PROCESSING)
            .put(Provider.SQS, Column.C_SQS)
            .put(Provider.SAAS, Column.C_SAAS)
            .put(Provider.SANDBOX, Column.C_SANDBOX)
            .put(Provider.SOLOMON, Column.C_SOLOMON)
            .put(Provider.YDB, Column.C_YDB)
            .put(Provider.YP, Column.C_YP)
            .put(Provider.YT, Column.C_YT)
            .build();

    private static final Map<Provider, Column> COLUMN_PRICE_BY_PROVIDER_COMPUTE = ImmutableMap.<Provider, Column>builder()
            .put(Provider.DISTBUILD, Column.PS_C_DISTBUILD)
            .put(Provider.LOGBROKER, Column.PS_C_LOGBROKER)
            .put(Provider.LOGFELLER, Column.PS_C_LOGFELLER)
            .put(Provider.MDB, Column.PS_C_MDB)
            .put(Provider.MDS, Column.PS_C_MDS)
            .put(Provider.NIRVANA, Column.PS_C_NIRVANA)
            .put(Provider.GEN_CFG, Column.PS_C_RTC_GENCFG)
            .put(Provider.RTMR_MIRROR, Column.PS_C_RTMR_MIRROR)
            .put(Provider.RTMR_PROCESSING, Column.PS_C_RTMR_PROCESSING)
            .put(Provider.SQS, Column.PS_C_SQS)
            .put(Provider.SAAS, Column.PS_C_SAAS)
            .put(Provider.SANDBOX, Column.PS_C_SANDBOX)
            .put(Provider.SOLOMON, Column.PS_C_SOLOMON)
            .put(Provider.YDB, Column.PS_C_YDB)
            .put(Provider.YP, Column.PS_C_YP)
            .put(Provider.YT, Column.PS_C_YT)
            .build();

    private static final Map<Provider, Column> COLUMN_PRICE_BY_PROVIDER_UPGRADE = ImmutableMap.<Provider, Column>builder()
            .put(Provider.DISTBUILD, Column.PS_U_DISTBUILD)
            .put(Provider.LOGBROKER, Column.PS_U_LOGBROKER)
            .put(Provider.LOGFELLER, Column.PS_U_LOGFELLER)
            .put(Provider.MDB, Column.PS_U_MDB)
            .put(Provider.MDS, Column.PS_U_MDS)
            .put(Provider.NIRVANA, Column.PS_U_NIRVANA)
            .put(Provider.GEN_CFG, Column.PS_U_RTC_GENCFG)
            .put(Provider.RTMR_MIRROR, Column.PS_U_RTMR_MIRROR)
            .put(Provider.RTMR_PROCESSING, Column.PS_U_RTMR_PROCESSING)
            .put(Provider.SQS, Column.PS_U_SQS)
            .put(Provider.SAAS, Column.PS_U_SAAS)
            .put(Provider.SANDBOX, Column.PS_U_SANDBOX)
            .put(Provider.SOLOMON, Column.PS_U_SOLOMON)
            .put(Provider.YDB, Column.PS_U_YDB)
            .put(Provider.YP, Column.PS_U_YP)
            .put(Provider.YT, Column.PS_U_YT)
            .build();


    private static final Map<Provider, Column> COLUMN_BY_PROVIDER_GPU = ImmutableMap.<Provider, Column>builder()
            .put(Provider.NIRVANA, Column.G_NIRVANA)
            .put(Provider.YP, Column.G_YP)
            .put(Provider.YT, Column.G_YT)
            .build();

    private static final Map<Provider, Column> COLUMN_PRICE_BY_PROVIDER_GPU = ImmutableMap.<Provider, Column>builder()
            .put(Provider.NIRVANA, Column.PS_G_NIRVANA)
            .put(Provider.YP, Column.PS_G_YP)
            .put(Provider.YT, Column.PS_G_YT)
            .build();

    public void writeReport(final CSVWriter csvWriter, final ResourcePreorderSummaryFilter filter,
                            final BotCampaignGroup currentCampaignGroup) {
        final Set<Long> bigOrderIds = filter.getBigOrderIds();
        final Set<Long> deliveryOrderIds = filter.getDeliveryOrderIds();
        final QuotaChangeRequestFilterImpl requestFilter = new QuotaChangeRequestFilterImpl();
        requestFilter.setOrderIds(bigOrderIds);

        final List<QuotaChangeRequest> requests = quotaChangeRequestDao.read(requestFilter);
        final Table<Long, Long, Double> preOrderIdsByChangeId = preOrderChangeDao.getPreOrderChangesByBigOrderId(deliveryOrderIds);
        final Collection<PreOrderRequest> preOrderRequests = preOrderRequestDao.getRequestPreordersByBigOrderIds(deliveryOrderIds);
        final Map<Long, CompleteBotConfiguration> configurationById = configurationDao.getConfigurationsInBigOrders(deliveryOrderIds);
        final Map<Long, MappedPreOrder> preOrderById = mappedPreOrderDao.getPreOrdersByBigOrderIds(deliveryOrderIds);
        final Set<BigOrder> bigOrders = bigOrderManager.getAll();

        final Map<Long, BigOrderConfig> bigOrderConfigById = new HashMap<>();
        final Map<Long, BigOrder> bigOrderById = new HashMap<>();
        for (final BigOrder bigOrder : bigOrders) {
            bigOrderById.put(bigOrder.getId(), bigOrder);
            for (final BigOrderConfig config : bigOrder.getConfigs()) {
                bigOrderConfigById.put(config.getId(), config);
            }
        }
        final Map<Long, Map<BotResourceType, Long>> resourceAmountPerServerByPreOrderId = new HashMap<>();
        for (final MappedPreOrder preOrder : preOrderById.values()) {
            resourceAmountPerServerByPreOrderId.put(preOrder.getId(), getPreOrderResources(preOrder, configurationById));
        }

        final Table<Long, Long, PreOrderRequest.Value> preOrderRequestsByIdAndOrderId = HashBasedTable.create();
        for (final PreOrderRequest preOrderRequest : preOrderRequests) {
            preOrderRequestsByIdAndOrderId.put(
                    preOrderRequest.getKey().getRequestId(),
                    preOrderRequest.getKey().getPreOrderId(),
                    preOrderRequest.getValue()
            );
        }

        final List<Column> columns = Arrays.asList(Column.values());

        final List<Map<Column, String>> rows = new ArrayList<>();
        final List<Map<Column, String>> requestRows = new ArrayList<>();

        rows.add(getHeaderRow(columns));

        for (final QuotaChangeRequest request : requests) {
            final Multimap<Long, ChangeServers> changeServers = ArrayListMultimap.create();

            final Map<Long, PreOrderRequest.Value> preOrderRequestsByOrderId = preOrderRequestsByIdAndOrderId.row(request.getId());
            final Map<Long, Double> sumChangeProportionByOrderId = new HashMap<>();
            for (final QuotaChangeRequest.Change change : request.getChanges()) {
                final Map<Long, Double> changeProportionByOrderId = preOrderIdsByChangeId.row(change.getId());
                for (final Long orderId : changeProportionByOrderId.keySet()) {
                    final PreOrderRequest.Value orderRequest = preOrderRequestsByOrderId.get(orderId);
                    if (orderRequest == null) {
                        continue;
                    }
                    final Double proportion = changeProportionByOrderId.get(orderId);
                    sumChangeProportionByOrderId.put(orderId, sumChangeProportionByOrderId.getOrDefault(orderId, 0.) + proportion);
                }
            }

            for (final QuotaChangeRequest.Change change : request.getChanges()) {
                final Map<Long, Double> changeProportionByOrderId = preOrderIdsByChangeId.row(change.getId());
                for (final Long orderId : changeProportionByOrderId.keySet()) {
                    final MappedPreOrder mappedPreOrder = preOrderById.get(orderId);
                    if (mappedPreOrder == null) {
                        continue;
                    }

                    final BigOrderConfig config = bigOrderConfigById.get(mappedPreOrder.getBigOrderConfigId());
                    final Location location = config == null ? null : Location.fromKey(config.getLocationKey());

                    final PreOrderRequest.Value orderRequest = preOrderRequestsByOrderId.get(orderId);
                    if (orderRequest == null) {
                        continue;
                    }
                    final Double proportion = changeProportionByOrderId.get(orderId);
                    final Double sum = sumChangeProportionByOrderId.get(orderId);
                    if (sum == null) {
                        continue;
                    }
                    final double requestProportion = sum == 0 ? 0 : proportion / sum;
                    final Long preOrderBigOrderId = mappedPreOrder.getBigOrderId();
                    final String deliveryDate = bigOrderById.get(preOrderBigOrderId).getDate().format(DateTimeFormatter.ISO_LOCAL_DATE);
                    changeServers.put(change.getId(), new ChangeServers(
                            orderRequest.getServerQuantity() * requestProportion,
                            orderRequest.getCost() * requestProportion,
                            resourceAmountPerServerByPreOrderId.getOrDefault(orderId, Collections.emptyMap()), location,
                            deliveryDate, config.isUpgrade()));
                }
            }

            requestRows.addAll(getRowsForRequest(request, changeServers, deliveryOrderIds));
        }

        requestRows.sort(REQUEST_ROW_COMPARATOR);
        rows.addAll(requestRows);

        final Set<ServersRemainder> remainders = serversRemainderDao.getByCampaignGroup(currentCampaignGroup.getId());
        rows.addAll(getRowsForRemainders(remainders, currentCampaignGroup));

        rows.addAll(getRowsFromBot(deliveryOrderIds, bigOrderConfigById, bigOrderById, configurationById));

        final String[] params = new String[columns.size()];
        for (final Map<Column, String> row : rows) {
            int i = 0;
            for (final Column column : columns) {
                final String value = row.get(column);
                params[i] = value == null ? "" : value;
                i += 1;
            }

            csvWriter.writeNext(params);
        }
    }

    private List<Map<Column, String>> getRowsForRequest(final QuotaChangeRequest request,
                                                        final Multimap<Long, ChangeServers> changeServers,
                                                        final Set<Long> allowedBigOrderIds) {
        final Map<Column, String> commonPart = new HashMap<>();
        commonPart.put(Column.STATUS, request.getStatus().name());
        commonPart.put(Column.SUBTICKET, request.getTrackerIssueKey());
        final String campaignKey = request.getCampaign() != null ? request.getCampaign().getKey() : "";
        commonPart.put(Column.CAMPAIGN, campaignKey);
        final String justification = request.getResourcePreorderReasonType() != null
                ? request.getResourcePreorderReasonType().name() : "";
        commonPart.put(Column.JUSTIFICATION, justification);
        final String goalUrl = request.getGoal() != null ? (goalsUrl + "/filter?goal=" + request.getGoal().getId()) : "";
        commonPart.put(Column.GOAL_URL, goalUrl);
        final String goalName = request.getGoal() != null ? request.getGoal().getName() : "";
        commonPart.put(Column.GOAL_NAME, goalName);
        commonPart.put(Column.SUMMARY, request.getSummary() != null ? request.getSummary() : "");

        final Hierarchy hierarchy = hierarchySupplier.get();
        commonPart.put(Column.RESPONSIBLES, QuotaChangeRequestTicketManager.getAssignee(request, hierarchy));

        commonPart.putAll(getProjectColumns(request.getProject(), hierarchy.getProjectReader()));

        final Map<QuotaChangeRequestExportService.RequestPartKey, List<QuotaChangeRequest.Change>> parts = request.getChanges().stream()
                .collect(Collectors.groupingBy(c -> QuotaChangeRequestExportService.RequestPartKey.of(request.getId(), c)));

        final List<Map<Column, String>> result = new ArrayList<>();
        for (final QuotaChangeRequestExportService.RequestPartKey partKey : parts.keySet()) {
            final Service service = partKey.getService();
            final QuotaChangeRequest.BigOrder order = partKey.getOrder();
            final String segments = partKey.getSegments()
                    .stream()
                    .sorted(Comparator.comparing(s -> s.getSegmentation().getId()))
                    .map(Segment::getPublicKey)
                    .collect(Collectors.joining("-"));

            final Map<String, Map<QuotaChangeRequest.Change, List<ChangeServers>>> changeAndServersByDeliveryDate = new HashMap<>();

            for (QuotaChangeRequest.Change change : parts.get(partKey)) {
                final Collection<ChangeServers> servers = changeServers.get(change.getId());
                if (servers.isEmpty()) {
                    final QuotaChangeRequest.BigOrder bigOrder = change.getBigOrder();
                    if (allowedBigOrderIds.contains(bigOrder.getId())) {
                        changeAndServersByDeliveryDate.computeIfAbsent(bigOrder.getDate().format(DateTimeFormatter.ISO_LOCAL_DATE), s -> new HashMap<>())
                                .put(change, Collections.emptyList());
                    }
                    continue;
                }

                final Map<String, List<ChangeServers>> serversByDeliveryDate = servers.stream()
                        .collect(Collectors.groupingBy(ChangeServers::getDeliveryDate, Collectors.toList()));

                for (Map.Entry<String, List<ChangeServers>> serversByDate : serversByDeliveryDate.entrySet()) {
                    final String deliveryDate = serversByDate.getKey();
                    final List<ChangeServers> dateServers = serversByDate.getValue();

                    final Map<QuotaChangeRequest.Change, List<ChangeServers>> deliveryDateChangeServers =
                            changeAndServersByDeliveryDate.computeIfAbsent(deliveryDate, s -> new HashMap<>());

                    deliveryDateChangeServers.put(change, dateServers);
                }
            }

            for (Map.Entry<String, Map<QuotaChangeRequest.Change, List<ChangeServers>>> entry : changeAndServersByDeliveryDate.entrySet()) {
                final String deliveryDate = entry.getKey();
                final Map<QuotaChangeRequest.Change, List<ChangeServers>> serversByChange = entry.getValue();

                final Map<Column, String> row = new HashMap<>();

                row.put(Column.ID, request.getId() + "-"
                        + service.getKey() + "-"
                        + (order == null ? "" : (order.getDate().format(DateTimeFormatter.ISO_LOCAL_DATE) + "-"))
                        + ("by-" + deliveryDate + "-")
                        + segments);
                row.put(Column.BIG_ORDER, order != null ? order.getDate().format(DateTimeFormatter.ISO_LOCAL_DATE) : null);
                row.put(Column.PROVIDER, service.getName());
                row.put(Column.SEGMENT, segments);
                row.put(Column.DELIVERY_DATE, deliveryDate);

                double totalPrice = 0.0;
                double totalServers = 0.0;
                double totalComputePrice = 0.0;
                double totalComputeServers = 0.0;
                double totalGpuPrice = 0.0;
                double totalGpuServers = 0.0;

                double totalUpgradeCount = 0.0;
                double totalUpgradePrice = 0.0;

                final Map<Location, Double> serversByLocation = new HashMap<>();
                final Map<Location, Double> computeServersByLocation = new HashMap<>();
                final Map<Location, Double> gpuServersByLocation = new HashMap<>();
                final Map<Location, Double> computePriceByLocation = new HashMap<>();
                final Map<Location, Double> gpuPriceByLocation = new HashMap<>();
                final Map<Location, Double> upgradePriceByLocation = new HashMap<>();
                final Map<BotResourceType, Double> resourceByType = new HashMap<>();

                final Map<Column, Double> serversByProvider = new HashMap<>();
                final Map<Column, Double> priceByProvider = new HashMap<>();

                for (Map.Entry<QuotaChangeRequest.Change, List<ChangeServers>> changeAndServers : serversByChange.entrySet()) {
                    final QuotaChangeRequest.Change change = changeAndServers.getKey();
                    final List<ChangeServers> servers = changeAndServers.getValue();

                    for (final ChangeServers server : servers) {

                        final Optional<Provider> provider = Provider.fromService(change.getResource().getService());

                        totalPrice += server.getPrice();
                        final Location location = server.getLocation();

                        if (server.isUpgrade()) {
                            totalUpgradeCount += server.getServers();
                            totalUpgradePrice += server.getPrice();
                            upgradePriceByLocation.put(location, upgradePriceByLocation.getOrDefault(location, 0.) + server.getPrice());
                            if (provider.isPresent()) {
                                final Column upgradePriceColumn = COLUMN_PRICE_BY_PROVIDER_UPGRADE.get(provider.get());
                                if (upgradePriceColumn != null) {
                                    priceByProvider.put(upgradePriceColumn, priceByProvider.getOrDefault(upgradePriceColumn, 0.) + server.getPrice());
                                }
                            }
                        } else {
                            totalServers += server.getServers();
                            if (server.isGpu()) {
                                totalGpuPrice += server.getPrice();
                                totalGpuServers += server.getServers();
                                gpuServersByLocation.put(location, gpuServersByLocation.getOrDefault(location, 0.)
                                        + server.getServers());
                                gpuPriceByLocation.put(location, gpuPriceByLocation.getOrDefault(location, 0.)
                                        + server.getPrice());
                            } else {
                                totalComputePrice += server.getPrice();
                                totalComputeServers += server.getServers();
                                computeServersByLocation.put(location, computeServersByLocation.getOrDefault(location, 0.)
                                        + server.getServers());
                                computePriceByLocation.put(location, computePriceByLocation.getOrDefault(location, 0.)
                                        + server.getPrice());
                            }
                            serversByLocation.put(location, serversByLocation.getOrDefault(location, 0.) + server.getServers());

                            final Map<BotResourceType, Double> amountByType = server.getAmountByType();
                            for (final BotResourceType botResourceType : amountByType.keySet()) {
                                final Double resourceAmount = amountByType.get(botResourceType);
                                resourceByType.put(botResourceType, resourceByType.getOrDefault(botResourceType, 0.) + resourceAmount);
                            }
                            if (provider.isPresent()) {
                                final Column serversColumn = (server.isGpu() ? COLUMN_BY_PROVIDER_GPU : COLUMN_BY_PROVIDER_COMPUTE).get(provider.get());
                                if (serversColumn != null) {
                                    serversByProvider.put(serversColumn, serversByProvider.getOrDefault(serversColumn, 0.) + server.getServers());
                                }
                                final Column priceColumn = (server.isGpu() ? COLUMN_PRICE_BY_PROVIDER_GPU : COLUMN_PRICE_BY_PROVIDER_COMPUTE).get(provider.get());
                                if (priceColumn != null) {
                                    priceByProvider.put(priceColumn, priceByProvider.getOrDefault(serversColumn, 0.) + server.getPrice());
                                }
                            }
                        }
                    }
                }

                row.put(Column.TOTAL_PRICE, NUMBER_FORMAT.format(totalPrice));
                row.put(Column.SERVERS_TOTAL, NUMBER_FORMAT.format(totalServers));
                row.put(Column.UPGRADES_COUNT, NUMBER_FORMAT.format(totalUpgradeCount));
                row.put(Column.HW_UPGRADES_PRICE, NUMBER_FORMAT.format(totalUpgradePrice));

                if (request.getResourcePreorderReasonType() == DiResourcePreorderReasonType.GOAL) {
                    row.put(Column.TOTAL_PRICE_GOALS, NUMBER_FORMAT.format(totalPrice));
                    row.put(Column.SERVERS_TOTAL_GOALS, NUMBER_FORMAT.format(totalServers));
                } else if (request.getResourcePreorderReasonType() == DiResourcePreorderReasonType.GROWTH) {
                    row.put(Column.TOTAL_PRICE_GROWTH, NUMBER_FORMAT.format(totalPrice));
                    row.put(Column.SERVERS_TOTAL_GROWTH, NUMBER_FORMAT.format(totalServers));
                }

                row.put(Column.D_SERVERS_SUM_COMPUTE, NUMBER_FORMAT.format(totalComputeServers));
                row.put(Column.PS_D_SERVERS_SUM_COMPUTE, NUMBER_FORMAT.format(totalComputePrice));
                row.put(Column.D_SERVERS_SUM_GPU, NUMBER_FORMAT.format(totalGpuServers));
                row.put(Column.PS_D_SERVERS_SUM_GPU, NUMBER_FORMAT.format(totalGpuPrice));

                fillLocations(row, serversByLocation, COLUMN_BY_LOCATION, Column.ETC);
                fillLocations(row, computeServersByLocation, COMPUTE_COLUMN_BY_LOCATION, Column.ETC_COMPUTE);
                fillLocations(row, computePriceByLocation, COMPUTE_PS_COLUMN_BY_LOCATION, Column.ETC_COMPUTE_PS);
                fillLocations(row, gpuServersByLocation, GPU_COLUMN_BY_LOCATION, Column.ETC_GPU);
                fillLocations(row, gpuPriceByLocation, GPU_PS_COLUMN_BY_LOCATION, Column.ETC_GPU_PS);
                fillLocations(row, upgradePriceByLocation, UPGRADE_PS_COLUMN_BY_LOCATION, Column.ETC_UPGRADE_PS);

                for (final BotResourceType botResourceType : resourceByType.keySet()) {
                    final Double amount = resourceByType.get(botResourceType);
                    final Column column = COLUMN_BY_RESOURCE_TYPE.get(botResourceType);
                    if (column != null) {
                        row.put(column, NUMBER_FORMAT.format(amount));
                    }
                }

                for (final Column column : serversByProvider.keySet()) {
                    row.put(column, NUMBER_FORMAT.format(serversByProvider.get(column)));
                }

                for (final Column column : priceByProvider.keySet()) {
                    row.put(column, NUMBER_FORMAT.format(priceByProvider.get(column)));
                }

                row.putAll(commonPart);
                result.add(row);
            }
        }

        return result;
    }

    private static void fillLocations(Map<Column, String> row,
                                      Map<Location, ? extends Number> values,
                                      Map<Location, Column> columnByLocation,
                                      Column defaultLocationColumn) {
        values.forEach((location, count) -> {
            final Column column = columnByLocation.getOrDefault(location, defaultLocationColumn);
            row.put(column, NUMBER_FORMAT.format(count));
        });
    }

    public List<Map<Column, String>> getRowsFromBot(final Set<Long> bigOrderIds,
                                                    final Map<Long, BigOrderConfig> bigOrderConfigById,
                                                    final Map<Long, BigOrder> bigOrderById,
                                                    final Map<Long, CompleteBotConfiguration> configurationById
    ) {
        final Collection<BotPreOrder> preOrders = botPreOrderDao.getNotMappedPreOrderIdsByBigOrderIds(bigOrderIds);
        final ProjectReader projectReader = hierarchySupplier.get().getProjectReader();

        final List<Map<Column, String>> rows = new ArrayList<>(preOrders.size());

        for (final BotPreOrder preOrder : preOrders) {
            final HashMap<Column, String> row = new HashMap<>();

            row.put(Column.ID, String.valueOf(preOrder.getId()));
            final Integer serviceId = preOrder.getAbcServiceId();
            if (serviceId != null) {
                final Set<Project> projects = projectReader.tryReadByAbcServiceIds(Collections.singleton(serviceId));
                if (!projects.isEmpty()) {
                    row.putAll(getProjectColumns(projects.iterator().next(), projectReader));
                }
            }

            row.put(Column.STATUS, formatPreOrderStatus(preOrder.getStatusId()));
            row.put(Column.PROVIDER, "HWServers");
            row.put(Column.SUBTICKET, preOrder.getTicketId());
            row.put(Column.RESPONSIBLES, preOrder.getResponsible());
            final BigOrder simpleBigOrder = bigOrderById.get(preOrder.getBigOrderId());
            if (simpleBigOrder != null) {
                row.put(Column.BIG_ORDER, simpleBigOrder.getDate().format(DateTimeFormatter.ISO_LOCAL_DATE));
                row.put(Column.DELIVERY_DATE, simpleBigOrder.getDate().format(DateTimeFormatter.ISO_LOCAL_DATE));
            }

            final long serversCount = preOrder.getServerQuantity() == null ? 0 : preOrder.getServerQuantity();
            final long storageCount = preOrder.getStorageQuantity() == null ? 0 : preOrder.getStorageQuantity();
            final long serversConfigurationsPrice = preOrder.getServerId() != null && configurationById.containsKey(preOrder.getServerId())
                    ? preOrder.getServerQuantity() * configurationById.get(preOrder.getServerId()).getPrice() : 0;
            final long storageConfigurationsPrice = preOrder.getStorageId() != null && configurationById.containsKey(preOrder.getStorageId())
                    ? preOrder.getStorageQuantity() * configurationById.get(preOrder.getStorageId()).getPrice() : 0;
            final long serversPrice = serversConfigurationsPrice + storageConfigurationsPrice;
            final long upgradePrice = preOrder.getUpgradesCost();

            final BigOrderConfig config = bigOrderConfigById.get(preOrder.getBigOrderConfigId());
            if (config != null) {
                Location location = Location.fromKey(config.getLocationKey());
                if (location != null) {
                    row.put(Column.SEGMENT, location.name());
                    fillLocations(row, Map.of(location, serversCount), COLUMN_BY_LOCATION, Column.ETC);
                    fillLocations(row, Map.of(location, serversCount), COMPUTE_COLUMN_BY_LOCATION, Column.ETC_COMPUTE);
                    fillLocations(row, Map.of(location, serversPrice), COMPUTE_PS_COLUMN_BY_LOCATION, Column.ETC_COMPUTE_PS);
                    fillLocations(row, Map.of(location, upgradePrice), UPGRADE_PS_COLUMN_BY_LOCATION, Column.ETC_UPGRADE_PS);
                } else {
                    row.put(Column.SEGMENT, "ETC");
                    row.put(Column.ETC, NUMBER_FORMAT.format(serversCount));
                    row.put(Column.ETC_COMPUTE, NUMBER_FORMAT.format(serversCount));
                    row.put(Column.ETC_COMPUTE_PS, NUMBER_FORMAT.format(serversPrice));
                    row.put(Column.ETC_UPGRADE_PS, NUMBER_FORMAT.format(upgradePrice));
                }
                final String summary = config.isUpgrade() ? "Апгрейды для ABC-Service" : "Физические серверы для ABC-Service";
                row.put(Column.SUMMARY, summary);
            }

            row.put(Column.JUSTIFICATION, DiResourcePreorderReasonType.GROWTH.name());

            final double totalPrice = serversPrice + upgradePrice;
            row.put(Column.TOTAL_PRICE, NUMBER_FORMAT.format(totalPrice));
            row.put(Column.SERVERS_TOTAL, NUMBER_FORMAT.format(serversCount));
            row.put(Column.STORAGES_TOTAL, NUMBER_FORMAT.format(storageCount));
            row.put(Column.STORAGES_PRICE, NUMBER_FORMAT.format(storageConfigurationsPrice));
            row.put(Column.HW_SERVERS_PRICE, NUMBER_FORMAT.format(serversPrice));
            row.put(Column.HW_UPGRADES_PRICE, NUMBER_FORMAT.format(upgradePrice));
            row.put(Column.HW_SERVERS, NUMBER_FORMAT.format(serversCount));

            final Map<BotResourceType, Long> preOrderResources = getPreOrderResources(preOrder, configurationById);

            for (final BotResourceType botResourceType : preOrderResources.keySet()) {
                final Long amount = preOrderResources.get(botResourceType) * preOrder.getServerQuantity();
                final Column column = COLUMN_BY_RESOURCE_TYPE.get(botResourceType);
                if (column != null) {
                    row.put(column, NUMBER_FORMAT.format(amount));
                }
            }

            row.put(Column.TOTAL_PRICE_GROWTH, NUMBER_FORMAT.format(totalPrice));
            row.put(Column.SERVERS_TOTAL_GROWTH, NUMBER_FORMAT.format(serversCount));

            rows.add(row);
        }
        return rows;
    }

    private static List<Map<Column, String>> getRowsForRemainders(final Collection<ServersRemainder> remainders,
                                                                  final BotCampaignGroup botCampaignGroup) {
        final List<Map<Column, String>> rows = new ArrayList<>(remainders.size());
        final CampaignForBot lastCampaign = botCampaignGroup.getActiveCampaigns().stream().max(Comparator.comparing(CampaignForBot::getStartDate))
                .orElseGet(() -> botCampaignGroup.getCampaigns().stream().max(Comparator.comparing(CampaignForBot::getStartDate)).get());
        final BigOrder lastBigOrder = lastCampaign.getBigOrders().stream().max(Comparator.comparing(BigOrder::getId)).get();
        for (ServersRemainder remainder : remainders) {
            final Map<Column, String> row = new HashMap<>();

            final Service service = remainder.getService();
            final Segment segment = remainder.getSegment();

            row.put(Column.ID, remainderId(remainder));
            row.put(Column.PROVIDER, service.getName());
            row.put(Column.SEGMENT, segment.getName());
            row.put(Column.CAMPAIGN, lastCampaign.getKey());
            row.put(Column.BIG_ORDER, lastBigOrder.getDate().format(DateTimeFormatter.ISO_LOCAL_DATE));
            row.put(Column.DELIVERY_DATE, lastBigOrder.getDate().format(DateTimeFormatter.ISO_LOCAL_DATE));
            row.put(Column.STATUS, "REMAINDER");
            row.put(Column.JUSTIFICATION, "GROWTH");
            row.put(Column.SUBTICKET, botCampaignGroup.getBotPreOrderIssueKey());

            final String formattedPrice = NUMBER_FORMAT.format(remainder.getPrice());
            final String formattedServers = NUMBER_FORMAT.format(remainder.getServerCount());
            final String formattedStorages = NUMBER_FORMAT.format(remainder.getStorageCount());

            row.put(Column.TOTAL_PRICE, formattedPrice);
            row.put(Column.TOTAL_PRICE_GROWTH, formattedPrice);
            final Location location = Location.fromKey(segment.getPublicKey());

            if (remainder.isUpgrade()) {
                row.put(Column.UPGRADES_COUNT, formattedServers);
                row.put(Column.HW_UPGRADES_PRICE, formattedPrice);
                if (location != null) {
                    fillLocations(row, Map.of(location, remainder.getPrice()), UPGRADE_PS_COLUMN_BY_LOCATION, Column.ETC_UPGRADE_PS);
                }
            } else {
                row.put(Column.SERVERS_TOTAL, formattedServers);
                row.put(Column.SERVERS_TOTAL_GROWTH, formattedServers);
                row.put(Column.STORAGES_TOTAL, formattedStorages);

                if (location != null) {
                    fillLocations(row, Map.of(location, remainder.getServerCount()), COLUMN_BY_LOCATION, Column.ETC);
                    fillLocations(row, Map.of(location, remainder.getServerCount()), COMPUTE_COLUMN_BY_LOCATION, Column.ETC_COMPUTE);
                    fillLocations(row, Map.of(location, remainder.getPrice()), COMPUTE_PS_COLUMN_BY_LOCATION, Column.ETC_COMPUTE_PS);
                }
            }

            rows.add(row);
        }

        return rows;
    }

    private static String remainderId(final ServersRemainder remainder) {
        final Service service = remainder.getService();
        final Segment segment = remainder.getSegment();
        final StringBuilder idBuilder = new StringBuilder("remainder-")
                .append(service.getKey())
                .append('-')
                .append(segment.getPublicKey())
                .append('-')
                .append(remainder.getGroupKey());
        if (remainder.getServerId() != null) {
            idBuilder.append("-srv").append(remainder.getServerId());
        }
        if (remainder.getStorageId() != null) {
            idBuilder.append("-stor").append(remainder.getStorageId());
        }
        return idBuilder.toString();
    }

    private static Map<Column, String> getProjectColumns(final Project requestProject, final ProjectReader projectReader) {
        final Map<Column, String> row = new HashMap<>();
        row.put(Column.ABC_SERVICE, requestProject.getName());

        List<Project> pathFromRoot = requestProject.getPathFromRoot();

        pathFromRoot = pathFromRoot.subList(Math.min(1, pathFromRoot.size()), pathFromRoot.size());

        if (pathFromRoot.size() >= 3) {
            final Project lastParent = pathFromRoot.get(pathFromRoot.size() - 2);
            pathFromRoot = pathFromRoot.subList(0, Math.min(pathFromRoot.size() - 2, 2));
            pathFromRoot.add(lastParent);
        }

        final List<Set<Person>> heads = new ArrayList<>();
        final List<String> headDepartments = new ArrayList<>();
        for (final Project project : pathFromRoot) {
            final Set<Person> projectResponsibles = projectReader.getLinkedResponsibles(project);
            heads.add(projectResponsibles);
            headDepartments.add(project.getName());
        }

        final List<Pair<Column, Column>> headColumns = Arrays.asList(
                Pair.of(Column.HEAD_1, Column.HEAD_DEPARTMENT_1),
                Pair.of(Column.HEAD_2, Column.HEAD_DEPARTMENT_2),
                Pair.of(Column.HEAD_3, Column.HEAD_DEPARTMENT_3)
        );

        for (int i = 0; i < headColumns.size(); i++) {
            final Column head = headColumns.get(i).getLeft();
            final Column headDepartment = headColumns.get(i).getRight();
            if (i < pathFromRoot.size()) {
                row.put(head, heads.get(i).stream().map(Person::getLogin).collect(Collectors.joining(",")));
                row.put(headDepartment, (headDepartments.get(i)));
            } else {
                row.put(head, "");
                row.put(headDepartment, "");
            }
        }

        final Long valueStreamAbcServiceId = requestProject.getValueStreamAbcServiceId();
        if (valueStreamAbcServiceId != null) {
            final Project project = Iterables.getFirst(
                    projectReader.tryReadByAbcServiceIds(Collections.singleton(valueStreamAbcServiceId.intValue())), null);
            if (project != null) {
                row.put(Column.VALUE_STREAM, project.getName());
                row.put(Column.VS_CAPACITY_PLANNER, getUsersForRole(projectReader, project, Role.RESPONSIBLE));
                row.put(Column.VS_MANAGER, getUsersForRole(projectReader, project, Role.STEWARD));
                row.put(Column.VS_LIDER, getUsersForRole(projectReader, project, Role.VS_LEADER));
            }
        }

        return row;
    }

    private static String getUsersForRole(final ProjectReader projectReader, final Project project, final Role role) {
        Project p = project;
        while (p != null) {
            final Set<Person> linkedPersons = projectReader.getLinkedPersons(p, role.getKey());
            if (!linkedPersons.isEmpty()) {
                return linkedPersons.stream()
                        .map(Person::getLogin)
                        .sorted()
                        .collect(Collectors.joining(", "));
            }
            p = p.isRoot() ? null : p.getParent();
        }
        return "";
    }


    private static Map<BotResourceType, Long> getPreOrderResources(final PreOrderBase preOrder, final Map<Long, CompleteBotConfiguration> configurationById) {

        final CompleteBotConfiguration server = preOrder.getServerId() == null ? null : configurationById.get(preOrder.getServerId());
        if (server == null || preOrder.getServerQuantity() == null) {
            return Collections.emptyMap();
        }

        final Map<BotResourceType, Long> resourcesByType = mapConfigurationComponents(server.getConfiguration());

        final CompleteBotConfiguration storage = preOrder.getStorageId() == null ? null : configurationById.get(preOrder.getStorageId());

        if (preOrder.getStorageQuantity() != null && preOrder.getServerQuantity() != null && storage != null) {
            final double storagePerServer = (double) preOrder.getStorageQuantity() / preOrder.getServerQuantity();
            final Map<BotResourceType, Long> storageResources = mapConfigurationComponents(storage.getConfiguration());
            storageResources.forEach((k, v) -> resourcesByType.merge(k, (long) (v * storagePerServer), Long::sum));
        }

        final Map<BotResourceType, Long> result = new HashMap<>();

        for (final BotResourceType type : resourcesByType.keySet()) {
            final Long unitAmount = RESOURCE_TYPE_BASE_UNITS.get(type);
            if (unitAmount == null) {
                continue;
            }
            result.put(type, resourcesByType.get(type) / unitAmount);
        }

        return result;
    }

    private static Map<BotResourceType, Long> mapConfigurationComponents(ConfigurationWithComponents configuration) {
        Map<BotResourceType, Long> result = new HashMap<>();
        for (ConfigurationComponent component : configuration.getConfigurationComponents()) {
            for (BotResourceType resourceType : BotResourceType.values()) {
                if (resourceType.is(component)) {
                    long amount = resourceType.getAmount(component);
                    if (amount > 0) {
                        result.put(resourceType, result.getOrDefault(resourceType, 0L) + amount);
                    }
                }
            }
        }
        return result;
    }

    private static String formatPreOrderStatus(final int statusId) {
        switch (statusId) {
            case 1:
                return "DRAFTED";
            case 2:
                return "COMMITTED";
            case 3:
                return "CONFIRMED";
            case 4:
                return "DONE";
            default:
                return "";
        }
    }

    @SuppressWarnings("UnstableApiUsage")
    private static Map<Column, String> getHeaderRow(final List<Column> columns) {
        return Maps.asMap(Sets.newHashSet(columns), Column::getTitle);
    }

    private static class ChangeServers {
        private final double servers;
        private final double price;
        private final Location location;
        private final boolean isGpu;
        private final Map<BotResourceType, Double> amountByType;
        private final String deliveryDate;
        private final boolean isUpgrade;

        private ChangeServers(final double servers, final double price, final Map<BotResourceType, Long> resourcesPerServer,
                              final Location location, final String deliveryDate, boolean isUpgrade) {
            this.servers = servers;
            this.price = price;
            this.location = location;
            isGpu = resourcesPerServer.containsKey(BotResourceType.GPU);
            this.deliveryDate = deliveryDate;
            this.isUpgrade = isUpgrade;

            amountByType = new HashMap<>();
            for (final BotResourceType type : resourcesPerServer.keySet()) {
                amountByType.put(type, resourcesPerServer.get(type) * servers);
            }
        }

        public double getServers() {
            return servers;
        }

        public double getPrice() {
            return price;
        }

        public Location getLocation() {
            return location;
        }

        public boolean isGpu() {
            return isGpu;
        }

        public Map<BotResourceType, Double> getAmountByType() {
            return amountByType;
        }

        public String getDeliveryDate() {
            return deliveryDate;
        }

        public boolean isUpgrade() {
            return isUpgrade;
        }
    }
}
