package ru.yandex.qe.dispenser.ws.quota.request.ticket;

import java.time.LocalDate;
import java.time.Month;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.inject.Inject;

import com.google.common.base.Suppliers;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;
import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.impl.ArrayListF;
import ru.yandex.qe.dispenser.api.v1.DiAmount;
import ru.yandex.qe.dispenser.api.v1.DiResourcePreorderReasonType;
import ru.yandex.qe.dispenser.domain.Campaign;
import ru.yandex.qe.dispenser.domain.ClosestResponsibleHelper;
import ru.yandex.qe.dispenser.domain.GoalQuestionHelper;
import ru.yandex.qe.dispenser.domain.Person;
import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.QuotaChangeRequest;
import ru.yandex.qe.dispenser.domain.Resource;
import ru.yandex.qe.dispenser.domain.ResourceGroup;
import ru.yandex.qe.dispenser.domain.Segment;
import ru.yandex.qe.dispenser.domain.Segmentation;
import ru.yandex.qe.dispenser.domain.Service;
import ru.yandex.qe.dispenser.domain.dao.goal.Goal;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;
import ru.yandex.qe.dispenser.domain.hierarchy.HierarchySupplier;
import ru.yandex.qe.dispenser.domain.tracker.IllegalTransitionException;
import ru.yandex.qe.dispenser.domain.tracker.TrackerManager;
import ru.yandex.qe.dispenser.domain.util.CollectionUtils;
import ru.yandex.qe.dispenser.quartz.trigger.TrackerJobRunner;
import ru.yandex.qe.dispenser.ws.quota.request.workflow.context.PerformerContext;
import ru.yandex.qe.dispenser.ws.quota.request.workflow.context.UpdateRequestContext;
import ru.yandex.startrek.client.error.StartrekClientException;
import ru.yandex.startrek.client.model.CommentCreate;
import ru.yandex.startrek.client.model.Issue;
import ru.yandex.startrek.client.model.IssueCreate;
import ru.yandex.startrek.client.model.IssueUpdate;
import ru.yandex.startrek.client.model.ServiceRef;
import ru.yandex.startrek.client.model.UserRef;

import static java.time.format.TextStyle.FULL_STANDALONE;
import static ru.yandex.qe.dispenser.domain.QuotaChangeRequest.Status.READY_FOR_REVIEW;

@SuppressWarnings({"OverlyComplexClass", "OverlyCoupledClass"})
@Component
@ParametersAreNonnullByDefault
public class QuotaChangeRequestTicketManager {

    private static final Logger LOG = LoggerFactory.getLogger(QuotaChangeRequestTicketManager.class);
    private static final String RESOURCE_PREORDER_PROPERTY_KEY = "resourcePreorder";
    private static final String DEFAULT_PERFORMER_LOGIN = "robot-dispenser";
    private static final String[] STRINGS = new String[0];
    public static final ServiceRef[] SERVICE_REFS = new ServiceRef[0];
    private final static Locale RUSSIAN_LOCALE = new Locale("ru");
    private final static DateTimeFormatter FORMATTER_WITH_RU_LOCALE = DateTimeFormatter.ofPattern("LLLL yyyy").withLocale(RUSSIAN_LOCALE);
    private final static DateTimeFormatter YEAR_FORMATTER = DateTimeFormatter.ofPattern("yyyy");

    private final static Map<QuotaChangeRequest.Status, String> TRACKER_STATUS_MAPPING = ImmutableMap.<QuotaChangeRequest.Status, String>builder()
            .put(QuotaChangeRequest.Status.NEW, "open")
            .put(READY_FOR_REVIEW, "readyToReview")
            .put(QuotaChangeRequest.Status.NEED_INFO, "needInfo")
            .put(QuotaChangeRequest.Status.APPROVED, "approved")
            .put(QuotaChangeRequest.Status.CONFIRMED, "confirmed")

            .put(QuotaChangeRequest.Status.COMPLETED, "closed")
            .put(QuotaChangeRequest.Status.CANCELLED, "closed")
            .put(QuotaChangeRequest.Status.REJECTED, "closed")

            .put(QuotaChangeRequest.Status.APPLIED, "closed")
            .build();

    private static final int MAX_FOLLOWER_COUNT = 20;
    public static final long FIXED_RESOLUTION_ID = 1;
    public static final long WONT_FIX_RESOLUTION_ID = 2;
    public static final String CRITICAL_TAG = "critical";

    private enum WikiTextColor {
        BLUE,
        GREEN,
        GREY,
        RED,
        YELLOW
    }

    @NotNull
    private final TrackerManager trackerManager;
    @NotNull
    private final MessageSource messageSource;
    @NotNull
    private final String ticketQueue;
    private final String clusterPrefix;
    private final String locationSegmentationKey;
    private final String abcFrontUrl;
    private final String goalServiceUrl;
    private final Map<String, Long> trackerComponents;
    private final GoalQuestionHelper goalQuestionHelper;
    private final HierarchySupplier hierarchySupplier;
    private final TrackerJobRunner trackerJobRunner;

    @SuppressWarnings("ConstructorWithTooManyParameters")
    @Inject
    public QuotaChangeRequestTicketManager(final TrackerManager trackerManager,
                                           final MessageSource messageSource,
                                           @Value("${dispenser.cluster.prefix}") final String clusterPrefix,
                                           @Value("${r.y.q.d.w.QuotaChangeRequestTicketManager.ticketQueue}") final String ticketQueue,
                                           @Value("${dispenser.location.segmentation.key}") final String locationSegmentationKey,
                                           @Value("${abc.front.url}") final String abcFrontUrl,
                                           @Value("${goal.service.url}") final String goalServiceUrl,
                                           @Value("#{${tracker.components}}") final Map<String, String> trackerComponents,
                                           final GoalQuestionHelper goalQuestionHelper,
                                           final HierarchySupplier hierarchySupplier,
                                           final TrackerJobRunner trackerJobRunner) {
        this.trackerManager = trackerManager;
        this.messageSource = messageSource;
        this.ticketQueue = ticketQueue;
        this.clusterPrefix = clusterPrefix;
        this.locationSegmentationKey = locationSegmentationKey;
        this.abcFrontUrl = abcFrontUrl;
        this.goalServiceUrl = goalServiceUrl;
        this.trackerComponents = trackerComponents.entrySet().stream()
                .collect(Collectors.toMap(Map.Entry::getKey, e -> Long.valueOf(e.getValue())));
        this.goalQuestionHelper = goalQuestionHelper;
        this.hierarchySupplier = hierarchySupplier;
        this.trackerJobRunner = trackerJobRunner;
    }

    @Nullable
    public String createTicketForQuotaChangeRequest(final QuotaChangeRequest quotaChangeRequest) {
        if (quotaChangeRequest.getType() != QuotaChangeRequest.Type.RESOURCE_PREORDER) {
            return null;
        }
        final IssueCreate issueCreate = getQuotaChangeRequestIssue(quotaChangeRequest);
        try {
            return trackerManager.createIssues(issueCreate);
        } catch (RuntimeException e) {
            //noinspection InstanceofCatchParameter
            if (e instanceof StartrekClientException) {
                LOG.error("Can't create request ticket: {} {}", issueCreate, ((StartrekClientException) e).getErrors());
            } else {
                LOG.error("Can't create request ticket: " + issueCreate, e);
            }
            LOG.debug("{}", issueCreate);
            return null;
        }

    }

    @Nullable
    public static String getAssignee(final QuotaChangeRequest quotaChangeRequest,
                                     final Hierarchy hierarchy) {
        if (quotaChangeRequest.getType() == QuotaChangeRequest.Type.RESOURCE_PREORDER) {
            final Optional<Pair<Project, ArrayList<Person>>> topLevelProject = getTopLevelProject(quotaChangeRequest, hierarchy);
            return topLevelProject.map(p -> ClosestResponsibleHelper.selectAssignee(p.getRight())).orElse(null);
        } else {
            final Set<Person> responsiblePersons = getResponsiblePersons(quotaChangeRequest, hierarchy);
            return getAssignee(getFollowers(responsiblePersons));
        }
    }

    private String format(final String key, final Object... args) {
        return messageSource.getMessage(key, args, Locale.getDefault());
    }

    @SuppressWarnings({"OverlyComplexMethod", "OverlyLongMethod"})
    @NotNull
    private IssueCreate getQuotaChangeRequestIssue(final QuotaChangeRequest quotaChangeRequest) {
        final Project project = quotaChangeRequest.getProject();

        final Set<Service> services = quotaChangeRequest.getChanges().stream()
                .map(c -> c.getResource().getService())
                .collect(Collectors.toSet());

        final String campaignKey = quotaChangeRequest.getCampaign() != null ? quotaChangeRequest.getCampaign().getKey() : null;
        final String requestSummary = quotaChangeRequest.getSummary();

        final String summary = campaignKey != null
                ? format("quota.request.preorder.ticket.summary", campaignKey, requestSummary, project.getName())
                : format("quota.request.ticket.summary", project.getName());
        final IssueCreate.Builder builder = IssueCreate.builder()
                .queue(ticketQueue)
                .summary(summary);

        builder.type("request");
        final List<String> tags = new ArrayList<>();

        quotaChangeRequest.getChanges().stream()
                .map(c -> c.getKey().getBigOrder())
                .filter(Objects::nonNull)
                .map(v -> v.getDate().format(DateTimeFormatter.ISO_LOCAL_DATE))
                .distinct()
                .forEach(tags::add);

        if (quotaChangeRequest.getCampaign() != null) {
            tags.add(campaignKey);
            tags.add(quotaChangeRequest.getCampaign().getType().name().toLowerCase(Locale.ENGLISH));
        }

        if (quotaChangeRequest.isImportantRequest()) {
            tags.add(CRITICAL_TAG);
        }

        builder.tags(tags.toArray(STRINGS));

        final Optional<Pair<Project, ArrayList<Person>>> topLevelProject = getTopLevelProject(quotaChangeRequest, Hierarchy.get());
        if (topLevelProject.isPresent()) {
            final List<ServiceRef> serviceRefs = new ArrayList<>();

            final Integer assigneeAbcService = topLevelProject.get().getLeft().getAbcServiceId();
            if (assigneeAbcService != null) {
                serviceRefs.add(new ServiceRef(Long.valueOf(assigneeAbcService)));
            }

            final Integer selfAbcService = quotaChangeRequest.getProject().getAbcServiceId();
            if (selfAbcService != null) {
                serviceRefs.add(new ServiceRef(Long.valueOf(selfAbcService)));
            }

            if (!serviceRefs.isEmpty()) {
                builder.set("abcService", serviceRefs.toArray(SERVICE_REFS));
            }
        }

        final long[] components = new long[services.size() + 1];
        int i = 0;
        components[i++] = trackerComponents.get(RESOURCE_PREORDER_PROPERTY_KEY);
        for (final Service service : services) {
            components[i++] = trackerComponents.get(service.getKey());
        }

        builder.set("components", components);

        return builder
                .description(getPreOrderRequestDescription(quotaChangeRequest))
                .author(quotaChangeRequest.getAuthor().getLogin())
                .unique("dispenser_qcr_" + clusterPrefix + quotaChangeRequest.getId())
                .build();
    }

    public static String toWikiTable(final List<List<String>> data) {
        final StringBuilder result = new StringBuilder();

        result.append("#|\n");

        for (final List<String> datum : data) {
            result.append("|| ")
                    .append(StringUtils.join(datum, " | "))
                    .append(" ||\n");
        }

        result.append("|#\n");
        return result.toString();
    }

    public static String toWikiTableWithBoldResourceSummary(final List<List<String>> data) {
        if (data.size() < 3) { // only resource header and single change
            return toWikiTable(data);
        }

        return toWikiTableWithBoldRow(data, 1);
    }

    public static String toWikiTableWithBoldRow(final List<List<String>> data, final int row) {
        final StringBuilder result = new StringBuilder();
        result.append("#|\n");

        for (int i = 0; i < data.size(); i++) {
            final List<String> datum = i == row ? toBoldDatum(data.get(i)) : data.get(i);

            result.append("|| ")
                    .append(StringUtils.join(datum, " | "))
                    .append(" ||\n");
        }

        result.append("|#\n");
        return result.toString();
    }

    private static List<String> toBoldDatum(@NotNull final List<String> datum) {
        return datum.stream()
                .map(QuotaChangeRequestTicketManager::toBold)
                .collect(Collectors.toList());
    }

    private static String toBold(final String text) {
        return StringUtils.isEmpty(text)
                ? text
                : String.format("**%s**", text);
    }

    @SuppressWarnings({"OverlyLongMethod"})
    private String getPreOrderRequestDescription(final QuotaChangeRequest quotaChangeRequest) {

        final StringBuilder description = new StringBuilder();

        final Integer abcServiceId = quotaChangeRequest.getProject().getAbcServiceId();

        description.append("**Тикет создан только для согласования. Для подтверждения или изменения перейдите к " + "((")
                .append(abcFrontUrl).append("/hardware/").append(quotaChangeRequest.getId())
                .append(" заявке))**").append("\n");

        description.append("===Подтверждение заявки\n");
        description.append("Ожидается подтверждение от ответственного за Capacity Planning сервиса ((").append(abcFrontUrl).append("/services/")
                .append(abcServiceId).append(" ").append(quotaChangeRequest.getProject().getName())
                .append(")). Когда заявка будет готова к защите, данные, нужные для защиты, появятся в этом тикете, а ответственный получит призыв.\n");

        description.append("===Данные заявки\n");

        final Project project = quotaChangeRequest.getProject();

        final String projectName = String.format("((%s/services/%s/hardware/?view=consuming %s))", abcFrontUrl, project.getPublicKey(), project.getName());

        final List<List<String>> properties = new ArrayList<>();

        properties.add(ImmutableList.of("**Сервис**", projectName));

        properties.add(ImmutableList.of("**Автор заявки**", "staff:" + quotaChangeRequest.getAuthor()));

        final Map<String, String> additionalProperties = quotaChangeRequest.getAdditionalProperties();
        if (additionalProperties != null) {
            properties.addAll(additionalProperties.entrySet().stream()
                    .map(e -> ImmutableList.of(String.format("**%s**", e.getKey()), e.getValue()))
                    .collect(Collectors.toList()));
        }

        final Set<Service> services = quotaChangeRequest.getChanges().stream()
                .map(c -> c.getResource().getService())
                .collect(Collectors.toSet());

        properties.add(ImmutableList.of("**Провайдеры**",
                services.stream()
                        .map(Service::getName)
                        .sorted()
                        .collect(Collectors.joining(", "))
        ));

        final QuotaChangeRequest.Campaign campaign = quotaChangeRequest.getCampaign();

        if (campaign != null) {
            properties.add(ImmutableList.of("**Кампания**", campaign.getName()));
            properties.add(ImmutableList.of("**Тип кампании**", getCampaignTypeTitle(campaign.getType())));
        }

        final String deliveryDates = fillDeliveryDates(quotaChangeRequest);
        if (deliveryDates != null) {
            properties.add(ImmutableList.of("**Даты поставки**", deliveryDates));
        }

        properties.add(ImmutableList.of("**Статус**", getStatusTitle(quotaChangeRequest)));

        description.append(toWikiTable(ImmutableList.copyOf(properties)));

        final Map<Service, List<QuotaChangeRequest.Change>> changesByResources = quotaChangeRequest.getChanges().stream()
                .collect(Collectors.groupingBy(c -> c.getResource().getService()));

        for (final Service service : changesByResources.keySet()) {
            description.append("===Ресурсы ")
                    .append(service.getName())
                    .append("\n");

            final List<List<String>> serviceResources = formatChanges(changesByResources.get(service),
                    QuotaChangeRequest.Change::getAmount, false);

            description.append(toWikiTableWithBoldResourceSummary(serviceResources));
        }

        if (quotaChangeRequest.getDescription() != null) {
            description.append("Комментарий:\n")
                    .append(quotaChangeRequest.getDescription())
                    .append("\n");
        }

        if (quotaChangeRequest.getStatus() != QuotaChangeRequest.Status.NEW) {
            description.append("===Защита\n");

            final List<List<String>> defendProperties = getDefendAnswersList(quotaChangeRequest);
            description.append(toWikiTable(ImmutableList.copyOf(defendProperties)));
        }
        return description.toString();
    }

    private String getStatusTitle(QuotaChangeRequest request) {
        return getStatusTitle(request.getStatus());
    }

    private String getStatusTitle(QuotaChangeRequest.Status status) {
        return format("quota.request.ticket.status." + status.name() + ".title");
    }

    private String getCampaignTypeTitle(Campaign.Type campaignType) {
        return format("quota.request.campaign.type." + campaignType.name() + ".title");
    }

    @NotNull
    private List<List<String>> getDefendAnswersList(final QuotaChangeRequest quotaChangeRequest) {
        final List<List<String>> defendProperties = new ArrayList<>();

        if (quotaChangeRequest.getResourcePreorderReasonType() != null) {
            defendProperties.add(ImmutableList.of("**Причина заказа ресурсов**", toReasonString(quotaChangeRequest)));
        }

        if (quotaChangeRequest.getCalculations() != null && !quotaChangeRequest.getCalculations().isEmpty()) {
            defendProperties.add(ImmutableList.of("**Расчёты**", quotaChangeRequest.getCalculations()));
        }

        if (!quotaChangeRequest.getChartLinks().isEmpty()) {
            defendProperties.add(ImmutableList.of("**Графики утилизации**", prepareChartLinks(quotaChangeRequest)));
        }

        if (quotaChangeRequest.getChartLinksAbsenceExplanation() != null) {
            defendProperties.add(ImmutableList.of("**Причина отсутствия графиков**", quotaChangeRequest.getChartLinksAbsenceExplanation()));
        }

        if (quotaChangeRequest.getRequestGoalAnswers() != null
                && (quotaChangeRequest.getResourcePreorderReasonType() == DiResourcePreorderReasonType.GROWTH
                || quotaChangeRequest.getResourcePreorderReasonType() == DiResourcePreorderReasonType.GOAL)) {
            final Map<Long, String> requestGoalAnswers = quotaChangeRequest.getRequestGoalAnswers();
            final DiResourcePreorderReasonType resourcePreorderReasonType = quotaChangeRequest.getResourcePreorderReasonType();

            requestGoalAnswers.keySet().stream()
                    .sorted()
                    .forEach(id -> defendProperties.add(ImmutableList.of(
                            String.format("**%s**", goalQuestionHelper.getRequestGoalQuestion(resourcePreorderReasonType, id)),
                            requestGoalAnswers.get(id))
                    ));
        }

        if (quotaChangeRequest.getComment() != null) {
            defendProperties.add(ImmutableList.of("**Комментарий**", quotaChangeRequest.getComment()));
        }
        return defendProperties;
    }

    public String toReasonString(final QuotaChangeRequest quotaChangeRequest) {
        final DiResourcePreorderReasonType resourcePreorderReasonType = Objects.requireNonNull(quotaChangeRequest.getResourcePreorderReasonType());

        switch (resourcePreorderReasonType) {
            case GOAL:
                final Goal goal = Objects.requireNonNull(quotaChangeRequest.getGoal());
                return String.format("((%s/filter?goal=%s %s))", goalServiceUrl, goal.getId(), goal.getName());
            case GROWTH:
                return "естественный рост";
            case NOTHING:
            default:
                return "-";
        }
    }

    @SuppressWarnings("OverlyLongMethod")
    public static List<List<String>> formatChanges(final Collection<QuotaChangeRequest.Change> changes,
                                                   final Function<QuotaChangeRequest.Change, Long> amountSupplier,
                                                   final boolean providerNameInHeaders) {
        final Set<Resource> usedResources = new HashSet<>();
        final HashMap<String, Map<Resource, Long>> data = new HashMap<>();
        final HashMap<String, Pair<LocalDate, String>> rawNameToDate = new HashMap<>();
        final HashMap<Resource, Long> totalResources = new HashMap<>();

        changes.forEach(change -> {
            //segment keys should be unique
            final Map<Segmentation.Key, String> segmentKeyBySegmentationKey = change.getSegments().stream()
                    .collect(Collectors.toMap(segment -> segment.getSegmentation().getKey(), Segment::getName));
            final ArrayList<Segmentation.Key> keys = Lists.newArrayList(segmentKeyBySegmentationKey.keySet());
            Collections.sort(keys);
            final QuotaChangeRequest.BigOrder order = change.getKey().getBigOrder();
            final String segments = keys.stream().map(segmentKeyBySegmentationKey::get).collect(Collectors.joining(", "));
            final String rawName;
            if (order != null) {
                final LocalDate localDate = order.getDate();
                final String date = StringUtils.capitalize(localDate.format(FORMATTER_WITH_RU_LOCALE));
                if (segments.isEmpty()) {
                    rawName = date;
                    rawNameToDate.put(rawName, Pair.of(localDate, ""));
                } else {
                    rawName = date + ": " + segments;
                    rawNameToDate.put(rawName, Pair.of(localDate, segments));
                }
            } else {
                rawName = segments;
                rawNameToDate.put(rawName, Pair.of(LocalDate.MIN, segments));
            }
            final long amount = amountSupplier.apply(change);
            final Resource resource = change.getResource();
            totalResources.merge(resource, amount, Long::sum);
            if (data.containsKey(rawName)) {
                if (data.get(rawName).containsKey(resource)) {
                    data.get(rawName).put(resource, data.get(rawName).get(resource) + amount);
                } else {
                    data.get(rawName).put(resource, amount);
                }
            } else {
                final HashMap<Resource, Long> resources = new HashMap<>();
                resources.put(resource, amount);
                data.put(rawName, resources);
            }
            usedResources.add(resource);
        });
        final ArrayList<Resource> sortedResources = Lists.newArrayList(usedResources);
        Collections.sort(sortedResources);

        final List<String> caption = new ArrayList<>();
        caption.add("");
        caption.addAll(sortedResources.stream().map(r -> prepareResourceName(r, providerNameInHeaders))
                .collect(Collectors.toList()));
        final List<List<String>> resources = new ArrayList<>();
        resources.add(caption);
        if (data.size() > 1) { // add total if more than one row
            resources.add(Stream.concat(Stream.of(""), sortedResources.stream().map(getResourceStringFunction(totalResources)))
                    .collect(Collectors.toList()));
        }

        final Comparator<Map.Entry<String, Map<Resource, Long>>> comparator =
                Comparator.comparing((Map.Entry<String, Map<Resource, Long>> o) -> rawNameToDate.get(o.getKey()).getLeft())
                        .thenComparing(o -> rawNameToDate.get(o.getKey()).getRight());

        resources.addAll(data.entrySet().stream()
                .sorted(comparator)
                .map(stringMapEntry -> {
                    final List<String> raw = new ArrayList<>();
                    raw.add(stringMapEntry.getKey());
                    raw.addAll(sortedResources.stream()
                            .map(getResourceStringFunction(stringMapEntry.getValue()))
                            .collect(Collectors.toList()));
                    return raw;
                })
                .collect(Collectors.toList()));
        return resources;
    }

    private static String prepareResourceName(Resource resource, boolean providerNameInHeaders) {
        if (providerNameInHeaders) {
            return resource.getService().getName() + " " + resource.getName();
        }
        return resource.getName();
    }

    @NotNull
    private static Function<Resource, String> getResourceStringFunction(final Map<Resource, Long> stringMapEntry) {
        return r -> {
            final Long amount = stringMapEntry.getOrDefault(r, 0L);
            final DiAmount.Humanized amountH = DiAmount.of(amount, r.getType().getBaseUnit()).humanize();
            return amountH.toString();
        };
    }

    public static List<List<String>> getResources(final QuotaChangeRequest quotaChangeRequest) {
        return formatChanges(quotaChangeRequest.getChanges(), QuotaChangeRequest.Change::getAmount, false);
    }

    private String formatResourcesDiff(final QuotaChangeRequest oldRequest, final QuotaChangeRequest newRequest) {
        final StringBuilder resourcesFormatted = new StringBuilder();

        final List<Pair<QuotaChangeRequest.Change, QuotaChangeRequest.Change>> changesDiff = CollectionUtils.outerJoin(oldRequest.getChanges(),
                newRequest.getChanges());

        final Map<Boolean, List<Pair<QuotaChangeRequest.Change, QuotaChangeRequest.Change>>> changesByHasGroupPredicate = changesDiff.stream()
                .collect(Collectors.partitioningBy(p -> Stream.of(p.getLeft(), p.getRight()).filter(Objects::nonNull).anyMatch(c -> c.getResource().getGroup() != null)));

        final Comparator<Pair<QuotaChangeRequest.Change, QuotaChangeRequest.Change>> pairComparator =
                Comparator.comparing(p -> Stream.of(p.getLeft(), p.getRight()).filter(Objects::nonNull).map(QuotaChangeRequest.ChangeAmount::getResource).findFirst().get());

        changesByHasGroupPredicate.get(false).stream()
                .sorted(pairComparator)
                .map(p -> formatChangeResourceDiffRow(p.getLeft(), p.getRight()))
                .forEach(resourcesFormatted::append);

        final Map<ResourceGroup, List<Pair<QuotaChangeRequest.Change, QuotaChangeRequest.Change>>> diffByGroups = changesByHasGroupPredicate.get(true)
                .stream()
                .collect(Collectors.groupingBy(p -> Stream.of(p.getLeft(), p.getRight()).filter(Objects::nonNull).map(c -> c.getResource().getGroup()).findFirst().get()));

        for (final ResourceGroup group : diffByGroups.keySet()) {

            resourcesFormatted
                    .append("* ")
                    .append(group.getName())
                    .append("\n");

            diffByGroups.get(group).stream()
                    .sorted(pairComparator)
                    .map(p -> formatChangeResourceDiffRow(p.getLeft(), p.getRight()))
                    .forEach(resourceFormatted -> resourcesFormatted.append("  ").append(resourceFormatted));
        }

        return resourcesFormatted.toString();
    }

    private String formatChangeResourceDiffRow(@Nullable final QuotaChangeRequest.Change oldChange, @Nullable final QuotaChangeRequest.Change newChange) {
        if (oldChange == null && newChange == null) {
            return "";
        }
        if (oldChange != null && newChange == null) {
            return "* " + formatChangeResourceRemoval(oldChange) + "\n";
        }
        QuotaChangeRequest.Change leftChange = oldChange;
        if (oldChange == null) {
            leftChange = newChange.copyBuilder().amount(0).build();
        }

        final Resource resource = leftChange.getResource();

        final StringBuilder resultBuilder = new StringBuilder();

        resultBuilder
                .append("* ")
                .append(resource.getName());

        final Set<Segment> segments = leftChange.getSegments();
        if (!segments.isEmpty()) {
            resultBuilder.append(formatSegments(segments));
        }

        final DiAmount.Humanized oldAmount = DiAmount.of(leftChange.getAmount(), resource.getType().getBaseUnit()).humanize();
        resultBuilder
                .append(": ")
                .append(oldAmount.toString());

        if (leftChange.getAmount() == newChange.getAmount()) {
            return resultBuilder
                    .append('\n')
                    .toString();
        }
        final DiAmount.Humanized newAmount = DiAmount.of(newChange.getAmount(), resource.getType().getBaseUnit()).humanize();

        final String color = newChange.getAmount() > leftChange.getAmount() ? "red" : "green";
        return resultBuilder
                .append(" -> **!!(")
                .append(color)
                .append(')')
                .append(newAmount.toString())
                .append("!!**\n")
                .toString();
    }

    private String formatChangeResourceRemoval(final QuotaChangeRequest.Change removedChange) {
        return "--" + formatChangeResourceAmount(removedChange) + "--";
    }

    private String formatChangeResourceAmount(final QuotaChangeRequest.Change change) {
        final Resource resource = change.getResource();
        final DiAmount.Humanized amount = DiAmount.of(change.getAmount(), resource.getType().getBaseUnit()).humanize();

        final StringBuilder resultBuilder = new StringBuilder();

        resultBuilder
                .append(resource.getName());

        final Set<Segment> segments = change.getSegments();
        if (!segments.isEmpty()) {
            resultBuilder.append(formatSegments(segments));
        }

        return resultBuilder
                .append(": ")
                .append(amount.toString())
                .toString();
    }

    private String formatSegments(final Set<Segment> segments) {
        Stream<Segment> segmentStream = segments.stream();

        if (segments.size() == 2) {
            segmentStream = segmentStream
                    .sorted(Comparator.comparing(segment ->
                            locationSegmentationKey.equals(segment.getSegmentation().getKey().getPublicKey()) ? 0 : 1));
        }

        return " ["
                + segmentStream
                .map(Segment::getName)
                .collect(Collectors.joining(", "))
                + "]";
    }

    public boolean applyTicketChanges(final QuotaChangeRequest quotaChangeRequest, final QuotaChangeRequest original,
                                      final PerformerContext ctx, final boolean suppressSummon) {
        return applyTicketChanges(quotaChangeRequest, original, IssueUpdate.builder(), false, ctx, suppressSummon);
    }

    public boolean refreshTicket(final QuotaChangeRequest request, final boolean summon, final boolean suppressSummon) {
        if (request.getType() != QuotaChangeRequest.Type.RESOURCE_PREORDER) {
            return false;
        }
        final IssueUpdate.Builder updateBuilder = IssueUpdate.builder()
                .description(getPreOrderRequestDescription(request));

        return applyTicketChanges(request, request, updateBuilder, summon,
                new PerformerContext(hierarchySupplier.get().getPersonReader().read(DEFAULT_PERFORMER_LOGIN)),
                suppressSummon);
    }

    @SuppressWarnings({"OverlyComplexMethod", "OverlyLongMethod"})
    private boolean applyTicketChanges(final QuotaChangeRequest quotaChangeRequest, final QuotaChangeRequest original,
                                       final IssueUpdate.Builder updateBuilder, final boolean summonAssignee,
                                       final PerformerContext ctx, final boolean suppressSummon) {
        if (quotaChangeRequest.getType() != QuotaChangeRequest.Type.RESOURCE_PREORDER) {
            return false;
        }
        final String trackerIssueKey = quotaChangeRequest.getTrackerIssueKey();
        if (trackerIssueKey == null) {
            return false;
        }
        final QuotaChangeRequest.Status status = quotaChangeRequest.getStatus();
        final QuotaChangeRequest.Status originalStatus = original.getStatus();

        final Supplier<Issue> issueSupplier =
                Suppliers.memoize(() -> trackerManager.getIssue(trackerIssueKey))::get;

        final List<CommentCreate> comments = Lists.newArrayList();

        boolean needToSummmonAssignee = summonAssignee;

        Optional<String> transition = Optional.empty();

        if (status != originalStatus) {

            switch (status) {
                case APPLIED:
                case COMPLETED:
                    updateBuilder.resolution(FIXED_RESOLUTION_ID);
                    break;
                case REJECTED:
                case CANCELLED:
                    updateBuilder.resolution(WONT_FIX_RESOLUTION_ID);
                    break;
                case APPROVED:
                case NEED_INFO:
                case CONFIRMED:
                case READY_FOR_REVIEW:
                    break;
                case NEW:
                    // For resource pre-orders do not summon capacity planners on transition to new
                    break;
            }

            final Issue issue = issueSupplier.get();
            final String authorLogin = issue.getCreatedBy().getLogin();

            final CommentCreate comment;
            if (status == READY_FOR_REVIEW) {
                final Supplier<List<Person>> responsible = () -> getTopLevelProjectResponsiblePersons(quotaChangeRequest, Hierarchy.get());
                final String assignee = prepareAssigneeUpdate(updateBuilder, issueSupplier.get(), responsible, false);
                comment = getStatusChangeRequestCommentForReadyForReview(quotaChangeRequest, assignee, ctx, suppressSummon);
            } else {
                comment = getStatusChangeRequestComment(status, authorLogin, quotaChangeRequest, ctx, suppressSummon);
            }

            comments.add(comment);

            final String originalStatusKey = TRACKER_STATUS_MAPPING.get(originalStatus);
            final String trackerStatusKey = TRACKER_STATUS_MAPPING.get(status);

            if (trackerStatusKey == null) {
                LOG.error("Ticket target status not founded for request: {}, update issue", quotaChangeRequest);
            } else if (!trackerStatusKey.equals(originalStatusKey)) {
                transition = Optional.of(trackerStatusKey);
            }
        } else {
            final Issue issue = issueSupplier.get();
            final String trackerStatusKey = TRACKER_STATUS_MAPPING.get(status);

            if (trackerStatusKey != null && !trackerStatusKey.equals(issue.getStatus().getKey())) {
                transition = Optional.of(trackerStatusKey);
            }
        }

        comments.addAll(updateBuilder.build().getComment());

        final boolean projectChanged = !quotaChangeRequest.getProject().equals(original.getProject());
        if (projectChanged) {
            needToSummmonAssignee = isAssigneeSummonedForResourcePreOrderRequestUpdate(status, originalStatus) || needToSummmonAssignee;
            comments.add(CommentCreate.comment(format("quota.request.projectChanged.comment", ctx.getPerson(),
                    quotaChangeRequest.getProject().getName())).build());
            updateBuilder.description(getPreOrderRequestDescription(quotaChangeRequest));
        } else {
            updateBuilder.description(getPreOrderRequestDescription(quotaChangeRequest));
        }

        boolean importantFlagUpdated = quotaChangeRequest.isImportantRequest() != original.isImportantRequest();
        if (importantFlagUpdated) {
            if (quotaChangeRequest.isImportantRequest()) {
                updateBuilder.tags(Cf.list(CRITICAL_TAG), Cf.list());
            } else {
                updateBuilder.tags(Cf.list(), Cf.list(CRITICAL_TAG));
            }
        }

        String requestSummary = quotaChangeRequest.getSummary();
        if (importantFlagUpdated || !Objects.equals(requestSummary, original.getSummary())) {
            final Project project = quotaChangeRequest.getProject();
            final String campaignKey = quotaChangeRequest.getCampaign() != null ? quotaChangeRequest.getCampaign().getKey() : null;
            final String summary;
            if (campaignKey == null) {
                summary = format("quota.request.ticket.summary", project.getName());
            } else {
                summary = quotaChangeRequest.isImportantRequest()
                        ? format("quota.request.preorder.ticket.summary.critical", campaignKey, CRITICAL_TAG, requestSummary, project.getName())
                        : format("quota.request.preorder.ticket.summary", campaignKey, requestSummary, project.getName());
            }

            updateBuilder.summary(summary);
        }

        try {
            final Issue issue = issueSupplier.get();
            if (needToSummmonAssignee || (projectChanged && issue.getAssignee().isPresent())) {
                final Supplier<List<Person>> responsible = () -> getTopLevelProjectResponsiblePersons(quotaChangeRequest, Hierarchy.get());
                final String assignee = prepareAssigneeUpdate(updateBuilder, issue, responsible, projectChanged);
                if (needToSummmonAssignee) {
                    CommentCreate.Builder commentCreateBuilder = CommentCreate
                            .comment(format("quota.request.preorder.ticket.summonee.comment", ctx.getPerson()));
                    if (!suppressSummon) {
                        commentCreateBuilder.summonees(assignee);
                    }
                    comments.add(commentCreateBuilder.build());
                }
            }

            mergeComments(comments).ifPresent(updateBuilder::comment);

            final IssueUpdate issueUpdate = updateBuilder.build();


            if (transition.isPresent()) {
                executeTransition(trackerIssueKey, transition.get(), issueUpdate);
            } else {
                updateIssue(trackerIssueKey, issueUpdate);
            }
            return true;
        } catch (RuntimeException e) {
            LOG.error("Can't update ticket status: {}", quotaChangeRequest, e);
            return false;
        }
    }

    private void updateIssue(final String trackerIssueKey, final IssueUpdate issueUpdate) {
        try {
            trackerManager.updateIssue(trackerIssueKey, issueUpdate);
        } catch (Exception ignored) {
            LOG.error("Can't update issue {} with body {}", trackerIssueKey, issueUpdate);
            final boolean scheduled = trackerJobRunner.scheduleUpdateIssue(trackerIssueKey, null, issueUpdate);
            if (!scheduled) {
                LOG.error("Can't schedule issue updated {} with body {}", trackerIssueKey, issueUpdate);
            }
        }
    }

    private void executeTransition(final String trackerIssueKey, final String transition, final IssueUpdate issueUpdate) {
        try {
            trackerManager.executeTransition(trackerIssueKey, transition, issueUpdate);
        } catch (IllegalTransitionException ignored) {
            LOG.error("Can't transit issue {} to {} with body {}", trackerIssueKey, transition, issueUpdate);
            updateIssue(trackerIssueKey, issueUpdate);
        } catch (Exception e) {
            final boolean scheduled = trackerJobRunner.scheduleUpdateIssue(trackerIssueKey, transition, issueUpdate);
            if (!scheduled) {
                LOG.error("Can't schedule issue transit {} to {} with body {}", trackerIssueKey, transition, issueUpdate);
            }
        }
    }

    private boolean isAssigneeSummonedForResourcePreOrderRequestUpdate(final QuotaChangeRequest.Status status,
                                                                       final QuotaChangeRequest.Status originalStatus) {
        // Do not summon capacity planners to updated resource pre-order issues with new status
        return (status == READY_FOR_REVIEW && originalStatus == READY_FOR_REVIEW)
                || (status == READY_FOR_REVIEW && originalStatus == QuotaChangeRequest.Status.APPROVED);
    }

    private Optional<CommentCreate> mergeComments(final List<CommentCreate> comments) {
        if (comments.isEmpty()) {
            return Optional.empty();
        }
        if (comments.size() == 1) {
            return Optional.of(comments.iterator().next());
        }


        final Set<String> summonees = new HashSet<>();
        final List<String> text = new ArrayList<>();

        for (final CommentCreate comment : comments) {
            summonees.addAll(comment.getSummonees());
            if (comment.getComment().isPresent()) {
                text.add(comment.getComment().get());
            }
        }

        return Optional.of(CommentCreate.builder()
                .summonees(new ArrayListF<>(summonees))
                .comment(StringUtils.join(text, "\n\n"))
                .build());
    }

    public void updateTicket(final QuotaChangeRequest request, final Set<QuotaChangeRequest.Field> changedFields,
                             final QuotaChangeRequest originalRequest, final UpdateRequestContext ctx,
                             final boolean suppressSummon) {
        if (request.getType() != QuotaChangeRequest.Type.RESOURCE_PREORDER) {
            return;
        }
        final String trackerIssueKey = request.getTrackerIssueKey();
        if (trackerIssueKey == null) {
            return;
        }

        final IssueUpdate.Builder updateBuilder = IssueUpdate.builder()
                .description(getPreOrderRequestDescription(request));

        // Do not summon capacity planners to updated resource pre-order issues with new status
        final boolean summonAssignee = isAssigneeSummonedForResourcePreOrderRequestUpdate(request.getStatus(), originalRequest.getStatus());
        updateBuilder.comment(getChangeComment(originalRequest, request, changedFields, ctx));

        applyTicketChanges(request, originalRequest, updateBuilder, summonAssignee, ctx, suppressSummon);
    }

    @NotNull
    private static Optional<Pair<Project, ArrayList<Person>>> getTopLevelProject(final QuotaChangeRequest quotaChangeRequest,
                                                                                 final Hierarchy hierarchy) {
        return ClosestResponsibleHelper.getTopLevelProject(hierarchy, quotaChangeRequest.getProject());
    }

    @NotNull
    private static List<Person> getTopLevelProjectResponsiblePersons(final QuotaChangeRequest quotaChangeRequest,
                                                                     final Hierarchy hierarchy) {
        final Optional<Pair<Project, ArrayList<Person>>> topLevelProject = getTopLevelProject(quotaChangeRequest, hierarchy);
        return topLevelProject.isPresent() ? topLevelProject.get().getRight() : Collections.emptyList();
    }

    @NotNull
    private static Set<Person> getResponsiblePersons(final QuotaChangeRequest quotaChangeRequest,
                                                     final Hierarchy hierarchy) {
        return getSourceProject(quotaChangeRequest).flatMap(source -> {
            final List<Project> pathToRoot = source.getPathToRoot();
            for (final Project project : pathToRoot) {
                final Set<Person> responsiblePersons = hierarchy.getProjectReader().getLinkedResponsibles(project);
                if (!responsiblePersons.isEmpty()) {
                    return Optional.of(responsiblePersons);
                }
            }
            return Optional.empty();
        }).orElseGet(() -> {
            final Set<Service> services = quotaChangeRequest.getChanges().stream()
                    .map(c -> c.getResource().getService())
                    .collect(Collectors.toSet());
            final Optional<Set<Person>> servicesAdmin = hierarchy.getServiceReader().getAdmins(services).asMap().values().stream()
                    .map(persons -> (Set<Person>) new HashSet<>(persons))
                    .reduce(Sets::intersection);
            return servicesAdmin.orElseGet(() -> hierarchy.getDispenserAdminsReader().getDispenserAdmins());
        });
    }

    @NotNull
    private static String[] getFollowers(final Set<Person> responsiblePersons) {
        return responsiblePersons.stream()
                .map(Person::getLogin)
                .sorted()
                .limit(MAX_FOLLOWER_COUNT)
                .toArray(String[]::new);
    }

    @Nullable
    private static String getAssignee(final String[] followers) {
        return followers.length > 0 ? followers[0] : null;
    }

    private String prepareAssigneeUpdate(final IssueUpdate.Builder updateBuilder, final Issue issue,
                                         final Supplier<List<Person>> responsible, final boolean forceChange) {
        if (issue.getAssignee().isPresent() && !forceChange) {
            return issue.getAssignee().get().getLogin();
        }
        final List<Person> responsibles = responsible.get();
        final String assignee = ClosestResponsibleHelper.selectAssignee(responsibles);
        if (assignee != null) {
            updateBuilder.assignee(assignee);
        }
        final Set<String> followers = issue.getFollowers().stream()
                .map(UserRef::getLogin)
                .collect(Collectors.toSet());
        final Set<String> responsibleFollowers = ImmutableSet.copyOf(selectFollowers(responsibles));
        final Set<String> newFollowers = Sets.union(followers, responsibleFollowers);
        updateBuilder.followers(newFollowers.stream().toArray(String[]::new));
        return assignee;
    }

    private static String[] selectFollowers(final List<Person> responsible) {
        return responsible.stream().map(Person::getLogin).sorted().limit(MAX_FOLLOWER_COUNT).toArray(String[]::new);
    }

    private CommentCreate getChangeComment(final QuotaChangeRequest originalRequest,
                                           final QuotaChangeRequest request,
                                           final Set<QuotaChangeRequest.Field> changedFields,
                                           final UpdateRequestContext ctx) {
        final List<String> messages = new ArrayList<>(2);
        final boolean hasChanges = changedFields.contains(QuotaChangeRequest.Field.CHANGES);
        if (hasChanges) {
            messages.add(format("quota.request.preorder.ticket.resource.change.comment", ctx.getPerson(),
                    formatResourcesDiff(originalRequest, request)));
        }
        if (!hasChanges || changedFields.size() > 1) {
            messages.add(format("quota.request.preorder.ticket.description.change.comment", ctx.getPerson()));
        }
        return CommentCreate.builder()
                .comment(StringUtils.join(messages, "\n\n"))
                .build();
    }

    private CommentCreate getStatusChangeRequestComment(final QuotaChangeRequest.Status status, @Nullable final String summonee,
                                                        final QuotaChangeRequest request, final PerformerContext ctx,
                                                        final boolean suppressSummon) {
        final CommentCreate.Builder builder = CommentCreate.builder();
        if (summonee != null) {
            // Do not summon author on transition to ready for review or cancelled or completed for resource preorder
            if (!suppressSummon && (status != READY_FOR_REVIEW
                    && status != QuotaChangeRequest.Status.CANCELLED
                    && status != QuotaChangeRequest.Status.COMPLETED)) {
                builder.summonees(summonee);
            }
        }

        final String statusTitle = getStatusTitle(status);
        return builder
                .comment(format("quota.request.preorder.ticket.status.change.comment", ctx.getPerson(), statusTitle))
                .build();
    }

    private CommentCreate getStatusChangeRequestCommentForReadyForReview(final QuotaChangeRequest request, final String assignee,
                                                                         final PerformerContext ctx, final boolean suppressSummon) {
        final String goalDataAnswersForComment = getAnswersForComment(request);
        CommentCreate.Builder commentBuilder = CommentCreate.builder()
                .comment(format("quota.request.preorder.ticket.status.change.READY_FOR_REVIEW.comment", ctx.getPerson(), goalDataAnswersForComment));
        if (!suppressSummon) {
            commentBuilder.summonees(assignee);
        }
        return commentBuilder.build();
    }

    private String prepareChartLinks(final QuotaChangeRequest quotaChangeRequest) {
        final List<String> chartLinks = quotaChangeRequest.getChartLinks();
        return chartLinks.isEmpty()
                ? "" : chartLinks.stream().map(link -> String.format("((%s))", link)).collect(Collectors.joining(", "));
    }

    public boolean sendDescriptionChangedComment(final QuotaChangeRequest request, final String changedBy, final String author,
                                                 final boolean suppressSummon) {
        if (request.getCampaign() == null || request.getCampaign().isDeleted()
                || (request.getCampaign().getStatus() != Campaign.Status.ACTIVE && !request.getCampaign().isAllowedRequestModificationWhenClosed())) {
            return false;
        }

        final boolean hasValidOrders = request.getChanges().stream()
                .anyMatch(c -> c.getKey().getBigOrder() != null && c.getKey().getBigOrder().isInCampaign());

        if (!hasValidOrders) {
            return false;
        }

        final String commentText = format("quota.request.ticket.descriptionChangedComment", changedBy, abcFrontUrl,
                request.getProject().getKey().getPublicKey(), Long.toString(request.getId()));

        final CommentCreate.Builder commentBuilder = CommentCreate.builder()
                .comment(commentText);
        if (!suppressSummon) {
            commentBuilder.summonees(author);
        }

        //noinspection ConstantConditions
        trackerManager.createComment(request.getTrackerIssueKey(), commentBuilder.build());

        return true;
    }

    public void tryAddIssueComment(final QuotaChangeRequest request, final String commentText) {
        try {
            final CommentCreate comment = CommentCreate.builder()
                    .comment(commentText)
                    .build();
            if (request.getTrackerIssueKey() != null) {
                trackerManager.createComment(request.getTrackerIssueKey(), comment);
            }
        } catch (RuntimeException e) {
            LOG.error("Can't add issue comment: " + request, e);
        }
    }


    @NotNull
    private static Optional<Project> getSourceProject(final QuotaChangeRequest request) {
        if (request.getSourceProject() != null) {
            return Optional.of(request.getSourceProject());
        }
        if (request.getProject().isRoot()) {
            return Optional.empty();
        }
        return Optional.of(request.getProject().getParent());
    }

    @NotNull
    public String getAnswersForComment(final QuotaChangeRequest request) {
        return getDefendAnswersList(request).stream()
                .map(answer -> String.format("%s\n%s\n", answer.get(0), answer.get(1)))
                .collect(Collectors.joining("\n"));
    }

    private String fillDeliveryDates(final QuotaChangeRequest quotaChangeRequest) {
        final List<LocalDate> bigOrdersDates = quotaChangeRequest.getChanges().stream()
                .map(c -> c.getKey().getBigOrder())
                .filter(Objects::nonNull)
                .map(QuotaChangeRequest.BigOrder::getDate)
                .distinct()
                .sorted()
                .collect(Collectors.toList());

        if (!bigOrdersDates.isEmpty()) {
            final boolean sameYear = bigOrdersDates.stream().allMatch(date -> date.getYear() == bigOrdersDates.get(0).getYear());
            if (sameYear) {
                final Set<Month> usedMonth = new HashSet<>();
                final StringBuilder sb = new StringBuilder();

                final int size = bigOrdersDates.size();
                for (int i = 0; i < size - 1; i++) {
                    if (usedMonth.add(bigOrdersDates.get(i).getMonth())) {
                        if (i != 0) {
                            sb.append(", ");
                        }
                        sb.append(StringUtils.capitalize(bigOrdersDates.get(i).getMonth().getDisplayName(FULL_STANDALONE, RUSSIAN_LOCALE)));
                    }
                }

                if (usedMonth.add(bigOrdersDates.get(size - 1).getMonth())) {
                    if (size > 1) {
                        sb.append(", ");
                    }
                    sb.append(StringUtils.capitalize(bigOrdersDates.get(size - 1).format(FORMATTER_WITH_RU_LOCALE)));
                } else {
                    sb.append(bigOrdersDates.get(size - 1).format(YEAR_FORMATTER));
                }

                return sb.toString();
            } else {
                return bigOrdersDates.stream().sorted()
                        .map(d -> StringUtils.capitalize(d.format(FORMATTER_WITH_RU_LOCALE)))
                        .collect(Collectors.joining(", "));
            }
        }

        return null;
    }
}
