package ru.yandex.qe.dispenser.ws.quota.request.unbalanced;

import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import org.jetbrains.annotations.NotNull;

import ru.yandex.qe.dispenser.api.v1.DiAmount;
import ru.yandex.qe.dispenser.domain.QuotaChangeRequest;
import ru.yandex.qe.dispenser.domain.Segment;

public class QuotaChangeRequestUnbalancedContext {
    private final long requestId;
    private final long campaignId;
    @NotNull
    private final String providerKey;
    @NotNull
    private final List<Change> changes;

    public QuotaChangeRequestUnbalancedContext(long requestId, long campaignId, @NotNull String providerKey,
                                               @NotNull List<Change> changes) {
        this.requestId = requestId;
        this.campaignId = campaignId;
        this.providerKey = providerKey;
        this.changes = changes;
    }

    public long getRequestId() {
        return requestId;
    }

    public long getCampaignId() {
        return campaignId;
    }

    @NotNull
    public String getProviderKey() {
        return providerKey;
    }

    @NotNull
    public List<Change> getChanges() {
        return changes;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        QuotaChangeRequestUnbalancedContext that = (QuotaChangeRequestUnbalancedContext) o;
        return requestId == that.requestId && campaignId == that.campaignId && providerKey.equals(that.providerKey)
                && changes.equals(that.changes);
    }

    @Override
    public int hashCode() {
        return Objects.hash(requestId, campaignId, providerKey, changes);
    }

    @Override
    public String toString() {
        return "QuotaChangeRequestUnbalancedContext{" +
                "requestId=" + requestId +
                ", campaignId=" + campaignId +
                ", providerKey='" + providerKey + '\'' +
                ", changes=" + changes +
                '}';
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder {
        private Long requestId;
        private Long campaignId;
        private String providerKey;
        private List<Change> changes;

        public Builder requestId(Long requestId) {
            this.requestId = requestId;
            return this;
        }

        public Builder campaignId(Long campaignId) {
            this.campaignId = campaignId;
            return this;
        }

        public Builder providerKey(String providerKey) {
            this.providerKey = providerKey;
            return this;
        }

        public Builder changes(List<Change> changes) {
            this.changes = changes;
            return this;
        }

        public Builder changesFromQCR(List<QuotaChangeRequest.Change> changesQRC) {
            this.changes = changesQRC.stream()
                    .map(c -> Change.builder()
                            .orderId(Objects.requireNonNull(c.getBigOrder())
                                    .getId())
                            .resourceKey(c.getResource().getPublicKey())
                            .segmentKeys(c.getSegments())
                            .amount(DiAmount.of(c.getAmount(), c.getResource().getType().getBaseUnit()))
                            .build())
                    .collect(Collectors.toList());
            return this;
        }

        public Builder changesFromCA(List<QuotaChangeRequest.ChangeAmount> changeAmounts) {
            this.changes = changeAmounts.stream()
                    .map(c -> Change.builder()
                            .orderId(Objects.requireNonNull(c.getBigOrder())
                                    .getId())
                            .resourceKey(c.getResource().getPublicKey())
                            .segmentKeys(c.getSegments())
                            .amount(DiAmount.of(c.getAmount(), c.getResource().getType().getBaseUnit()))
                            .build())
                    .collect(Collectors.toList());
            return this;
        }

        public QuotaChangeRequestUnbalancedContext build() {
            return new QuotaChangeRequestUnbalancedContext(
                    Objects.requireNonNull(requestId),
                    Objects.requireNonNull(campaignId),
                    Objects.requireNonNull(providerKey),
                    Objects.requireNonNull(changes)
            );
        }
    }

    public static class Change {
        private final long orderId;
        @NotNull
        private final String resourceKey;
        @NotNull
        private final Set<Segment> segments;
        @NotNull
        private final DiAmount amount;

        public Change(long orderId, @NotNull String resourceKey,
                      @NotNull Set<Segment> segments, @NotNull DiAmount amount) {
            this.orderId = orderId;
            this.resourceKey = resourceKey;
            this.segments = segments;
            this.amount = amount;
        }

        public long getOrderId() {
            return orderId;
        }

        @NotNull
        public String getResourceKey() {
            return resourceKey;
        }

        @NotNull
        public Set<Segment> getSegments() {
            return segments;
        }

        @NotNull
        public DiAmount getAmount() {
            return amount;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            Change change = (Change) o;
            return orderId == change.orderId && resourceKey.equals(change.resourceKey)
                    && segments.equals(change.segments)
                    && amount.equals(change.amount);
        }

        @Override
        public int hashCode() {
            return Objects.hash(orderId, resourceKey, segments, amount);
        }

        @Override
        public String toString() {
            return "Change{" +
                    "orderId=" + orderId +
                    ", resourceKey='" + resourceKey + '\'' +
                    ", segments=" + segments +
                    ", amount=" + amount +
                    '}';
        }

        public static Builder builder() {
            return new Builder();
        }

        public static class Builder {
            private Long orderId;
            private String resourceKey;
            private Set<Segment> segments;
            private DiAmount amount;

            public Builder orderId(Long orderId) {
                this.orderId = orderId;
                return this;
            }

            public Builder resourceKey(String resourceKey) {
                this.resourceKey = resourceKey;
                return this;
            }

            public Builder segmentKeys(Set<Segment> segments) {
                this.segments = segments;
                return this;
            }

            public Builder amount(DiAmount amount) {
                this.amount = amount;
                return this;
            }

            public Change build() {
                return new Change(
                        Objects.requireNonNull(orderId),
                        Objects.requireNonNull(resourceKey),
                        Objects.requireNonNull(segments),
                        Objects.requireNonNull(amount)
                );
            }
        }
    }
}
