package ru.yandex.qe.dispenser.ws.quota.request.unbalanced;


import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import ru.yandex.qe.dispenser.api.v1.request.unbalance.DiQuotaChangeRequestUnbalancedContext;
import ru.yandex.qe.dispenser.domain.QuotaChangeRequest;
import ru.yandex.qe.dispenser.domain.Resource;
import ru.yandex.qe.dispenser.domain.Segment;
import ru.yandex.qe.dispenser.domain.Service;
import ru.yandex.qe.dispenser.domain.dao.segment.SegmentUtils;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;
import ru.yandex.qe.dispenser.ws.quota.request.unbalanced.formula.UnbalancedFormula;
import ru.yandex.qe.dispenser.ws.quota.request.workflow.context.CreateRequestContext;

import static ru.yandex.qe.dispenser.domain.util.ValidationUtils.requireNonNullFieldLocalized;
import static ru.yandex.qe.dispenser.domain.util.ValidationUtils.requireNotEmptyFieldLocalized;
import static ru.yandex.qe.dispenser.domain.util.ValidationUtils.validateChangeBodies;

/**
 * Manager for calculating unbalance of quota requests.
 *
 * @author Ruslan Kadriev <aqru@yandex-team.ru>
 */
@Component
public class QuotaChangeUnbalancedManager {
    private static final Logger LOG = LoggerFactory.getLogger(QuotaChangeUnbalancedManager.class);
    private final Set<Long> campaignIds;
    private final Map<String, UnbalancedFormula> formulaByProviderKey;

    private boolean isNotAllowedCampaign(QuotaChangeRequest.@Nullable Campaign campaign) {
        return campaign == null || !campaignIds.contains(campaign.getId());
    }

    public QuotaChangeUnbalancedManager(@Value("#{${quota.request.unbalanced.campaign.ids}}") List<Long> campaignIdSet,
                                        List<UnbalancedFormula> unbalancedFormulas) {
        this.campaignIds = ImmutableSet.copyOf(campaignIdSet);
        this.formulaByProviderKey = ImmutableMap.copyOf(unbalancedFormulas.stream()
                .collect(Collectors.toMap(UnbalancedFormula::getProviderKey, Function.identity())));
    }

    @NotNull
    public Collection<QuotaChangeRequestUnbalancedResult> calculateFromRequests(
            @NotNull Collection<QuotaChangeRequest> quotaChangeRequests) {
        return quotaChangeRequests.stream()
                .map(this::fromQuotaChangeRequest)
                .flatMap(c -> calculate(c).stream())
                .collect(Collectors.toList());
    }

    private List<QuotaChangeRequestUnbalancedContext> fromQuotaChangeRequest(QuotaChangeRequest quotaChangeRequest) {
        Map<String, List<QuotaChangeRequest.Change>> changesByServiceKey = quotaChangeRequest.getChanges().stream()
                .collect(Collectors.groupingBy(c -> c.getResource().getService().getKey()));
        return  changesByServiceKey.entrySet().stream()
                .map(e -> QuotaChangeRequestUnbalancedContext.builder()
                        .requestId(quotaChangeRequest.getId())
                        .campaignId(quotaChangeRequest.getCampaignId())
                        .providerKey(e.getKey())
                        .changesFromQCR(e.getValue())
                        .build())
                .collect(Collectors.toList());
    }

    @NotNull
    public Collection<QuotaChangeRequestUnbalancedResult> calculate(
            @NotNull Collection<QuotaChangeRequestUnbalancedContext> quotaChangeRequestUnbalancedContext) {
        return quotaChangeRequestUnbalancedContext.stream()
                .map(this::calculate)
                .collect(Collectors.toList());
    }

    /**
     * Calculate unbalance for new quota request.
     * @param ctx  new quota request context
     * @return is unbalance in new quota request
     */
    public boolean calculateForNewRequest(
            CreateRequestContext ctx) {
        if (isNotAllowedCampaign(ctx.getCampaign())) {
            return false;
        }

        Map<String, List<QuotaChangeRequest.Change>> changesByServiceKey = ctx.getChanges().stream()
                .collect(Collectors.groupingBy(c -> c.getResource().getService().getKey()));
        return changesByServiceKey.entrySet().stream()
                .anyMatch(e -> {
                    QuotaChangeRequestUnbalancedContext quotaChangeRequestUnbalancedContext =
                            QuotaChangeRequestUnbalancedContext.builder()
                                    .requestId(-1L)
                                    .campaignId(Objects.requireNonNull(ctx.getCampaign())
                                            .getId())
                                    .providerKey(e.getKey())
                                    .changesFromQCR(e.getValue())
                                    .build();

                    return calculate(quotaChangeRequestUnbalancedContext).isUnbalanced();
                });
    }

    /**
     * Calculate unbalance for update of quota request.
     * @param oldRequest of updated request
     * @param newChanges from updated request
     * @return is unbalance in updated quota request
     */
    public boolean calculateForUpdateRequest(QuotaChangeRequest oldRequest,
                                             List<? extends QuotaChangeRequest.ChangeAmount> newChanges) {
        if (isNotAllowedCampaign(oldRequest.getCampaign())) {
            return false;
        }

        Map<String, List<QuotaChangeRequest.ChangeAmount>> changesByServiceKey = newChanges.stream()
                .collect(Collectors.groupingBy(c -> c.getResource().getService().getKey()));

        return changesByServiceKey.entrySet().stream()
                .anyMatch(e -> {
                    QuotaChangeRequestUnbalancedContext quotaChangeRequestUnbalancedContext =
                            QuotaChangeRequestUnbalancedContext.builder()
                                    .requestId(oldRequest.getId())
                                    .campaignId(Objects.requireNonNull(oldRequest.getCampaign())
                                            .getId())
                                    .providerKey(e.getKey())
                                    .changesFromCA(e.getValue())
                                    .build();

                    return calculate(quotaChangeRequestUnbalancedContext).isUnbalanced();
                });
    }

    /**
     * Calculate unbalance for request context.
     * @param quotaChangeRequestUnbalancedContext  calculation context
     * @return calculation result
     */
    @NotNull
    public QuotaChangeRequestUnbalancedResult calculate(
            @NotNull QuotaChangeRequestUnbalancedContext quotaChangeRequestUnbalancedContext) {
        String providerKey = quotaChangeRequestUnbalancedContext.getProviderKey();
        if (!campaignIds.contains(quotaChangeRequestUnbalancedContext.getCampaignId())) {
            return QuotaChangeRequestUnbalancedResult.builder()
                    .requestId(quotaChangeRequestUnbalancedContext.getRequestId())
                    .providerKey(providerKey)
                    .unbalanced(false)
                    .changes(List.of())
                    .build();
        }

        if (!formulaByProviderKey.containsKey(providerKey)) {
            return QuotaChangeRequestUnbalancedResult.builder()
                    .requestId(quotaChangeRequestUnbalancedContext.getRequestId())
                    .providerKey(providerKey)
                    .unbalanced(false)
                    .changes(List.of())
                    .build();
        }

        UnbalancedFormula unbalancedFormula = formulaByProviderKey.get(providerKey);
        QuotaChangeRequestUnbalancedResult calculate;
        try {
            calculate = unbalancedFormula.calculate(quotaChangeRequestUnbalancedContext);
        } catch (Exception e) {
            LOG.warn("Error while calculating unbalance for context: " + quotaChangeRequestUnbalancedContext, e);
            calculate = QuotaChangeRequestUnbalancedResult.builder()
                    .requestId(quotaChangeRequestUnbalancedContext.getRequestId())
                    .providerKey(providerKey)
                    .unbalanced(false)
                    .changes(List.of())
                    .build();
        }
        return calculate;
    }

    public Set<Long> getCampaignIds() {
        return campaignIds;
    }

    public QuotaChangeRequestUnbalancedResult calculate(DiQuotaChangeRequestUnbalancedContext body) {
        validateDiQuotaChangeRequestUnbalancedContext(body);
        List<QuotaChangeRequestUnbalancedContext.Change> changes = toChanges(body);
        QuotaChangeRequestUnbalancedContext quotaChangeRequestUnbalancedContext = QuotaChangeRequestUnbalancedContext.builder()
                .requestId(-1L)
                .campaignId(body.getCampaignId())
                .providerKey(body.getProviderKey())
                .changes(changes)
                .build();

        return calculate(quotaChangeRequestUnbalancedContext);
    }

    private void validateDiQuotaChangeRequestUnbalancedContext(DiQuotaChangeRequestUnbalancedContext body) {
        requireNonNullFieldLocalized(body.getProviderKey(), "providerKey");
        requireNonNullFieldLocalized(body.getCampaignId(), "campaignId");
        requireNonNullFieldLocalized(body.getChanges(), "changes");
        requireNotEmptyFieldLocalized(body.getChanges(), "changes");
        validateChangeBodies(body.getChanges());
        body.getChanges().forEach(changeBody -> requireNonNullFieldLocalized(changeBody.getOrderId(), "change.orderId"));
    }

    private List<QuotaChangeRequestUnbalancedContext.Change> toChanges(DiQuotaChangeRequestUnbalancedContext body) {
        List<QuotaChangeRequestUnbalancedContext.Change> changes = new ArrayList<>();

        String providerKey = Objects.requireNonNull(body.getProviderKey());
        final Service service = Hierarchy.get().getServiceReader().read(providerKey);
        requireNonNullFieldLocalized(service, "providerKey");
        Objects.requireNonNull(body.getChanges()).forEach(changeBody -> {
            String resourceKey = Objects.requireNonNull(changeBody.getResourceKey());
            final Resource resource = Hierarchy.get().getResourceReader().read(new Resource.Key(resourceKey, service));
            requireNonNullFieldLocalized(resource, "change.resourceKey");
            final Set<Segment> segments = SegmentUtils.getCompleteSegmentSet(resource, Objects.requireNonNull(changeBody.getSegmentKeys()));

            QuotaChangeRequestUnbalancedContext.Change change = QuotaChangeRequestUnbalancedContext.Change.builder()
                    .orderId(changeBody.getOrderId())
                    .resourceKey(resourceKey)
                    .segmentKeys(segments)
                    .amount(changeBody.getAmount())
                    .build();
            changes.add(change);
        });

        return changes;
    }
}
