package ru.yandex.qe.dispenser.ws.quota.request.unbalanced.formula;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.math.MathContext;
import java.math.RoundingMode;

import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;

import ru.yandex.qe.dispenser.api.v1.DiUnit;
import ru.yandex.qe.dispenser.ws.quota.request.unbalanced.QuotaChangeRequestUnbalancedContext;
import ru.yandex.qe.dispenser.ws.quota.request.unbalanced.QuotaChangeRequestUnbalancedResult;

/**
 * Interface for per-provider unbalance calculation formula
 *
 * @author Ruslan Kadriev <aqru@yandex-team.ru>
 */
public interface UnbalancedFormula {
    /**
     * Math context, that should be used for all calculation in implemented formulas.
     */
    MathContext MATH_CONTEXT = new MathContext(18, RoundingMode.HALF_UP);

    /**
     * Calculate unbalance for quota request context.
     * @param quotaChangeRequestUnbalancedContext of quota request
     * @return calculation result
     */
    @NotNull
    QuotaChangeRequestUnbalancedResult calculate(@NotNull QuotaChangeRequestUnbalancedContext quotaChangeRequestUnbalancedContext);

    /**
     * Used for mapping changes of provider to its formula.
     * @return formulas provider key
     */
    @NotNull
    String getProviderKey();

    /**
     * Converting change to unit with math context and fail-safe.
     * @param change to convert
     * @param to unit to convert change
     * @param logger for logging fail, if occurred
     * @return {@link BigDecimal} with converted value, or {@link BigDecimal#ZERO} on fail
     */
    default BigDecimal convert(QuotaChangeRequestUnbalancedContext.Change change, final DiUnit to, Logger logger) {
        DiUnit unit = change.getAmount().getUnit();

        return convert(new BigDecimal(change.getAmount().getValue(), MATH_CONTEXT), unit, to, logger);
    }

    default BigDecimal convert(BigInteger amount, final DiUnit from, final DiUnit to, Logger logger) {
        return convert(new BigDecimal(amount), from , to, logger);
    }

    default BigDecimal convert(BigDecimal amount, final DiUnit from, final DiUnit to, Logger logger) {
        BigDecimal result = BigDecimal.ZERO;
        try {
            result = to.convert(amount, from, MATH_CONTEXT);
        } catch (Exception e) {
            logger.warn("Failed convert " + amount + ", from " + from + " unit to " + to + " unit!", e);
        }

        return result;
    }
}
