package ru.yandex.qe.dispenser.ws.quota.request.workflow.context;

import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import com.google.common.collect.Sets;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import ru.yandex.qe.dispenser.api.v1.DiResourcePreorderReasonType;
import ru.yandex.qe.dispenser.api.v1.request.quota.BodyUpdate;
import ru.yandex.qe.dispenser.domain.BotCampaignGroup;
import ru.yandex.qe.dispenser.domain.Person;
import ru.yandex.qe.dispenser.domain.QuotaChangeRequest;
import ru.yandex.qe.dispenser.domain.dao.goal.Goal;
import ru.yandex.qe.dispenser.ws.quota.request.workflow.ResourceWorkflow;

public class UpdateResourceRequestContext extends UpdateRequestContext {

    public UpdateResourceRequestContext(final Person person,
                                        final QuotaChangeRequest quotaChangeRequest,
                                        final List<? extends QuotaChangeRequest.ChangeAmount> changes,
                                        final BodyUpdate bodyUpdate,
                                        @Nullable final Goal goal,
                                        @Nullable final QuotaChangeRequest.Campaign campaign,
                                        @Nullable final BotCampaignGroup campaignGroup) {
        super(person, quotaChangeRequest, changes, bodyUpdate, goal, campaign, campaignGroup);
    }

    @Override
    protected void prepareFields(final @NotNull QuotaChangeRequest quotaChangeRequest,
                                 final @NotNull BodyUpdate bodyUpdate,
                                 final @NotNull QuotaChangeRequest.Builder updatedRequestBuilder) {

        final DiResourcePreorderReasonType resourcePreorderReasonType = bodyUpdate.getResourcePreorderReasonType();
        if (isFieldChanged(resourcePreorderReasonType, quotaChangeRequest.getResourcePreorderReasonType())) {
            updatedRequestBuilder.resourcePreorderReasonType(resourcePreorderReasonType);
            changedFields.add(QuotaChangeRequest.Field.RESOURCE_PREORDER_REASON_TYPE);
        }

        if (!Objects.equals(getGoal(), quotaChangeRequest.getGoal())) {
            updatedRequestBuilder.goal(getGoal());
            changedFields.add(QuotaChangeRequest.Field.GOAL_ID);
        }

        final Map<Long, String> requestGoalAnswers = bodyUpdate.getRequestGoalAnswers();
        if (isFieldChanged(requestGoalAnswers, quotaChangeRequest.getRequestGoalAnswers())) {
            updatedRequestBuilder.requestGoalAnswers(requestGoalAnswers);
            changedFields.add(QuotaChangeRequest.Field.REQUEST_GOAL_ANSWERS);
        }


        super.prepareFields(quotaChangeRequest, bodyUpdate, updatedRequestBuilder);

        final Set<QuotaChangeRequest.Field> sensibleFields = Sets.difference(changedFields, ResourceWorkflow.NOT_SENSIBLE_FIELDS);
        if (!sensibleFields.isEmpty()) {
            final boolean onlyChangesUpdate = sensibleFields.equals(EnumSet.of(QuotaChangeRequest.Field.CHANGES));
            final boolean transitionApplied = transitionOnUpdate(quotaChangeRequest, updatedRequestBuilder, onlyChangesUpdate);
            if (transitionApplied) {
                changedFields.add(QuotaChangeRequest.Field.STATUS);
            }
        }


    }


    /**
     * Add changes in {@code updateBuilder} for {@code request}, if needed
     *
     * @param request       to be check
     * @param updateBuilder of request
     * @return {@code true} if new changes was added to {@code updateBuilder}, otherwise {@code false}
     */
    private boolean transitionOnUpdate(final QuotaChangeRequest request, final QuotaChangeRequest.Builder updateBuilder, final boolean onlyChangesUpdate) {

        final QuotaChangeRequest.Status newStatus;

        switch (request.getStatus()) {
            case CONFIRMED:
            case APPROVED:
                if (onlyChangesUpdate && onlyDecrease) {
                    return false;
                }
                newStatus = QuotaChangeRequest.Status.READY_FOR_REVIEW;
                break;
            case CANCELLED:
            case REJECTED:
                newStatus = QuotaChangeRequest.Status.NEW;
                break;
            default:
                return false;
        }

        updateBuilder.status(newStatus);
        return true;
    }
}
