package ru.yandex.qe.dispenser.ws.quota.request.workflow.service;

import java.time.format.DateTimeFormatter;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.inject.Inject;

import org.jetbrains.annotations.NotNull;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import ru.yandex.qe.dispenser.api.v1.DiCampaign;
import ru.yandex.qe.dispenser.domain.QuotaChangeRequest;
import ru.yandex.qe.dispenser.domain.Resource;
import ru.yandex.qe.dispenser.domain.Segment;
import ru.yandex.qe.dispenser.domain.Service;
import ru.yandex.qe.dispenser.domain.dictionaries.impl.FrontDictionariesManager;
import ru.yandex.qe.dispenser.domain.dictionaries.model.CampaignProvidersSettingsDictionary;
import ru.yandex.qe.dispenser.domain.exception.SingleMessageException;
import ru.yandex.qe.dispenser.ws.quota.request.workflow.context.RequestContext;

@Component
@ParametersAreNonnullByDefault
public class ServiceDictionaryRestrictionManager {

    @Inject
    private FrontDictionariesManager frontDictionariesManager;

    public void checkRequestByDictionary(
            final RequestContext context,
            final List<? extends QuotaChangeRequest.ChangeAmount> changes) {

        final CampaignProvidersSettingsDictionary campaignProvidersSettingsDictionary =
                frontDictionariesManager.getCampaignProviderSettings(context.getCampaign().getId());

        if (campaignProvidersSettingsDictionary == null) {
            throw SingleMessageException.illegalArgument("settings.for.campaign.not.found");
        }

        final Map<@NotNull Long, @NotNull Long> bigOrderDictionaryIdByBigOrderId =
                campaignProvidersSettingsDictionary.getCampaign().getCampaignBigOrders()
                        .stream().collect(Collectors.toMap(DiCampaign.BigOrder::getBigOrderId, DiCampaign.BigOrder::getId));

        final Map<Service, ? extends List<? extends QuotaChangeRequest.ChangeAmount>> changesByService = changes.stream()
                .collect(Collectors.groupingBy(c -> c.getResource().getService()));

        final Set<CampaignProvidersSettingsDictionary.Resource> dictionaryResourcesInRequest = new HashSet<>();

        for (final Service service : changesByService.keySet()) {
            final CampaignProvidersSettingsDictionary.Provider dictionaryProvider =
                    getDictionaryProviderForService(campaignProvidersSettingsDictionary, service);

            final List<? extends QuotaChangeRequest.ChangeAmount> changeAmounts = changesByService.get(service);
            for (final QuotaChangeRequest.ChangeAmount changeAmount : changeAmounts) {
                final CampaignProvidersSettingsDictionary.Resource dictionaryResource =
                        getDictionaryResource(changeAmount.getResource(), dictionaryProvider);

                checkChange(changeAmount, campaignProvidersSettingsDictionary, bigOrderDictionaryIdByBigOrderId, dictionaryResource);
                dictionaryResourcesInRequest.add(dictionaryResource);
            }

            final List<CampaignProvidersSettingsDictionary.Resource> requiredNotInRequestResources = dictionaryProvider.getResources().stream()
                    .filter(resource -> resource.isRequired() && !dictionaryResourcesInRequest.contains(resource)).collect(Collectors.toList());

            if (!requiredNotInRequestResources.isEmpty()) {
                throw SingleMessageException.illegalArgument("required.resource.for.service.are.not.in.request", requiredNotInRequestResources.stream()
                                .map(CampaignProvidersSettingsDictionary.Resource::getName).collect(Collectors.joining(", ", "(", ")")),
                        service.getName());
            }
        }

    }

    private void checkChange(
            final QuotaChangeRequest.ChangeAmount changeAmount,
            final CampaignProvidersSettingsDictionary campaignProvidersSettingsDictionary,
            final Map<@NotNull Long, @NotNull Long> bigOrderDictionaryIdByBigOrderId,
            final CampaignProvidersSettingsDictionary.Resource dictionaryResource) {

        final @NotNull Service service = changeAmount.getResource().getService();
        final QuotaChangeRequest.BigOrder bigOrder = changeAmount.getKey().getBigOrder();

        if (bigOrder == null) {
            throw SingleMessageException.illegalArgument("resource.change.for.service.has.no.big.order", changeAmount.getResource().getName(), service.getName());
        }

        if (!bigOrderDictionaryIdByBigOrderId.containsKey(bigOrder.getId())) {
            throw SingleMessageException.illegalArgument("big.order.in.resource.change.for.service.is.not.in.campaign",
                    bigOrder.getDate().format(DateTimeFormatter.ISO_LOCAL_DATE), changeAmount.getResource().getName(), service.getName());
        }

        final Map<String, Long> dictionarySegmentsIdByKey = getDictionarySegmentsIdByKey(dictionaryResource.getSegmentations(),
                campaignProvidersSettingsDictionary.getSegmentations());

        boolean checkedBySegmentedBigOrders = false;

        final Long bigOrderDictionaryId = bigOrderDictionaryIdByBigOrderId.get(bigOrder.getId());
        Set<Long> segmentIds = changeAmount.getSegments().stream()
                .map(segment -> {
                    final Long segmentId = dictionarySegmentsIdByKey.get(segment.getPublicKey());
                    if (segmentId == null) {
                        throw SingleMessageException.illegalArgument("segment.not.allowed.in.resource.for.service",
                                segment.getName(), changeAmount.getResource().getName(), service.getName());
                    }
                    return segmentId;
                })
                .collect(Collectors.toSet());
        if (!CollectionUtils.isEmpty(dictionaryResource.getSegmentedBigOrders())) {
            checkedBySegmentedBigOrders = dictionaryResource.getSegmentedBigOrders().stream()
                    .filter(segmentsBigOrders -> segmentIds.containsAll(segmentsBigOrders.getSegments()))
                    .peek(segmentsBigOrders -> {
                        if (!segmentsBigOrders.getBigOrders().contains(bigOrderDictionaryId)) {
                            String segmentsName = changeAmount.getSegments().stream()
                                    .map(Segment::getName)
                                    .collect(Collectors.joining(", ", "(", ")"));
                            throw SingleMessageException.illegalArgument(
                                    "big.order.not.allowed.for.segment.in.resource.for.service",
                                    bigOrder.getId(), segmentsName, changeAmount.getResource().getName(),
                                    service.getName());
                        }
                    }).count() > 0;
        }

        if (!checkedBySegmentedBigOrders && !CollectionUtils.isEmpty(dictionaryResource.getBigOrders()) &&
                !dictionaryResource.getBigOrders().contains(bigOrderDictionaryId)) {
            throw SingleMessageException.illegalArgument("big.order.in.resource.change.for.service.is.not.in.segmented.orders.for.campaign",
                    bigOrder.getId(), changeAmount.getResource().getName(), service.getName());
        }
    }

    private CampaignProvidersSettingsDictionary.Provider getDictionaryProviderForService(
            final CampaignProvidersSettingsDictionary campaignProvidersSettingsDictionary, final Service service) {
        final List<CampaignProvidersSettingsDictionary.Provider> providers =
                campaignProvidersSettingsDictionary.getProviders().stream().filter(provider -> service.getId() ==
                        provider.getId()).collect(Collectors.toList());
        if (providers.size() == 0) {
            throw SingleMessageException.illegalArgument("service.is.not.in.campaign", service.getName());
        }
        if (providers.size() > 1) {
            //такого быть не должно
            throw SingleMessageException.unknown("conflict.for.service.in.campaign", service.getName());
        }
        return providers.get(0);
    }

    private CampaignProvidersSettingsDictionary.Resource getDictionaryResource(
            final Resource resource, final CampaignProvidersSettingsDictionary.Provider provider) {
        final @NotNull Service service = resource.getService();
        final List<CampaignProvidersSettingsDictionary.Resource> dictionaryResources = provider.getResources().stream()
                .filter(res -> res.getId() == resource.getId()).collect(Collectors.toList());
        if (dictionaryResources.size() == 0) {
            throw SingleMessageException.illegalArgument("resource.for.service.is.not.in.campaign", resource.getName(), service.getName());
        }
        if (dictionaryResources.size() > 1) {
            //такого быть не должно
            throw SingleMessageException.unknown("conflict.for.resource.in.campaign", resource.getName(), service.getName());
        }

        return dictionaryResources.get(0);
    }

    private Map<String, Long> getDictionarySegmentsIdByKey(final List<CampaignProvidersSettingsDictionary.ResourceSegmentation> resourceSegmentations,
                                                           final List<CampaignProvidersSettingsDictionary.Segmentation> segmentations) {
        final Map<String, Long> result = new HashMap<>();

        if (!CollectionUtils.isEmpty(resourceSegmentations)) {
            for (final CampaignProvidersSettingsDictionary.ResourceSegmentation resourceSegmentation : resourceSegmentations) {
                final CampaignProvidersSettingsDictionary.Segmentation segmentation =
                        segmentations.stream().filter(s -> s.getId() == resourceSegmentation.getId()).findAny().orElseThrow(() ->
                                SingleMessageException.unknown("wrong.dictionary.resource.segmententation.not.in.segmentations", resourceSegmentation.getId()));

                if (!CollectionUtils.isEmpty(resourceSegmentation.getSegments())) {
                    resourceSegmentation.getSegments().stream().map(id ->
                            segmentation.getSegments().stream().filter(s -> s.getId() == id).findAny().orElseThrow(
                                    () -> SingleMessageException.unknown("wrong.dictionary.resource.segment.not.in.segmentation", id)
                            )).forEach(segment -> result.put(segment.getKey(), segment.getId()));
                } else {
                    segmentation.getSegments().forEach(segment -> result.put(segment.getKey(), segment.getId()));
                }
            }
        } else {
            segmentations.stream().flatMap(segmentation -> segmentation.getSegments().stream())
                    .forEach(segment -> result.put(segment.getKey(), segment.getId()));
        }

        return result;
    }
}
