package ru.yandex.qe.dispenser.ws.quota.request.workflow.service;

import java.util.Map;
import java.util.Set;
import java.util.regex.Pattern;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;

import ru.yandex.qe.dispenser.domain.exception.SingleMessageException;
import ru.yandex.qe.dispenser.ws.quota.request.workflow.context.RequestContext;

import static org.apache.commons.lang3.StringUtils.trimToEmpty;

@ParametersAreNonnullByDefault
public class SolomonPropertyValidator extends SimpleServicePropertyValidator {

    private static final String PROJECT = "project";
    private static final Set<String> REQUIRED_PROPERTIES = ImmutableSet.of(PROJECT);
    private static final Pattern VALID_PROJECTS_PATTERN = Pattern.compile("^[a-zA-Z0-9_][a-zA-Z0-9_-]{0,54}(,\\s*[a-zA-Z0-9_][a-zA-Z0-9_-]{0,54})*$");

    public SolomonPropertyValidator() {
        super(REQUIRED_PROPERTIES);
    }

    @Override
    public void validateAdditionalPropertiesOnCreate(final RequestContext context, final Map<String, String> additionalProperties) {
        super.validateAdditionalPropertiesOnCreate(context, additionalProperties);
        validateProject(additionalProperties);
    }

    @Override
    public void validateAdditionalPropertiesOnUpdate(final RequestContext context, final Map<String, String> additionalProperties,
                                                     final Map<String, String> currentAdditionalProperties) {
        super.validateAdditionalPropertiesOnUpdate(context, additionalProperties, currentAdditionalProperties);
        if (context.getCampaign() == null || !context.getCampaign().isAllowedModificationOnMissingAdditionalFields()) {
            // Base case - do usual validation
            validateProject(additionalProperties);
            return;
        }
        // Special flag is set
        final boolean hasCurrentProject = currentAdditionalProperties.containsKey(PROJECT)
                && !trimToEmpty(currentAdditionalProperties.get(PROJECT)).isEmpty();
        final boolean hasProject = additionalProperties.containsKey(PROJECT)
                && !trimToEmpty(additionalProperties.get(PROJECT)).isEmpty();
        if (hasProject) {
            // Project is present - do usual validation
            validateProject(additionalProperties);
            return;
        }
        if (!hasCurrentProject) {
            // No project is present, current project value is also missing, nothing to validate
            return;
        }
        // Otherwise - it's invalid update
        throw SingleMessageException.illegalArgument("solomon.project.is.required");
    }

    private void validateProject(Map<String, String> additionalProperties) {
        final String projectValue = additionalProperties.getOrDefault(PROJECT, "");
        if (!VALID_PROJECTS_PATTERN.matcher(projectValue).matches()) {
            throw SingleMessageException.illegalArgument("solomon.invalid.project.property");
        }
    }

}
