package ru.yandex.qe.dispenser.ws.quota.request.workflow.service;

import java.util.Collections;
import java.util.Map;
import java.util.Set;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.qe.dispenser.domain.exception.SingleMessageException;
import ru.yandex.qe.dispenser.ws.quota.request.workflow.context.RequestContext;

import static org.apache.commons.lang3.StringUtils.trimToEmpty;

@ParametersAreNonnullByDefault
public class YtPropertyValidator extends SimpleServicePropertyValidator {

    private static final Pattern VALID_PROPERTIES_PATTERN = Pattern.compile("^[\\w-]+(,\\s*[\\w-]+)*$");

    private static final String POOLS = "pools";
    private static final String BAD_POOLS = "pulls"; // TODO remove me when UI ready
    private static final String ACCOUNTS = "accounts";
    private static final String TABLET_CELL_BUNDLES = "tabletCellBundles";

    private static final Set<String> OPTIONAL_PROPERTIES = ImmutableSet.of(POOLS, ACCOUNTS, TABLET_CELL_BUNDLES);

    public YtPropertyValidator() {
        super(Collections.emptySet(), OPTIONAL_PROPERTIES);
    }

    @Override
    public void validateAdditionalPropertiesOnCreate(final RequestContext context, final Map<String, String> additionalProperties) {
        if (additionalProperties.containsKey(BAD_POOLS)) { // TODO remove me when UI ready
            additionalProperties.put(POOLS, additionalProperties.remove(BAD_POOLS));
        }
        super.validateAdditionalPropertiesOnCreate(context, additionalProperties);
        checkProperties(additionalProperties);
    }

    @Override
    public void validateAdditionalPropertiesOnUpdate(final RequestContext context, final Map<String, String> additionalProperties,
                                                     final Map<String, String> currentAdditionalProperties) {
        if (additionalProperties.containsKey(BAD_POOLS)) { // TODO remove me when UI ready
            additionalProperties.put(POOLS, additionalProperties.remove(BAD_POOLS));
        }
        super.validateAdditionalPropertiesOnUpdate(context, additionalProperties, currentAdditionalProperties);
        if (context.getCampaign() == null || !context.getCampaign().isAllowedModificationOnMissingAdditionalFields()) {
            // Base case - do usual validation
            checkProperties(additionalProperties);
            return;
        }
        // Special flag is set
        // Skip validation when new value is empty and current value is either missing or empty
        final Set<String> propertiesToSkipCheck = additionalProperties.entrySet().stream()
                .filter(e -> trimToEmpty(e.getValue()).isEmpty()
                        && (!currentAdditionalProperties.containsKey(e.getKey())
                        || trimToEmpty(currentAdditionalProperties.get(e.getKey())).isEmpty()))
                .map(Map.Entry::getKey)
                .collect(Collectors.toSet());
        final Map<String, String> filteredProperties = additionalProperties.entrySet().stream()
                .filter(e -> !propertiesToSkipCheck.contains(e.getKey()))
                .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
        if (!filteredProperties.isEmpty()) {
            checkProperties(filteredProperties);
        }

    }

    private void checkProperties(final Map<String, String> additionalProperties) {
        final Set<String> wrongProperties = additionalProperties.entrySet().stream()
                .filter(entry -> !VALID_PROPERTIES_PATTERN.matcher(entry.getValue()).matches())
                .map(Map.Entry::getKey)
                .collect(Collectors.toSet());
        if (!wrongProperties.isEmpty()) {
            throw SingleMessageException.illegalArgument("yt.request.has.invalid.properties.with.valid",
                    StringUtils.join(wrongProperties, ", "), "value1, value2, value3");
        }
    }

}
