package ru.yandex.qe.dispenser.ws.reqbody;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;

@JsonIgnoreProperties(ignoreUnknown = true)
public class AllocationDestinationProvider {

    private final String providerKey;
    private final String providerName;
    private final boolean eligible;
    private final List<String> ineligibilityReasons;
    private final List<AllocationDestinationGroup> destinationGroups;

    @JsonCreator
    public AllocationDestinationProvider(
            @JsonProperty("providerKey") String providerKey,
            @JsonProperty("providerName") String providerName,
            @JsonProperty("eligible") boolean eligible,
            @JsonProperty("ineligibilityReasons") List<String> ineligibilityReasons,
            @JsonProperty("destinationGroups") List<AllocationDestinationGroup> destinationGroups) {
        this.providerKey = providerKey;
        this.providerName = providerName;
        this.eligible = eligible;
        this.ineligibilityReasons = ineligibilityReasons != null ? ineligibilityReasons : List.of();
        this.destinationGroups = destinationGroups;
    }

    public static Builder builder() {
        return new Builder();
    }

    public String getProviderKey() {
        return providerKey;
    }

    public String getProviderName() {
        return providerName;
    }

    public boolean isEligible() {
        return eligible;
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public List<String> getIneligibilityReasons() {
        return ineligibilityReasons;
    }

    public List<AllocationDestinationGroup> getDestinationGroups() {
        return destinationGroups;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        AllocationDestinationProvider that = (AllocationDestinationProvider) o;
        return Objects.equals(providerKey, that.providerKey) &&
                Objects.equals(providerName, that.providerName) &&
                Objects.equals(destinationGroups, that.destinationGroups);
    }

    @Override
    public int hashCode() {
        return Objects.hash(providerKey, providerName, destinationGroups);
    }

    @Override
    public String toString() {
        return "AllocationDestinationProvider{" +
                "providerKey='" + providerKey + '\'' +
                ", providerName='" + providerName + '\'' +
                ", destinationGroups=" + destinationGroups +
                '}';
    }

    public static final class Builder {

        private String providerKey;
        private String providerName;
        private Boolean eligible;
        private final Set<String> ineligibilityReasons = new HashSet<>();
        private final List<AllocationDestinationGroup> destinationGroups = new ArrayList<>();

        private Builder() {
        }

        public Builder providerKey(String providerKey) {
            this.providerKey = providerKey;
            return this;
        }

        public Builder providerName(String providerName) {
            this.providerName = providerName;
            return this;
        }

        public Builder eligible(boolean eligible) {
            this.eligible = eligible;
            return this;
        }

        public Builder addIneligibilityReason(String ineligibilityReason) {
            this.ineligibilityReasons.add(ineligibilityReason);
            return this;
        }

        public Builder addDestinationGroup(AllocationDestinationGroup destinationGroup) {
            this.destinationGroups.add(destinationGroup);
            return this;
        }

        public AllocationDestinationProvider build() {
            Objects.requireNonNull(providerKey, "Provider key is required");
            Objects.requireNonNull(providerName, "Provider name is required");
            Objects.requireNonNull(eligible, "Eligible is required");
            return new AllocationDestinationProvider(providerKey, providerName, eligible,
                    new ArrayList<>(ineligibilityReasons), destinationGroups);
        }

    }

}
