package ru.yandex.qe.dispenser.ws.reqbody;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;

@JsonIgnoreProperties(ignoreUnknown = true)
public class AllocationDestinationQuotaRequest {

    private final long quotaRequestId;
    private final boolean eligible;
    private final List<String> ineligibilityReasons;
    private final List<AllocationDestinationProvider> providers;

    @JsonCreator
    public AllocationDestinationQuotaRequest(@JsonProperty("quotaRequestId") long quotaRequestId,
                                             @JsonProperty("eligible") boolean eligible,
                                             @JsonProperty("ineligibilityReasons") List<String> ineligibilityReasons,
                                             @JsonProperty("providers") List<AllocationDestinationProvider> providers) {
        this.quotaRequestId = quotaRequestId;
        this.eligible = eligible;
        this.ineligibilityReasons = ineligibilityReasons != null ? ineligibilityReasons : List.of();
        this.providers = providers;
    }

    public static Builder builder() {
        return new Builder();
    }

    public long getQuotaRequestId() {
        return quotaRequestId;
    }

    public boolean isEligible() {
        return eligible;
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public List<String> getIneligibilityReasons() {
        return ineligibilityReasons;
    }

    public List<AllocationDestinationProvider> getProviders() {
        return providers;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        AllocationDestinationQuotaRequest that = (AllocationDestinationQuotaRequest) o;
        return quotaRequestId == that.quotaRequestId &&
                eligible == that.eligible &&
                Objects.equals(ineligibilityReasons, that.ineligibilityReasons) &&
                Objects.equals(providers, that.providers);
    }

    @Override
    public int hashCode() {
        return Objects.hash(quotaRequestId, eligible, ineligibilityReasons, providers);
    }

    @Override
    public String toString() {
        return "AllocationDestinationQuotaRequest{" +
                "quotaRequestId=" + quotaRequestId +
                ", eligible=" + eligible +
                ", ineligibilityReasons=" + ineligibilityReasons +
                ", providers=" + providers +
                '}';
    }

    public static final class Builder {

        private Long quotaRequestId;
        private Boolean eligible;
        private final Set<String> ineligibilityReasons = new HashSet<>();
        private final List<AllocationDestinationProvider> providers = new ArrayList<>();

        private Builder() {
        }

        public Builder quotaRequestId(long quotaRequestId) {
            this.quotaRequestId = quotaRequestId;
            return this;
        }

        public Builder eligible(boolean eligible) {
            this.eligible = eligible;
            return this;
        }

        public Builder addIneligibilityReason(String ineligibilityReason) {
            this.ineligibilityReasons.add(ineligibilityReason);
            return this;
        }

        public Builder addProvider(AllocationDestinationProvider provider) {
            this.providers.add(provider);
            return this;
        }

        public AllocationDestinationQuotaRequest build() {
            Objects.requireNonNull(quotaRequestId, "Quota request id is required");
            Objects.requireNonNull(eligible, "Eligible is required");
            return new AllocationDestinationQuotaRequest(quotaRequestId, eligible,
                    new ArrayList<>(ineligibilityReasons), providers);
        }

    }

}
