package ru.yandex.qe.dispenser.ws.resources_model;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.NotNull;
import org.springframework.stereotype.Component;

import ru.yandex.qe.dispenser.domain.QuotaChangeRequest;
import ru.yandex.qe.dispenser.domain.Resource;
import ru.yandex.qe.dispenser.domain.Service;
import ru.yandex.qe.dispenser.domain.dao.resources_model.ResourcesModelMappingDao;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;
import ru.yandex.qe.dispenser.domain.hierarchy.HierarchySupplier;
import ru.yandex.qe.dispenser.domain.resources_model.ResourceModelMappingTarget;
import ru.yandex.qe.dispenser.domain.resources_model.ResourcesModelMapping;
import ru.yandex.qe.dispenser.domain.util.CollectionUtils;

@Component("ydbResourceModelMapper")
public class YdbResourceModelMapper extends SimpleResourceModelMapper {

    public static final String GROUP_COUNT_KEY = "group_count";
    public static final String HOST_COUNT_KEY = "host_count";
    public static final Set<String> RESOURCES_TO_CEIL_KEYS = Set.of(
            GROUP_COUNT_KEY,
            HOST_COUNT_KEY
    );

    private static final String YDB_RU_RPS = "ydb_ru-rps";
    private static final String YDB_RU_USERPOOL_CORES = "ydb_ru-userpool_cores";
    private static final BigDecimal USERPOOL_CORES_PER_RPS = BigDecimal.valueOf(500_000L);
    private static final BigDecimal CORE_AMOUNT = BigDecimal.valueOf(1_000L);

    public YdbResourceModelMapper(ResourcesModelMappingDao mappingDao, HierarchySupplier hierarchySupplier) {
        super(mappingDao, hierarchySupplier);
    }

    @Override
    protected Pair<Map<InternalKey, BigDecimal>,
            List<QuotaChangeRequest.Change>> processChanges(QuotaChangeRequest request,
                                                            Service provider,
                                                            Collection<QuotaChangeRequest.Change> changes,
                                                            Map<InternalResourceKey,
                                                                    List<ResourcesModelMapping>> mappings) {
        final Map<Boolean, List<QuotaChangeRequest.Change>> changesByRpsPresence = changes.stream()
                .filter(eligibleToResourcesModelAllocation(provider))
                .collect(Collectors.groupingBy(c -> c.getResource().getPublicKey().equals(YDB_RU_RPS)));
        final Pair<Map<InternalKey, BigDecimal>, List<QuotaChangeRequest.Change>> defaultResult =
                super.processChanges(request, provider, changesByRpsPresence.getOrDefault(false,
                        Collections.emptyList()), mappings);
        final List<QuotaChangeRequest.Change> rpsChanges = changesByRpsPresence.getOrDefault(true,
                Collections.emptyList());
        final Resource ydbRuUserpoolCores = Hierarchy.get()
                .getResourceReader()
                .readOrNull(new Resource.Key(YDB_RU_USERPOOL_CORES, provider));
        if (rpsChanges.isEmpty() || ydbRuUserpoolCores == null) {
            return defaultResult;
        }

        Map<InternalKey, BigDecimal> internalAmounts = defaultResult.getLeft();
        List<QuotaChangeRequest.Change> affectedChanges = defaultResult.getRight();
        for (QuotaChangeRequest.Change change : rpsChanges) {
            Objects.requireNonNull(change.getBigOrder(), "Change big order is required");
            final Pair<InternalKey, BigDecimal> amountByUserpool = calculateUserpoolCoresFromRps(change,
                    ydbRuUserpoolCores);
            final InternalKey rpsInternalKey = toInternalKey(change);

            InternalResourceKey changeInternalResourceKey = toInternalResourceKey(amountByUserpool.getKey());
            List<ResourcesModelMapping> resourceMappings = mappings
                    .getOrDefault(changeInternalResourceKey, List.of());
            if (resourceMappings.isEmpty()) {
                continue;
            }
            internalAmounts.merge(amountByUserpool.getKey(), amountByUserpool.getValue(), BigDecimal::add);

            BigDecimal changeAmount = BigDecimal.valueOf(change.getAmountReady() - change.getAmountAllocating());
            internalAmounts.merge(rpsInternalKey, changeAmount, BigDecimal::add);
            affectedChanges.add(change);
        }
        return Pair.of(internalAmounts, affectedChanges);
    }

    private Pair<InternalKey, BigDecimal> calculateUserpoolCoresFromRps(QuotaChangeRequest.Change change, Resource ydbRuUserpool) {
        BigDecimal changeAmount = BigDecimal.valueOf(change.getAmountReady() - change.getAmountAllocating());
        final BigDecimal userpoolAmount =
                changeAmount.divide(USERPOOL_CORES_PER_RPS, 0, RoundingMode.UP).multiply(CORE_AMOUNT);
        final InternalKey key = new InternalKey(ydbRuUserpool.getId(), CollectionUtils.ids(change.getSegments()),
                change.getBigOrder().getId());
        return Pair.of(key, userpoolAmount);
    }

    @Override
    @NotNull
    protected BigDecimal calculateExternalAmount(
            BigDecimal internalAmount,
            ResourceModelMappingTarget target,
            RoundingMode rounding
    ) {
        if (target.getKey() == null) {
            return super.calculateExternalAmount(internalAmount, target, rounding);
        }

        if (RESOURCES_TO_CEIL_KEYS.contains(target.getKey())) {
            return internalAmount.multiply(BigDecimal.valueOf(target.getNumerator()))
                    .divide(BigDecimal.valueOf(target.getDenominator()), 0, RoundingMode.CEILING);
        }
        return super.calculateExternalAmount(internalAmount, target, rounding);
    }
}
