package ru.yandex.qe.dispenser.ws.resources_model;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import javax.inject.Inject;

import org.apache.commons.lang3.tuple.Pair;
import org.springframework.stereotype.Component;

import ru.yandex.qe.dispenser.domain.QuotaChangeRequest;
import ru.yandex.qe.dispenser.domain.Resource;
import ru.yandex.qe.dispenser.domain.Segment;
import ru.yandex.qe.dispenser.domain.Service;
import ru.yandex.qe.dispenser.domain.dao.resources_model.ResourcesModelMappingDao;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;
import ru.yandex.qe.dispenser.domain.hierarchy.HierarchySupplier;
import ru.yandex.qe.dispenser.domain.resources_model.ResourcesModelMapping;
import ru.yandex.qe.dispenser.ws.allocation.AllocationHelper;
import ru.yandex.qe.dispenser.ws.allocation.YpResource;
import ru.yandex.qe.dispenser.ws.bot.Provider;

@Component("ypResourceModelMapper")
public class YpResourceModelMapper extends SimpleResourceModelMapper {
    private static final BigDecimal IO_SSD_BPS_PER_CORE = BigDecimal.valueOf(5 * 1024 * 1024);
    private static final BigDecimal IO_HDD_BPS_PER_CORE = BigDecimal.valueOf(2 * 1024 * 1024);
    private static final BigDecimal PER_CORE = BigDecimal.valueOf(1000L);

    private final AllocationHelper allocationHelper;

    @Inject
    public YpResourceModelMapper(ResourcesModelMappingDao resourcesModelMappingDao,
                                 AllocationHelper allocationHelper,
                                 HierarchySupplier hierarchySupplier) {
        super(resourcesModelMappingDao, hierarchySupplier);
        this.allocationHelper = allocationHelper;
    }

    @Override
    protected Pair<Map<InternalKey, BigDecimal>, List<QuotaChangeRequest.Change>> processChanges(QuotaChangeRequest request,
                                                                                                 Service provider,
                                                                                                 Collection<QuotaChangeRequest.Change> changes,
                                                                                                 Map<InternalResourceKey, List<ResourcesModelMapping>> mappings) {
        final boolean hasIoResourcesAvailableInCampaign = allocationHelper
                .hasIoResourcesAvailableInCampaign(provider.getId(), request.getCampaignId());
        if (hasIoResourcesAvailableInCampaign) {
            return super.processChanges(request, provider, changes, mappings);
        }
        final Map<Set<Segment>, List<QuotaChangeRequest.Change>> changesBySegment = changes.stream()
                .filter(SimpleResourceModelMapper.eligibleToResourcesModelAllocation(provider))
                .collect(Collectors.groupingBy(QuotaChangeRequest.ChangeAmount::getSegments));
        final Service yp = Provider.YP.getService();
        Objects.requireNonNull(yp, "Provider YP not found in cache!");
        final Resource ioSsd = Hierarchy.get().getResourceReader().read(new Resource.Key(YpResource.IO_SSD.getKey(), yp));
        final Resource ioHdd = Hierarchy.get().getResourceReader().read(new Resource.Key(YpResource.IO_HDD.getKey(), yp));
        Objects.requireNonNull(ioSsd, "IO SSD resource not found for YP allocation");
        Objects.requireNonNull(ioHdd, "IO HDD resource not found for YP allocation");

        final Map<SimpleResourceModelMapper.InternalKey, BigDecimal> internalAmount = new HashMap<>();
        final List<QuotaChangeRequest.Change> affectedChanges = new ArrayList<>();
        final Map<YpResource, Long> idByResource = new HashMap<>();
        for (List<QuotaChangeRequest.Change> segmentChanges : changesBySegment.values()) {
            final Map<YpResource, BigDecimal> internalAmountByResource = new HashMap<>();
            for (QuotaChangeRequest.Change change : segmentChanges) {
                final Resource resource = change.getResource();
                final YpResource ypResource = YpResource.from(resource.getPublicKey());
                idByResource.put(ypResource, resource.getId());
                final BigDecimal changeAmount = BigDecimal.valueOf(change.getAmountReady() - change.getAmountAllocating());
                final SimpleResourceModelMapper.InternalKey internalKey = SimpleResourceModelMapper.toInternalKey(change);
                final SimpleResourceModelMapper.InternalResourceKey internalResourceKey = SimpleResourceModelMapper.toInternalResourceKey(change);
                if (mappings.containsKey(internalResourceKey)) {
                    affectedChanges.add(change);
                }
                internalAmount.merge(internalKey, changeAmount, BigDecimal::add);
                internalAmountByResource.merge(ypResource, changeAmount, BigDecimal::add);
            }
            final Long cpuResourceId = idByResource.get(YpResource.CPU);
            if (internalAmountByResource.containsKey(YpResource.HDD)) {
                final Map<SimpleResourceModelMapper.InternalKey, BigDecimal> ioAmount = calculateIoAmount(internalAmount,
                        cpuResourceId, ioHdd, IO_HDD_BPS_PER_CORE);
                ioAmount.forEach((k, v) -> internalAmount.merge(k, v, BigDecimal::add));
            }
            if (internalAmountByResource.containsKey(YpResource.SSD)) {
                final Map<SimpleResourceModelMapper.InternalKey, BigDecimal> ioAmount = calculateIoAmount(internalAmount,
                        cpuResourceId, ioSsd, IO_SSD_BPS_PER_CORE);
                ioAmount.forEach((k, v) -> internalAmount.merge(k, v, BigDecimal::add));
            }
        }
        return Pair.of(internalAmount, affectedChanges);
    }

    @Override
    protected List<ResourcesModelMapping> getMappings(Service provider, long campaignId) {
        return mappingDao.getResourcesMappingsForCampaign(campaignId);
    }

    @Override
    protected List<ResourcesModelMapping> getMappings(Set<Resource> resources, long campaignId) {
        return mappingDao.getResourcesMappingsForCampaign(campaignId);
    }

    private static Map<SimpleResourceModelMapper.InternalKey, BigDecimal> calculateIoAmount(
            Map<SimpleResourceModelMapper.InternalKey, BigDecimal> internalAmount,
            Long cpuResourceId,
            Resource ioResource,
            BigDecimal amountPerCore) {
        final Map<SimpleResourceModelMapper.InternalKey, BigDecimal> ioAmountByKey = new HashMap<>();
        internalAmount.entrySet()
                .stream()
                .filter(e -> Objects.equals(e.getKey().getResourceId(), cpuResourceId))
                .forEach(e -> {
                    final SimpleResourceModelMapper.InternalKey key = e.getKey();
                    final SimpleResourceModelMapper.InternalKey ioInternalKey = new SimpleResourceModelMapper
                            .InternalKey(ioResource.getId(), key.getSegmentIds(), key.getBigOrderId());
                    final BigDecimal ioAmount = e.getValue()
                            .divideToIntegralValue(PER_CORE)
                            .multiply(amountPerCore);
                    ioAmountByKey.put(ioInternalKey, ioAmount);
                });
        return ioAmountByKey;
    }

}
