package ru.yandex.qe.dispenser.ws.sensors;

import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;

import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.Histogram;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

public class SensorsHolder {

    private static final String RATE = "http.requests_rate";
    private static final String TIME = "http.requests_time_millis";
    private static final String SIZE = "http.requests_size_bytes";
    private static final String CODE = "code";
    private static final String ENDPOINT = "endpoint";
    // 'service' conflicts with default solomon label so we use 'provider' instead
    private static final String SERVICE = "provider";
    private static final String ALL = "all";
    private static final String CODE_2XX = "2xx";
    private static final String CODE_3XX = "3xx";
    private static final String CODE_4XX = "4xx";
    private static final String CODE_499 = "499";
    private static final String CODE_429 = "429";
    private static final String CODE_5XX = "5xx";
    private static final String ENDPOINT_NOT_FOUND = "not_found";
    private static final String SERVICE_NONE = "none";

    private final MetricRegistry registry;
    private final Rate totalRate;
    private final Rate total2xxRate;
    private final Rate total3xxRate;
    private final Rate total4xxRate;
    private final Rate total5xxRate;
    private final Rate total499Rate;
    private final Rate total429Rate;
    private final Histogram totalTime;
    private final Histogram totalSize;
    private final ConcurrentHashMap<String, Rate> perEndpointRates = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String, Rate> perEndpoint5xxRates = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String, Histogram> perEndpointTimes = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String, Rate> perServiceRates = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String, Rate> perService5xxRates = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String, Histogram> perServiceTimes = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<ServiceEndpoint, Rate> perServiceEndpointRates = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<ServiceEndpoint, Rate> perServiceEndpoint5xxRates = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<ServiceEndpoint, Histogram> perServiceEndpointTimes = new ConcurrentHashMap<>();

    public SensorsHolder(final MetricRegistry registry) {
        this.registry = registry;
        this.totalRate = registry.rate(RATE, Labels.of(CODE, ALL, ENDPOINT, ALL, SERVICE, ALL));
        this.total2xxRate = registry.rate(RATE, Labels.of(CODE, CODE_2XX, ENDPOINT, ALL, SERVICE, ALL));
        this.total3xxRate = registry.rate(RATE, Labels.of(CODE, CODE_3XX, ENDPOINT, ALL, SERVICE, ALL));
        this.total4xxRate = registry.rate(RATE, Labels.of(CODE, CODE_4XX, ENDPOINT, ALL, SERVICE, ALL));
        this.total5xxRate = registry.rate(RATE, Labels.of(CODE, CODE_5XX, ENDPOINT, ALL, SERVICE, ALL));
        this.total499Rate = registry.rate(RATE, Labels.of(CODE, CODE_499, ENDPOINT, ALL, SERVICE, ALL));
        this.total429Rate = registry.rate(RATE, Labels.of(CODE, CODE_429, ENDPOINT, ALL, SERVICE, ALL));
        this.totalTime = registry.histogramRate(TIME, Labels.of(CODE, ALL, ENDPOINT, ALL, SERVICE, ALL),
                Histograms.exponential(22, 2.0d, 1.0d));
        this.totalSize = registry.histogramRate(SIZE, Labels.of(CODE, "all", ENDPOINT, "all", SERVICE, "all"),
                Histograms.exponential(12, 10.0d, 1.0d));
    }

    public void onRequest(final String endpoint, final String service, final Long durationMillis, final Long sizeBytes,
                          final boolean knownEndpoint, final Integer code) {
        final String endpointKey = endpoint == null || !knownEndpoint ? ENDPOINT_NOT_FOUND : prepareLabel(endpoint);
        final String serviceKey = service == null ? SERVICE_NONE : prepareLabel(service);
        final ServiceEndpoint serviceEndpointKey = new ServiceEndpoint(endpointKey, serviceKey);
        // Total rate per instance
        totalRate.inc();
        final Rate perEndpointRate = perEndpointRates.computeIfAbsent(endpointKey,
                key -> registry.rate(RATE, Labels.of(CODE, ALL, ENDPOINT, key, SERVICE, ALL)));
        // Total rate per endpoint
        perEndpointRate.inc();
        final Rate perServiceRate = perServiceRates.computeIfAbsent(serviceKey,
                key -> registry.rate(RATE, Labels.of(CODE, ALL, ENDPOINT, ALL, SERVICE, key)));
        // Total rate per service
        perServiceRate.inc();
        final Rate perServiceEndpointRate = perServiceEndpointRates.computeIfAbsent(serviceEndpointKey,
                key -> registry.rate(RATE, Labels.of(CODE, ALL, ENDPOINT, key.getEndpoint(), SERVICE, key.getService())));
        // Total rate per service endpoint
        perServiceEndpointRate.inc();
        if (code != null && code >= 200 && code < 300) {
            // Total 2xx rate per instance
            total2xxRate.inc();
        }
        if (code != null && code >= 300 && code < 400) {
            // Total 3xx rate per instance
            total3xxRate.inc();
        }
        if (code != null && code >= 400 && code < 500) {
            // Total 4xx rate per instance
            total4xxRate.inc();
            if (code == 499) {
                total499Rate.inc();
            }
            if (code == 429) {
                total429Rate.inc();
            }
        }
        if (code != null && code >= 500 && code < 600) {
            // Total 5xx rate per instance
            total5xxRate.inc();
            final Rate perEndpoint5xxRate = perEndpoint5xxRates.computeIfAbsent(endpointKey,
                    key -> registry.rate(RATE, Labels.of(CODE, CODE_5XX, ENDPOINT, key, SERVICE, ALL)));
            // Total 5xx rate per endpoint
            perEndpoint5xxRate.inc();
            final Rate perService5xxRate = perService5xxRates.computeIfAbsent(serviceKey,
                    key -> registry.rate(RATE, Labels.of(CODE, CODE_5XX, ENDPOINT, ALL, SERVICE, key)));
            // Total 5xx rate per service
            perService5xxRate.inc();
            final Rate perServiceEndpoint5xxRate = perServiceEndpoint5xxRates.computeIfAbsent(serviceEndpointKey,
                    key -> registry.rate(RATE, Labels.of(CODE, CODE_5XX, ENDPOINT, key.getEndpoint(), SERVICE, key.getService())));
            // Total 5xx rate per service endpoint
            perServiceEndpoint5xxRate.inc();
        }
        if (durationMillis != null) {
            // Request processing duration histogram per instance
            totalTime.record(durationMillis);
            final Histogram perEndpointTime = perEndpointTimes.computeIfAbsent(endpointKey,
                    key -> registry.histogramRate(TIME, Labels.of(CODE, ALL, ENDPOINT, key, SERVICE, ALL),
                            Histograms.exponential(22, 2.0d, 1.0d)));
            // Request processing duration histogram per endpoint
            perEndpointTime.record(durationMillis);
            final Histogram perServiceTime = perServiceTimes.computeIfAbsent(serviceKey,
                    key -> registry.histogramRate(TIME, Labels.of(CODE, ALL, ENDPOINT, ALL, SERVICE, key),
                            Histograms.exponential(22, 2.0d, 1.0d)));
            // Request processing duration histogram per service
            perServiceTime.record(durationMillis);
            final Histogram perServiceEndpointTime = perServiceEndpointTimes.computeIfAbsent(serviceEndpointKey,
                    key -> registry.histogramRate(TIME, Labels.of(CODE, ALL, ENDPOINT, key.getEndpoint(), SERVICE, key.getService()),
                            Histograms.exponential(22, 2.0d, 1.0d)));
            // Request processing duration histogram per service endpoint
            perServiceEndpointTime.record(durationMillis);
        }
        if (sizeBytes != null) {
            // Response size histogram per instance
            totalSize.record(sizeBytes);
        }
    }

    private String prepareLabel(final String label) {
        return label.replaceAll("[{]", "[").replaceAll("[}]", "]")
                .replace('|', '!').substring(0, Math.min(label.length(), 198));
    }

    private static final class ServiceEndpoint {

        private final String endpoint;
        private final String service;

        private ServiceEndpoint(final String endpoint, final String service) {
            this.endpoint = endpoint;
            this.service = service;
        }

        public String getEndpoint() {
            return endpoint;
        }

        public String getService() {
            return service;
        }

        @Override
        public boolean equals(final Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            final ServiceEndpoint that = (ServiceEndpoint) o;
            return Objects.equals(endpoint, that.endpoint)
                    && Objects.equals(service, that.service);
        }

        @Override
        public int hashCode() {
            return Objects.hash(endpoint, service);
        }

        @Override
        public String toString() {
            return "ServiceEndpoint{"
                    + "endpoint='" + endpoint + '\''
                    + ", service='" + service + '\''
                    + '}';
        }

    }

}
