package ru.yandex.qe.dispenser.ws.sensors;

import java.io.IOException;
import java.util.Optional;

import javax.servlet.ServletConfig;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.annotation.WebServlet;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.springframework.web.context.WebApplicationContext;
import org.springframework.web.context.support.WebApplicationContextUtils;

import ru.yandex.qe.dispenser.InstrumentedQloudTvmService;
import ru.yandex.qe.dispenser.solomon.SolomonHolder;
import ru.yandex.qe.dispenser.ws.intercept.AccessLogFilter;
import ru.yandex.qe.hitman.tvm.TvmConstants;
import ru.yandex.qe.hitman.tvm.qloud.TvmServiceTicketInfo;
import ru.yandex.monlib.metrics.encode.spack.MetricSpackEncoder;
import ru.yandex.monlib.metrics.encode.spack.format.CompressionAlg;
import ru.yandex.monlib.metrics.encode.spack.format.TimePrecision;

@WebServlet(
        urlPatterns = "/sensors",
        name = "SolomonSensorsServlet",
        loadOnStartup = 1
)
public class SolomonServlet extends HttpServlet {

    private InstrumentedQloudTvmService qloudTvmService;
    private SolomonHolder solomonHolder;

    @Override
    protected void doGet(final HttpServletRequest req, final HttpServletResponse resp) throws IOException {
        final String serviceTicket = req.getHeader(TvmConstants.TVM_SERVICE_HEADER_NAME);
        if (serviceTicket == null) {
            resp.sendError(HttpServletResponse.SC_UNAUTHORIZED, "Valid TVM ticket is required");
            return;
        }
        final Optional<TvmServiceTicketInfo> validatedTicketO = qloudTvmService.validateServiceTicket(serviceTicket);
        if (!validatedTicketO.isPresent()) {
            resp.sendError(HttpServletResponse.SC_UNAUTHORIZED, "Valid TVM ticket is required");
            return;
        }
        final int requestSource = validatedTicketO.get().getSource();
        req.setAttribute(AccessLogFilter.TVM_CLIENT_ID, String.valueOf(requestSource));
        if (solomonHolder.getSolomonTvmId() != requestSource) {
            resp.sendError(HttpServletResponse.SC_UNAUTHORIZED, "Only Solomon requests are allowed");
            return;
        }
        resp.setStatus(HttpServletResponse.SC_OK);
        resp.setContentType("application/x-solomon-spack");
        try (final MetricSpackEncoder e = new MetricSpackEncoder(TimePrecision.SECONDS, CompressionAlg.LZ4, resp.getOutputStream())) {
            solomonHolder.getRootRegistry().supply(0, e);
        }
    }

    @Override
    public void destroy() {
        this.qloudTvmService = null;
        this.solomonHolder = null;
        super.destroy();
    }

    @Override
    public void init(final ServletConfig config) throws ServletException {
        super.init(config);
        final ServletContext servletContext = config.getServletContext();
        final WebApplicationContext webApplicationContext = WebApplicationContextUtils.getWebApplicationContext(servletContext);
        this.qloudTvmService = webApplicationContext.getBean(InstrumentedQloudTvmService.class);
        this.solomonHolder = webApplicationContext.getBean(SolomonHolder.class);
    }

}
